#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/formatting/FormatParser.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/DataSizeAction.h>
#include <utils/formatting/PrintAction.h>
#include <utils/formatting/ReadAction.h>
#include <utils/formatting/DeleteAction.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/PrintFormatAction.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/EqualsAction.h>
#include <utils/formatting/CopyAction.h>
#include <utils/formatting/VerbatimSpec.h>
#include <utils/formatting/VectorSpec.h>

main(int argc, char** argv)
{
    __UTILS::FormatAction::initClasses();
    __UTILS::FormatParser parser;
    parser.registerFormat("vector", new __UTILS::VectorSpec);
    parser.registerFormat("verbatim", new __UTILS::VerbatimSpec);
    parser.registerFormat("vec3f",
                       parser.parseString("{ float, float, float }"));
    parser.registerFormat("lineSeg",
                       parser.parseString("{ vec3f, vec3f }"));
    parser.registerFormat("vec2f",
                       parser.parseString("{ float, float }"));

    __UTILS::FormatSpec* spec = parser.parseString(argv[1]);
    spec->ref();
    if (!spec) {
        fprintf(stderr, "Spec was invalid\n");
        exit(-1);
    }

    __UTILS::Input dataSrc;
    dataSrc.setBuffer(argv[2], strlen(argv[2]));

    __UTILS::DataSizeAction sizer;
    spec->act(&sizer);

    printf("Format %s (size = %d)\n", argv[1], sizer.getSize());
    printf("Input '%s'\n", argv[2]);

    printf("Format reprinted as: ");
    __UTILS::Output output;
    __UTILS::PrintFormatAction print_form_act(output);
    spec->act(&print_form_act);
    printf("\n");

    __UTILS::FormatEqualsAction equal_act(spec);
    spec->act(&equal_act);
    if (equal_act.result())
        printf("Format equals itself\n");
    else
        printf("******* Big problem: Format does not equal itself\n");

    unsigned char* data = new unsigned char[sizer.getSize()];
    __UTILS::ReadAction reader(dataSrc, data);
    spec->act(&reader);

    printf("Read as:\n");
    __UTILS::PrintAction print_act(data);
    spec->act(&print_act);
    printf("\n");

    __UTILS::BufferSizeAction bs_act(data);
    spec->act(&bs_act);
    int buf_size = bs_act.getBufSize();
    printf("Buffer size %d\n", buf_size);

    unsigned char* buffer = new unsigned char[buf_size];
    __UTILS::PackAction pack_act(buffer, buf_size, data);
    spec->act(&pack_act);
    unsigned char* packed_data = new unsigned char[spec->dataSize()];
    __UTILS::UnpackAction unpack_act(buffer, buf_size, 0, packed_data);
    spec->act(&unpack_act);
    printf("After packing and unpacking, data is,\n");
    print_act.resetData(packed_data);
    spec->act(&print_act);
    printf("\n");

    if (__UTILS::EqualsAction::equals(spec, data, packed_data)) 
        printf("Data equals packed data\n");
    else
        printf("******* Big problem: Data does not equal packed data\n");

    unsigned char* copy_data = new unsigned char[spec->dataSize()];
    __UTILS::CopyAction::copy(spec, data, copy_data);
    printf("After copying, data is,\n");
    print_act.resetData(copy_data);
    spec->act(&print_act);
    printf("\n");
    if (!__UTILS::EqualsAction::equals(spec, data, copy_data))
        printf("******* Big problem: Data does not equal copied data\n");

    __UTILS::DeleteAction delete_act(data);
    spec->act(&delete_act);
    delete [] data;

    delete_act.reset(packed_data, buffer, buf_size);
    spec->act(&delete_act);
    delete [] packed_data;
    delete [] buffer;

    delete_act.reset(copy_data);
    spec->act(&delete_act);
    delete [] copy_data;
    
    printf("\n");

    spec->unref();
}
