#include <stdio.h>
#include <time.h>
#include <utils/CannedDataRead.h>
#include <utils/CannedDataWrite.h>
#include <utils/Time.h>

static void usage_error()
{
  printf("Usage:  cdsplit [-a] <input name> <start> <end> <output name>\n");
  exit(-1);
}

int main(int argc, char** argv)
{
  double start, end;
  bool absolute = false;

  if (argc < 2) {
    usage_error();
  }
  int arg_index = 1;
  if (argv[1][0] == '-') {
    if (argv[1][1] == 'a') {
      absolute = true;
    } else
      usage_error();
    arg_index = 2;
  }

  if (argc < arg_index + 4)
    usage_error();

  if (sscanf(argv[arg_index+1], "%lf", &start) != 1)
    usage_error();
  if (sscanf(argv[arg_index+2], "%lf", &end) != 1)
    usage_error();

  if (end > 0 && start > end) {
    printf("cdsplit: end must be greater than start\n");
    exit(-1);
  }
      
  try {
    utils::CannedDataReader reader;
    reader.open(argv[arg_index]);
    utils::CannedDataWriter writer;
    writer.open(argv[arg_index+3]);

    writer.writeType(reader.getType());
    uint32_t major, minor;
    reader.getVersion(&major, &minor);
    writer.writeVersion(major, minor);
    writer.writeDescription(reader.getDescription().getString());
    int max_size = reader.getHeaderSize();
    char* buffer = new char[max_size];
    int size = fread(buffer, 1, reader.getHeaderSize(), reader.getHeader());
    writer.writeHeader(buffer, size);

    uint64_t secs, usecs;
    reader.getFirstTimeStamp(&secs, &usecs);
    utils::Time first_time(secs, usecs);

    utils::Time end_time;
    if (end < 0) {
      reader.getLastTimeStamp(&secs, &usecs);
      end_time.setValue(secs, usecs);
    } else {
      end_time = utils::Time(end);
      if (!absolute)
        end_time += first_time;
    }

    if (start < 0) 
      reader.first();
    else {
      reader.first();
      utils::Time t(start);
      if (!absolute) {
        t += first_time;
      }
      reader.seek(t);
    }

    while (1) {
      size = reader.getRecordSize();
      if (size > max_size) {
        delete [] buffer;
        max_size = size;
        buffer = new char[max_size];
      }
      fread(buffer, 1, size, reader.getRecord());
      uint64_t secs, usecs;
      reader.getCurTimeStamp(&secs, &usecs);
      writer.writeData(buffer, size, secs, usecs);

      utils::Time tag(secs, usecs);
      if (tag >= end_time)
        break;

      reader.next();
    }
  } catch (utils::CannedDataError err) {
    fprintf(stderr, "Ack: %s\n", err.getMsg().getString());
  }
}           
           

  


  
