///////////////////////////////////////////////////////////////////////////////
//
//                                VectorSpec.cc
//
// Implements the custom format specifier which indicates a pointer to an
// instance of Vector.  Implements the various methods for reading, printing, 
// etc..
//
// Classes implemented for export:
//     VectorSpec
//     SizedList - a subclass of Vector<char> which allows us to fool with
//                 the internals to read in lists generically
//
///////////////////////////////////////////////////////////////////////////////

#include <utils/Vector.h>
#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/formatting/VectorSpec.h>
#include <utils/formatting/Token.h>
#include <utils/formatting/PrintAction.h>
#include <utils/formatting/DeleteAction.h>
#include <utils/formatting/ReadAction.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/CopyAction.h>
#include <utils/formatting/CloneAction.h>
#include <utils/formatting/PrintFormatAction.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/EqualsAction.h>
#include <utils/formatting/FormatParser.h>

__UTILS_BEGIN_NAMESPACE

UTILS_BASE_SOURCE(VectorSpec);

void VectorSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(VectorSpec, "VectorSpec", "CustomPtrSpec");
}

VectorSpec::VectorSpec(FormatSpec* fmt)
{
    UTILS_BASE_CONSTRUCTOR(VectorSpec);

    addFormatAction(DeleteAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::deleteList));
    addFormatAction(PrintAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::printList));
    addFormatAction(ReadAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::readList));
    addFormatAction(PrintFormatAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::printListFormat));
    addFormatAction(FormatEqualsAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::listFormatEquals));
    addFormatAction(PackAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::packList));
    addFormatAction(UnpackAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::unpackList));
    addFormatAction(BufferSizeAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::getBufferSize));
    addFormatAction(CopyAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::makeCopy));
    addFormatAction(CloneAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::makeClone));
    addFormatAction(EqualsAction::getClassTypeId(),
                    new ClassApplyFormatAction<VectorSpec>
                    (this, &VectorSpec::listEquals));

    _fmt = fmt;
    if (_fmt)
        _fmt->ref();
}

VectorSpec::~VectorSpec()
{
    if (_fmt)
        _fmt->unref();
}

// set the format to be fmt, with all the appropriate ref'ing and unref'ing
void VectorSpec::setFormat(FormatSpec* fmt)
{
    if (_fmt == fmt)
        return;
    if (_fmt)
        _fmt->unref();
    _fmt = fmt;
    if (_fmt)
        _fmt->ref();
}

// apply the delete action act to the specified format
bool VectorSpec::deleteList(FormatAction* act)
{
    DeleteAction* delAct = (DeleteAction*) act;
    Vector<char>* list = (Vector<char>*) delAct->getSubData();
    delAct->addSize(sizeof(Vector<char>*));
    // make sure there is a valid list
    if (!list)
        return true;

    // if the contents need deleted, do so
    if (!_fmt->isFlatType()) {
        DeleteAction subaction(list->getData(),
                                 delAct->getBuffer(), delAct->getBufSize());
        subaction.inherit(*delAct);
        int i;
        for (i=0;i<list->numElems();i++) 
            if (!_fmt->act(&subaction))
                break;
    }

    // delete the list
    delete list;
    return true;
}

// apply the read action act to the specified format to read in a list
bool VectorSpec::readList(FormatAction* act)
{
    ReadAction* readAct = (ReadAction*) act;
    SizedList* newList = (SizedList*) NULL;

    char c;
    bool res = true;
    if (readAct->getInput().read(c) && c == '(') {  // check opening list paren
        if (readAct->getInput().read(c) && c != ')') { // check for empty list
            readAct->getInput().putBack(c);
            int size_type = _fmt->dataSize();
            if (!size_type) {
                res = false;
                goto finished;
            }
            newList = new SizedList;   // create new generic list
            while (1) {   // read in the elements
                ReadAction subaction(readAct->getInput(),
                                       newList->makeRoom(size_type));
                subaction.inherit(*act);
                if (!_fmt->act(&subaction)) {
                    delete newList;
                    newList = 0L;
                    res = false;
                    goto finished;
                }
                if (!subaction.getInput().read(c) || c != ',')
                    break;
            }
            if (c != ')') {   // check closing list paren
                delete newList;
                newList = 0L;
                readAct->getInput().putBack(c);
                res = false;
            } else  // make sure new list has proper elem. numbers for its size
                newList->finishSized(size_type);
        }
    }

finished:
    // store the result
    readAct->putData(&newList, sizeof(SizedList*));
    readAct->addSize(sizeof(SizedList*));

    return res;
}

// apply the print action act to the specified format
bool VectorSpec::printList(FormatAction* act)
{
    PrintAction* printAct = (PrintAction*) act;
    Vector<char>* list = (Vector<char>*) printAct->getSubData();

    Output& output = printAct->getOutput();
    if (!list) { // make sure there is a valid list
        output.write("( )");
        printAct->addSize(sizeof(Vector<char>*));
        return true;
    }

    char* data = list->getData();     // get data
    int size_type = _fmt->dataSize(); // and data element size
    output.write("( ");               // print opening paren
    // and print the elements
    for (int i=0;i<list->numElems();i++) {
        PrintAction subaction(data + i*size_type, output);
        subaction.inherit(*act);
        if (!_fmt->act(&subaction))
            return false;
        if (i != list->numElems()-1)
            output.write(", ");
    }
    output.write(" )");              // print closing paren
    printAct->addSize(sizeof(Vector<char>*));
    return true;
}

// apply the print format action to print out that this is a list of whatever
bool VectorSpec::printListFormat(FormatAction* action)
{
    PrintFormatAction* pfa = (PrintFormatAction*) action;
    pfa->getOutput().write("(list ");
    if (!_fmt->act(pfa))
        return false;
    pfa->getOutput().write(" )");
    return true;
}

// Given that action is a FormatEqualsAction, accumulate whether the other
// format is a list specifier containing the same type of formatted data
bool VectorSpec::listFormatEquals(FormatAction* action)
{
    FormatEqualsAction* ea = (FormatEqualsAction*) action;
    if (!ea->getOther()->isType(getClassTypeId())) {
        ea->indicateInequality();
        return true;
    }

    FormatEqualsAction sub_action(((VectorSpec*) ea->getOther())->_fmt);
    if (!_fmt->act(&sub_action))
        return false;
    if (!sub_action.result())
        ea->indicateInequality();
    return true;
}

// Given that action is an BufferSizeAction, accumulate the number of 
// bytes need to pack the list header and the list data
bool VectorSpec::getBufferSize(FormatAction* action)
{
    BufferSizeAction* bs = (BufferSizeAction*) action;

    bs->addBufSize(sizeof(int));  // header is number of elements
    
    // and the elements themselves
    Vector<char>* list = (Vector<char>*) bs->getSubData();
    bs->addSize(sizeof(Vector<char>*)); // increment index
    if (!list)
        return true;
    int size_type = _fmt->dataSize();
    char* data = list->getData();     // get data
    for (int i=0;i<list->numElems();i++) {
        BufferSizeAction subaction(data + i*size_type);
        subaction.inherit(*action);
        if (!_fmt->act(&subaction))
            return false;
        bs->addBufSize(subaction.getBufSize());
    }
    return true;
}

// Given that action is an PackAction, pack a list into a flat buffer        
bool VectorSpec::packList(FormatAction* action)
{
    PackAction* pa = (PackAction*) action;

    int size;
    Vector<char>* list = (Vector<char>*) pa->getSubData();
    if (!list) {
        // null lists get sent as 0-length lists
        size = 0;
        pa->toBuffer((unsigned char*) &size, sizeof(int));
        pa->addSize(sizeof(Vector<char>*));
        return true;
    }
    // pack the number of elements
    size = list->numElems();
    pa->toBuffer((unsigned char*) &size, sizeof(int));
    
    // pack the elements
    int size_type = _fmt->dataSize();
    char* data = list->getData();     // get data
    int buf_index = pa->getIndex();
    for (int i=0;i<list->numElems();i++) {
        PackAction subaction(pa->getBuffer(), pa->getBufSize(), 
                               data + i*size_type, buf_index);
        subaction.inherit(*action);
        if (!_fmt->act(&subaction))
            return false;
        buf_index = subaction.getIndex();
    }
    pa->setIndex(buf_index);
    pa->addSize(sizeof(Vector<char>*));

    return true;
}

// Given that action is an UnpackAction, pack a flat buffer into
// a list of formatted data
bool VectorSpec::unpackList(FormatAction* action)
{
    UnpackAction* ua = (UnpackAction*) action;

    int size = ua->getInt();
    int size_type = _fmt->dataSize();
    bool res;
    SizedList* list;
    if (size_type) {
        list = new SizedList(size*size_type);
        char* data = list->getData();     // get data
        UnpackAction subaction(ua->getBuffer(), ua->getBufSize(), ua->getIndex(),
                               data,
                               ua->getAlignment(), ua->getByteOrder());
        subaction.inherit(*action);
        res = true;
        for (int i=0;i<size;i++) 
            if (!_fmt->act(&subaction)) {
                res = false;
                break;
            }
        list->finishSized(size_type, size);
        ua->setIndex(subaction.getIndex());
    } else {
        res = false;
        list = 0L;
    }

    ua->toData(&list, sizeof(SizedList*));
    ua->addSize(sizeof(SizedList*));
    ua->needsPacking();
    return res;
}

// Given that action is an CopyAction, copy the list from the source to
// the destination
bool VectorSpec::makeCopy(FormatAction* action)
{
    CopyAction* ca = (CopyAction*) action;

    // get information on original
    Vector<char>* orig = (Vector<char>*) ca->getSubData();
    if (!orig) 
        ca->toDest(&orig, sizeof(SizedList*));
    else {
        int size = orig->numElems();
        int size_type = _fmt->dataSize();

        // make a copy of it
        SizedList* copy = new SizedList(size*size_type);
        CopyAction subaction(orig->getData(), copy->getData());
        subaction.inherit(*action);
        for (int i=0;i<size;i++) 
            if (!_fmt->act(&subaction))
                return false;

        // store that copy in the destination
        copy->finishSized(size_type, size);
        ca->toDest(&copy, sizeof(SizedList*));
    }

    // advance pointers
    ca->advance(sizeof(SizedList*));
    ca->needsPacking();
    return true;
}

// Given that action is an CloneAction, copy the list from the source to
// the destination
bool VectorSpec::makeClone(FormatAction* action)
{
    CloneAction* ca = (CloneAction*) action;

    // get information on original
    Vector<char>* orig = (Vector<char>*) ca->getSubData();
    if (!orig) 
        ca->toDest(&orig, sizeof(SizedList*));
    else {
        int size = orig->numElems();
        int size_type = _fmt->dataSize();

        // make a copy of it
        SizedList* copy = new SizedList(size*size_type);
        CloneAction subaction(orig->getData(), copy->getData());
        subaction.inherit(*action);
        for (int i=0;i<size;i++) 
            if (!_fmt->act(&subaction))
                return false;

        // store that copy in the destination
        copy->finishSized(size_type, size);
        ca->toDest(&copy, sizeof(SizedList*));
    }

    // advance pointers
    ca->advance(sizeof(SizedList*));
    ca->needsPacking();
    return true;
}

// Given that aciton is an EqualsAction, accumulates whether the two
// lists equal each other
bool VectorSpec::listEquals(FormatAction* action)
{
    EqualsAction* ea = (EqualsAction*) action;

    // get the two lists
    Vector<char>* one = (Vector<char>*) ea->getSubData();
    Vector<char>* two = (Vector<char>*) ea->getOther().getSubData();
    // zero length lists get treated as NULL
    if (one && one->numElems() == 0)  
        one = 0L;
    if (two && two->numElems() == 0)
        two = 0L;

    if (one == two) {  // pointer equality is list equality
        ea->advance(sizeof(Vector<char>*));
        return true;
    }
    // check easy stuff, such as NULL's and list size mismatches
    if (!one || !two || one->numElems() != two->numElems()) {
        ea->setResult(false);
        ea->advance(sizeof(Vector<char>*));
        return true;
    }
    // check if elements are equal all at once, if this is a packed type
    int size_type = _fmt->dataSize();
    int size = one->numElems();
    if (_fmt->isPackedType()) {
        if (memcmp(one->getData(), two->getData(), size_type*size)) 
            ea->setResult(false);
        ea->advance(sizeof(Vector<char>*));
        return true;
    }

    // or one by one, if necessary
    EqualsAction subaction(one->getData(), two->getData());
    subaction.inherit(*action);
    for (int i=0;i<size;i++) {
        if (!_fmt->act(&subaction))
            return false;
        if (!subaction.result()) {
            ea->setResult(false);
            break;
        }
    }

    ea->advance(sizeof(Vector<char>*));
    return true;
}

// process the specified format to come up with the parameterized VectorSpec
FormatSpec* VectorSpec::processParameters(FormatParser* parser)
{
    // parse the element format
    FormatSpec* fmt = parser->parse((StructSpec*) NULL);

    // create the parameterized list format specifier
    VectorSpec* res = new VectorSpec(fmt);
    copyExternalsTo(res);
    return res;
}

// change applyAction so that actions will have access to this spec under
// the name listSpec, since listSpec is a "special" format spec which 
// gets copied for different types of lists, and actions may need to know
// exactly which listSpec they are working with.
bool VectorSpec::applyAction(FormatAction* action)
{
    action->set("listSpec", this);
    if (canApplyAction(action)) 
        return CustomPtrSpec::applyAction(action);
    else if (action->isOfType(DataAction::getClassTypeId())) {
        DataAction* daction = (DataAction*) action;
        Vector<char>* list = (Vector<char>*) daction->getSubData();
        if (!list)
            return true;
        unsigned char* old_data = daction->getData();
        int old_index = daction->getSize();
        daction->resetData(list->getData());
        for (int i=0;i<list->numElems();i++) 
            if (!_fmt->act(daction))
                return false;
        daction->resetData(old_data);
        daction->setIndex(old_index+sizeof(Vector<char>*));
        return true;
    }

    return false;
}

__UTILS_END_NAMESPACE
