///////////////////////////////////////////////////////////////////////////////
//
//                                 Token.cc
//
// This file implements the class that converts format string into 
// tokens that can be manipulated by FormatParser
//
// Classes defined for export:
//    Token - a symbolic token representing a component of a format string
//    TokenReader - Reads tokens from text
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <utils/formatting/Token.h>

__UTILS_BEGIN_NAMESPACE;

///////////////////////////////////////////////////////////////////////////////
// Token - a symbolic token representing a component of a format string
///////////////////////////////////////////////////////////////////////////////

// Takes a token and frees the memory allocated for it.
// Does nothing if input in NULL
// Basically free(token) unless it has a string for value tok->value.str
Token::~Token()
{
    if (this) {
        if (type == STR)
            delete [] value.str;
    }
}

// Takes a token and displays its type to stdout
// Checks token->type against token type and prints the type
void Token::print()
{
    fprintf(stderr, "loc: %d: type: ", loc);

    switch(type) {
      case LPAREN:
        fprintf(stderr, "LPAREN\n");
        break;
      case RPAREN:
        fprintf(stderr, "RPAREN\n");
        break;
      case LBRACE:
        fprintf(stderr, "LBRACE\n");
        break;
      case RBRACE:
        fprintf(stderr, "RBRACE\n");
        break;
      case LBRACK:
        fprintf(stderr, "LBRACK\n");
        break;
      case RBRACK:
        fprintf(stderr, "RBRACK\n");
        break;
      case STAR:
        fprintf(stderr, "STAR\n");
        break;
      case BANG:
        fprintf(stderr, "BANG\n");
        break;
      case COMMA:
        fprintf(stderr, "COMMA\n");
        break;
      case LT:
        fprintf(stderr, "LT\n");
        break;
      case GT:
        fprintf(stderr, "GT\n");
        break;
      case COLON:
        fprintf(stderr, "COLON\n");
        break;
      case EOSTRING:
        fprintf(stderr, "EOSTRING\n");
        break;
      case INT:
        fprintf(stderr, "INT: %d\n", value.num);
        break;
      case STR:
        fprintf(stderr, "STR: %s\n", value.str);
        break;
      default:
        fprintf(stderr, "Token of Unknown type: %d\n", type);
        break;
    }
}


///////////////////////////////////////////////////////////////////////////////
// TokenReader - Reads tokens from text
///////////////////////////////////////////////////////////////////////////////

// ungetToken() - backs up one position in the string
void TokenReader::ungetToken(Token* token)
{
    _unget_token = token;
}


// Reads a string into the current token's value
// Returns the token w/ the string in token->value.str
// Read string of characters not including delimiters i.e. {}[]<>\0()
Token* TokenReader::readString(const char* s)
{
    const char *t2;
    Token* tmp;
    int i, amount;

    for (i=_current_location;
         (s[i] != ',' && s[i] != '}' && s[i] != '{' && s[i] != '[' 
          && s[i] != ']' && s[i] != ':' && s[i] != ' ' && s[i] != '>'
          && s[i] != '<' && s[i] != '(' && s[i] != ')' && s[i] != '\0');
         i++);

    amount = i - _current_location;
    tmp = new Token(Token::STR, _current_location);
    tmp->value.str = new char[amount+1];
    t2 = s + _current_location;
    strncpy(tmp->value.str, t2, amount);
    tmp->value.str[amount] = '\0';

    _current_location += amount;

    return tmp;
}


// Reads the integers from a string s
// Returns the token w/ the integer in token->value.num
// Read string of characters not include delimiters i.s. {}[]<>\0
Token* TokenReader::readInt(const char *s)
{
    char *t;
    const char *t2;
    Token* tmp;
    int i, amount;

    for (i = _current_location;
         (isdigit(s[i]) && s[i] != ',' && s[i] != '}' && s[i] != '{' 
          && s[i] != '[' && s[i] != ']' && s[i] != ':' && s[i] != ' ' 
          && s[i] != '>' && s[i] != '<' && s[i] != '(' && s[i] != ')'
          && s[i] != '\0');
         i++);

    amount = i - _current_location;
    tmp = new Token( Token::INT, _current_location);
    t = new char[amount+1];
    t2 = s + _current_location;
    strncpy(t, t2, amount);
    t[amount] = '\0';

#ifndef VXWORKS
    tmp->value.num = atoi(t);
#else
    int num;
    sscanf(t, "%d", &num);
    tmp->value.num = num;
#endif

    delete [] t;

    _current_location += amount;

    return tmp;
}    


// Reads a character and returns the tokenized result
// Return token w/ the token->type updated
// Checks each character against token table.  Then creates token  *
Token* TokenReader::nextToken()
{
    Token* token;

    // check backup token
    if (_unget_token) {
        token = _unget_token;
        _unget_token = NULL;
        return token;
    }

    // make sure we have a string defined
    if (!_current_string) {
        return new Token(Token::EOSTRING, _current_location);
    }

    // parse the token
  Start:
    switch(_current_string[_current_location]) {
      case ' ':
      case '\t':
      case '\f':
        _current_location++;
        goto Start;
      case '{':
        token = new Token( Token::LBRACE, _current_location);
        _current_location++;
        break;
      case '}':
        token = new Token( Token::RBRACE, _current_location);
        _current_location++;
        break;
      case '[':
        token = new Token( Token::LBRACK, _current_location);
        _current_location++;
        break;
      case ']':
        token = new Token( Token::RBRACK, _current_location);
        _current_location++;
        break;
      case '(':
        token = new Token( Token::LPAREN, _current_location);
        _current_location++;
        break;
      case ')':
        token = new Token( Token::RPAREN, _current_location);
        _current_location++;
        break;
      case '*':
        token = new Token( Token::STAR, _current_location);
        _current_location++;
        break;
      case '!':
        token = new Token( Token::BANG, _current_location);
        _current_location++;
        break;
      case ',':
        token = new Token( Token::COMMA, _current_location);
        _current_location++;
        break;
      case '<':
        token = new Token( Token::LT, _current_location);
        _current_location++;
        break;
      case '>':
        token = new Token( Token::GT, _current_location);
        _current_location++;
        break;
      case ':':
        token = new Token( Token::COLON, _current_location);
        _current_location++;
        break;
      case '\0':
        token = new Token( Token::EOSTRING, _current_location);
        _current_location++;
        break;
      default:
        if (isdigit(_current_string[_current_location]))
            token = readInt(_current_string);
        else 
            token = readString(_current_string);
    }

    return token;
}


// Initializes the lexer the read first element in char string
// Sets the current location to 0. That is the index for the string.
// Sets the current string to s.
// This should be called before you lex a string
void TokenReader::init(const char* s)
{
    _current_string = s;
    _current_location = 0;
    _unget_token = NULL;
}

// create a token reader 
TokenReader::TokenReader()
{
    init(NULL);
}

TokenReader::~TokenReader()
{
    if (_unget_token)
        delete _unget_token;
}

__UTILS_END_NAMESPACE;
