///////////////////////////////////////////////////////////////////////////////
//
//                               Time.cc
//
// Implements a class for managing time.  An Time state instance holds
// a time, and gives methods for extracting it in a variety of forms.  Static
// methods are available to find the current time, and to control how that
// current time proceeds, either at a constant scaling of real time or 
// by discrete jumps.
//
// Classes implemented for export:
//     Time
//
// Classes implemented for internal use
//     TimeState - specifies the current state of time flow
//    
///////////////////////////////////////////////////////////////////////////////

#define NANOSLEEP_DEFINED 1
#define BILLION 1000000000

extern "C" {
#include <stdio.h>
#include <unistd.h>
#include <time.h>
#ifdef NANOSLEEP_DEFINED
#include <asm/param.h>
#endif
}

#include <utils/Basic.h>

#ifdef VXWORKS
# include <vxWorks.h>
# include <timers.h>
#else
# ifdef TIMEVAL_IN_TIME_H
#  include <time.h>
#  include <sys/time.h>
# elif defined(TIMEVAL_IN_SYS_TIME_H)
#  include <sys/time.h>
# else
#  include <sys/types.h>
#  include <sys/timeb.h>
// #  define USE_FTIME 1   uncomment for Cygwin - should be a test in configure
#endif

#ifndef STDC_HEADERS
extern "C" int gettimeofday(timeval*, struct timezone*);
#endif
#endif

#include <utils/Time.h>

__UTILS_BEGIN_NAMESPACE

static inline long abs_diff(long val1, long val2)
{
        if (val1 > val2) {
                return val1 - val2;
        }
        return val2 - val1;
}

class TimeInitializer {
public:
    TimeInitializer() { Time::init(); }
};

static TimeInitializer _TimeInitializer;

// structure that specifies what state time is in
struct TimeState {
    int time_mode;        // 0 for continuous time, 1 for discrete time
    float time_constant;  // scale factor for continuous time
    float time_step;      // time step for discrete time
    TimeState* prev;    // Previous time step in the stack
};

// static Time variables for controlling the flow of time ITSELF.
//                                                    Bwahahahahahahaha
int Time::_time_mode = 0;
Time* Time::_time_base = (Time*) NULL;
Time* Time::_time_start = (Time*) NULL;
float Time::_time_constant = 1.0;
float Time::_time_step = 0.25;
TimeState* Time::_state_stack = (TimeState*) NULL;

Time::Time() :
        sec(_sec),
        usec(_usec)
{
    _sec = _usec = 0;
}

Time::Time( double sec ) :
        sec(_sec),
        usec(_usec)
{
    setValue( sec );
}

Time::Time( long sec, long usec ) :
        sec(_sec),
        usec(_usec)
{
    setValue( sec, usec );
}

Time Time::getRealTimeOfDay()
{
#ifdef VXWORKS
    struct timespec tp;

    clock_gettime(CLOCK_REALTIME, &tp);

    return Time(tp.tv_sec, tp.tv_nsec/1000);
#else
#ifdef USE_FTIME
    struct timeb tp;
    ftime(&tp);
    return Time (int(tp.time), int(tp.millitm*1000));
#else    
    struct timeval tp;
    gettimeofday(&tp, NULL);
    
    return Time( int(tp.tv_sec), int(tp.tv_usec) );
#endif
#endif
}

void Time::setToTimeOfDay()
{
    *this = getTimeOfDay();
}

Time Time::zero()
{
    return Time( 0.0 );
}

Time Time::maxt()
{
    return Time( 2147483647, 2147483647 );
}

void Time::setValue( double sec )
{
    _sec = long (sec);
    _usec = long ((sec - double (_sec)) * 1000000.0 );
}

void Time::setValue( long sec, long usec )
{
    _sec = sec;
    _usec = usec;
}

double Time::getValue() const
{
    return ( ( double(_usec) / 1000000.0 ) + double(_sec) );
}

void Time::getValue( long &sec, long &usec ) const
{
    sec = _sec;
    usec = _usec;
}

void Time::getValue( int &sec, int &usec ) const
{
    sec = _sec;
    usec = _usec;
}

void Time::getValue( unsigned int &sec, unsigned int &usec ) const
{
    sec = (unsigned) _sec;
    usec = (unsigned) _usec;
}

int Time::operator ==( const Time &t ) const
{
    return (t._sec == this->_sec &&
            t._usec == this->_usec);
}

int Time::operator !=( const Time &t ) const
{
    return !operator==(t);
}

int Time::operator <(const Time& other) const
{
    return ((_sec < other._sec) ||
            (_sec == other._sec && _usec < other._usec));
}

int Time::operator >(const Time& other) const
{
    return ((_sec > other._sec) ||
            (_sec == other._sec && _usec > other._usec));
}

int Time::operator <=(const Time& other) const
{
    return ((_sec < other._sec) ||
            (_sec == other._sec && _usec <= other._usec));
}

int Time::operator >=(const Time& other) const
{
    return ((_sec > other._sec) ||
            (_sec == other._sec && _usec >= other._usec));
}

Time Time::absDiff(const Time &rhs) const
{
        /* There's got to be a better way to do this */
        Time retval(abs_diff(_sec, rhs._sec), abs_diff(_usec, rhs._usec));
        if (rhs < *this) {
                if (rhs._usec > _usec) {
                        retval._sec--;
                        retval._usec = (1000000 - retval._usec); 
                }
        } else {
                if (rhs.usec < usec) {
                        retval._sec--;
                        retval._usec = (1000000 - retval._usec); 
                }
        }
        return retval;
}

// initialize the static Time flow control variables
void Time::init()
{
    if (_time_base == NULL) {
        _time_base = new Time();
        _time_start = new Time();
    }
    _time_mode = 0;
    _time_constant = 1.0;
    _time_step = 0.25;
    *_time_base = Time::getRealTimeOfDay();
    *_time_start = *_time_base;
    _state_stack = 0L;
}

// static method for getting the filtered time of day
Time Time::getTimeOfDay()
{
    // if in continous mode and time is frozen, or in discrete mode
    // return the start time
    if (_time_mode || _time_constant <= 0.0)
        return *_time_start;
    
    // otherwise, return the scaled time that has elapsed since the
    // continuous mode was set.
    double elapsed = (Time::getRealTimeOfDay() - *_time_base).getValue();
    elapsed *= _time_constant;

    return *_time_start + Time(elapsed);
}

// static method which in discrete mode will jump time by the time step
void Time::stepTime()
{
    if (_time_mode) 
        *_time_start += Time(_time_step);
}

// sets the time mode to be continuous time, marking the time of the change
// with _time_start and _time_base. If Already in continuous time, do nothing
void Time::useContinuousTime()
{
    if (_time_mode) {
        *_time_start = getTimeOfDay();
        _time_mode = 0;
        *_time_base = Time::getRealTimeOfDay();
    }
}

// sets the time constant used in continuous time mode.  If in continuous time
// mode, mark the time of this change with _time_start and _time_base
void Time::setTimeConstant(float time_constant)
{
    if (!_time_mode) {
        *_time_start = getTimeOfDay();
        *_time_base = Time::getRealTimeOfDay();
    }
    _time_constant = time_constant;
}
        
// if in continuous time, change to discrete time.  Mark the change with the
// _time_start
void Time::useDiscreteTime()
{
    if (!_time_mode) {
        *_time_start = getTimeOfDay();
        _time_mode = 1;
    }
}    

// set the time increment for discrete time mode
void Time::setTimeStep(float time_step)
{
    _time_step = time_step;
}

// push the state of time control
void Time::pushState()
{
    TimeState* state = new TimeState;
    state->time_constant = _time_constant;
    state->time_step = _time_step;
    state->time_mode = _time_mode;
    state->prev = _state_stack;
    _state_stack = state;
}

// pop to a previous state of time control
void Time::popState()
{
    if (!_state_stack)
        return;

    TimeState* state = _state_stack;

    _time_mode = state->time_mode;
    setTimeConstant(state->time_constant);
    setTimeStep(state->time_step);

    _state_stack = state->prev;
    delete state;
}
    
Time operator +(const Time& t0, const Time &t1)
{
    Time res(t0._sec + t1._sec, t0._usec + t1._usec);
    while (res._usec > 1000000) {
        res._sec++;
        res._usec-=1000000;
    }
    while (res._usec < 0) {
        res._sec--;
        res._usec+=1000000;
    }

    return res;
}

Time operator -(const Time& t0, const Time &t1)
{
    Time res(t0._sec - t1._sec, t0._usec - t1._usec);
    while (res._usec > 1000000) {
        res._sec++;
        res._usec-=1000000;
    }
    while (res._usec < 0) {
        res._sec--;
        res._usec+=1000000;
    }

    return res;
}

#ifdef NANOSLEEP_DEFINED
#define HZ_INTERVAL (1.0/(double(HZ)))
#endif

void Time::sleep(double interval)
{
  if (interval <= 0)
    return;
  if (!_time_mode && _time_constant)
    interval /= _time_constant;
    
  // note:  wimp out for now on what to do if usleep not defined
#ifdef NANOSLEEP_DEFINED
  struct timespec sleep_spec;
  double real_interval;
  interval -= HZ_INTERVAL;
  if (interval < 0)
    interval = HZ_INTERVAL/2.0;
  real_interval = interval;
  
  sleep_spec.tv_sec = (long) real_interval;
  sleep_spec.tv_nsec = (long)
    ((real_interval - ((double) sleep_spec.tv_sec))*double(BILLION));
  nanosleep(&sleep_spec, NULL);
#elif USLEEP_DEFINED
  usleep((int) (interval * 1000000.0));
#endif
}

__UTILS_END_NAMESPACE
