///////////////////////////////////////////////////////////////////////////////
//
//                               SymbolTable.cc
//
// Implements classes for managing a symbol table
//
// Classes implemented for export:
//   SymbolManager - class for memory management of symbols
//   SymbolTable - the symbol table class
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <utils/SymbolTable.h>
#include <utils/Vector.h>

__UTILS_BEGIN_NAMESPACE

StringSymManager* SymbolTable::stringManager = NULL;
ManagedSymManager* SymbolTable::managedManager = NULL;

class SymbolTableInitializer {
public:
    SymbolTableInitializer() { SymbolTable::init(); }
};

static void delete_symbol(const char* name, Symbol* symbol, void* data)
{
    if (symbol->cleaner) {
        if (data && symbol->cleaner->isLast()) {
            Vector<Symbol*>* last_ones = (Vector<Symbol*>*) data;
            last_ones->append(symbol);
            return;
        }
        symbol->cleaner->unlink(symbol->data);
        symbol->cleaner->unref();
    }
    delete symbol;
}

SymbolTable::SymbolTable(int num_buckets)
  : StringDict<Symbol*>(num_buckets)
{
    init();
}

SymbolTable::~SymbolTable()
{
    Vector<Symbol*> last_ones(5);
    applyToAll(delete_symbol, (void*) &last_ones);
    for (int i=0;i<last_ones.numElems();i++) 
        delete_symbol(NULL, last_ones[i], NULL);
}

void SymbolTable::init()
{
    if (stringManager)
        return;

    stringManager = new StringSymManager;
    stringManager->makeStatic();
    managedManager = new ManagedSymManager;
    managedManager->makeStatic();
}

bool SymbolTable::set(const char* key, const void* data, 
                          SymbolManager* cleaner, bool overwrite)
{
    if (cleaner)
        cleaner->ref();

    Symbol* symbol;
    bool exists = find(key, symbol);
    if (exists && symbol->valid && !overwrite) {
        if (cleaner) {
            cleaner->link((void*) data);
            cleaner->unlink((void*) data);
            cleaner->unref();
        }
        return false;
    }

    if (!exists) {
        symbol = new Symbol;
        enter(key, symbol);
    } else if (symbol->cleaner) {
        symbol->cleaner->unlink(symbol->data);
        symbol->cleaner->unref();
    }

    symbol->valid = true;
    symbol->data = (void*) data;
    symbol->cleaner = cleaner;
    if (cleaner) 
        cleaner->link((void*) data);

    return true;
}

bool SymbolTable::remove(const char* key)
{
    Symbol* symbol;
    if (!find(key, symbol))
        return false;

    if (symbol->cleaner) {
        symbol->cleaner->unlink(symbol->data);
        symbol->cleaner->unref();
        symbol->cleaner = NULL;
    }
    symbol->data = NULL;
    symbol->valid = false;

    return true;
}

static void clear_symbol(const char*, Symbol* symbol)
{
    if (symbol->cleaner) {
        symbol->cleaner->unlink(symbol->data);
        symbol->cleaner->unref();
        symbol->cleaner = NULL;
    }
    symbol->valid = false;
    symbol->data = NULL;
}

void SymbolTable::clear()
{
    applyToAll(clear_symbol);
}

int SymbolTable::getInt(const char* name) const
{
    void* value = get(name);
    if (!value)
        return 0;
    int res;
    if (sscanf((char*) value, "%d", &res) != 1)
        return 0;
    return res;
}
    
float SymbolTable::getFloat(const char* name) const
{
    void* value = get(name);
    if (!value)
        return 0;
    float res;
    if (sscanf((char*) value, "%f", &res) != 1)
        return 0;
    return res;
}
    
double SymbolTable::getDouble(const char* name) const
{
    void* value = get(name);
    if (!value)
        return 0;
    double res;
    if (sscanf((char*) value, "%lf", &res) != 1)
        return 0;
    return res;
}

void* SymbolTable::get(const char* key) const
{
    if (!key)
        return NULL;

    Symbol* symbol;
    if (find(key, symbol) && symbol->valid)
        return symbol->data;
    else
        return NULL;
}

bool SymbolTable::get(const char* key, void*& value) const
{
    Symbol* symbol;
    if (find(key, symbol) && symbol->valid) {
        value = symbol->data;
        return true;
    } else
        return false;
}

__UTILS_END_NAMESPACE
