/***************************************************************************

                                Player.cc

  Implements a class can be used to read values from a canned data file
  with formatted records.

  Classes defined for export:
    Player - the playback class

***************************************************************************/

#include <utils/Player.h>
#include <utils/Input.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/UnpackSizeAction.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/DeleteAction.h>

__UTILS_BEGIN_NAMESPACE

class PlayElem {
public:
  PlayElem(FormatSpec* spec, void* address);
  ~PlayElem();

  int unpackData(unsigned char* src, int src_size,
                 int byte_order, int alignment);
  void release(unsigned char* buffer, int size);

  FormatSpec* getSpec() const { return _spec; }

protected:
  FormatSpec* _spec;
  void* _address;
};

PlayElem::PlayElem(FormatSpec* spec, void* address)
{
  spec->ref();
  _spec = spec;
  _address = address;
}

PlayElem::~PlayElem()
{
  _spec->unref();
}

int PlayElem::unpackData(unsigned char* src, int src_size,
                         int align, int byte_order)
{
  if (!_address) {
    return UnpackSizeAction::unpackSize(_spec, src, byte_order);
  } else {
    if (_spec->isPackedType() && byte_order == BYTE_ORDER) {
      memcpy((char*) _address, (char*) src, _spec->dataSize());
      return _spec->dataSize();
    } else {
      UnpackAction action(src, src_size, 0, _address, align, byte_order);
      if (!_spec->act(&action))
        return 0;
      return action.getIndex();
    }
  }
}

void PlayElem::release(unsigned char* buffer, int size)
{
  DeleteAction::deleteStruct(_spec, _address, buffer, size);
}
  

Player::Player()
{
  FormatAction::initClasses();
  _parser = new FormatParser();
  _buf_size = 200;
  _buffer = new unsigned char[_buf_size];
}

Player::~Player()
{
  StringDictIterator<PlayElem*> iter(_elems);
  for (PlayElem* current = iter.first(); current; current = iter.next()) 
    delete current;
  _elems.clear();
  delete [] _buffer;
  delete _parser;
}

PlayElem* Player::expect(const char* name, const char* format,
                    void* address)
{
  FormatSpec* spec = _parser->parseString(format);
  if (!spec) {
    fprintf(stderr, "Invalid format string '%s'\n", format);
    return NULL;
  }

  PlayElem* elem = new PlayElem(spec, address);
  _elems.enter(name, elem);

  return elem;
}

bool Player::open(const char* name, bool do_setup)
{
  try {
    _reader.open(name);
    FILE* fp = _reader.getHeader();
    int header_size = _reader.getHeaderSize();
    if (_buf_size < header_size) {
      _buf_size = header_size;
      delete [] _buffer;
      _buffer = new unsigned char[_buf_size];
    }
    if (fread(_buffer, 1, header_size, fp) < (unsigned int) header_size) {
      printf("Can't read header\n");
      return false;
    }
    Input input;
    input.setBuffer(_buffer, header_size);
    if (!_header.parse(input))
      return false;
    _reader.first();
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }

  if (do_setup)
    return setup();
  else
    return true;
}

bool Player::setup()
{
  _byte_order = _header.getInt("DataFormat.byte_order", BYTE_ORDER);
  _alignment = _header.getInt("DataFormat.alignment", ALIGN);

  int num_elems = _header.numValues("DataFormat.names");
  if (!num_elems) {
    printf("No log elements\n");
    return false;
  }
  const char** names = new const char*[num_elems];
  const char** formats = new const char*[num_elems];

  if (_header.getStrings("DataFormat.names", names, num_elems) != num_elems ||
      _header.getStrings("DataFormat.formats", formats,
                         num_elems) != num_elems) {
    printf("Mismatch getting header names and formats\n");
    delete [] names;
    delete [] formats;
    return false;
  }

  bool res = true;
  for (int i=0; i < num_elems; i++) {
    FormatSpec* spec = _parser->parseString(formats[i]);
    if (!spec) {
      printf("Error parsing header format '%s'\n", formats[i]);
      res = false;
      break;
    }
    PlayElem* elem;
    if (!_elems.find(names[i], elem)) {
      elem = new PlayElem(spec, NULL);
      _elems.enter(names[i], elem);
    } else {
      spec->ref();
      bool equals = FormatEqualsAction::equals(spec, elem->getSpec());
      spec->unref();
      if (!equals) {
        printf("Format %s (%s) does not equal declared format\n",
               formats[i], names[i]);
        spec->unref();
        res = false;
        break;
      }
    }
    _input.append(elem);
  }

  delete [] names;
  delete [] formats;

  return res;
}

bool Player::process(Time& time)
{
  int size = _reader.getRecordSize();
  if (size > _buf_size) {
    delete [] _buffer;
    _buf_size = size;
    _buffer = new unsigned char[_buf_size];
  }

  uint64_t secs, usecs;
  _reader.getCurTimeStamp(&secs, &usecs);
  time.setValue(secs, usecs);
  if (time != _last_time) {
    if (fread(_buffer, 1, size, _reader.getRecord()) != (unsigned int) size) {
      printf("Mismatch reading record\n");
      return false;
    }
  }

  ListIterator<PlayElem> iter(_input);
  int index = 0;
  for (PlayElem* elem = iter.first(); elem; elem = iter.next()) {
    int cur_size = elem->unpackData(_buffer + index, size-index,
                                    _alignment, _byte_order);
    if (!cur_size) {
      printf("Problem unpacking element\n");
      return false;
    }
    index += cur_size;
  }

  _last_time = time;
  return true;
}

bool Player::get(Time& time)
{
  time -= _offset;
  long secs, usecs;
  time.getValue(secs, usecs);

  try {
    _reader.seekBefore(secs, usecs);
    if (_reader.eof()) {
      time = Time();
      return false;
    }
    bool res = process(time);
    time += _offset;
    return res;
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }
}

bool Player::next()
{
  try {
    _reader.next();
    if (_reader.eof()) {
      return false;
    }
    return true;
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }
}

bool Player::prev()
{
  try {
    _reader.prev();
    return true;
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }
}

void Player::release(PlayElem* elem)
{
  elem->release(_buffer, _buf_size);
}

Time Player::getStart()
{
  uint64_t secs, usecs;
  _reader.getFirstTimeStamp(&secs, &usecs);
  return Time(secs, usecs);
}

Time Player::getEnd()
{
  uint64_t secs, usecs;
  _reader.getLastTimeStamp(&secs, &usecs);
  return Time(secs, usecs);
}

void Player::setZero(Time zero_time)
{
  _zero_time = zero_time;
  _offset = _zero_time - getStart();
}

bool Player::eof()
{
  return _reader.eof();
}

void Player::first()
{
  _reader.first();
}

void Player::getCurTime(utils::Time& time)
{
  uint64_t secs, usecs;
  _reader.getCurTimeStamp(&secs, &usecs);
  time.setValue(secs, usecs);
}

__UTILS_END_NAMESPACE
