///////////////////////////////////////////////////////////////////////////////
//
//                               Input.cc
//
// Implements a class that takes inputs from either a file or from a buffer.
// The class can open files by name, and can be given a set of directories
// to search for the file.  Once an input source is established, the class
// provides methods for reading in strings and numbers, while skipping white
// space and skipping over comments (started by '#')
//
// Classes implemented for export:
//     Input - the input class
//
// Classes implemented for internal use
//     InputFile - a structure for holding an input source
//    
///////////////////////////////////////////////////////////////////////////////

#define _LARGEFILE_SOURCE 1
#define _LARGEFILE64_SOURCE 1
#define _FILE_OFFSET_BITS 64

#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <utils/Input.h>

__UTILS_BEGIN_NAMESPACE
    
// Create an empty input source
File::File()
{
    name = "";
    fullName = "";
    fp = (FILE*) NULL;
    buffer = NULL;
    bufSize = 0;
    lineNum = 0;
    openedHere = false;
}

// The directories list is static
Vector<String*>* Input::_directories = 0L;

// Add a directory to the beginning of the load path
void Input::addDirectoryFirst(const char* dirName)
{
    if (!_directories)
        _directories = new Vector<String*>(1);
    _directories->prepend(new String(dirName));
}

// Add a directory to the end of the load path
void Input::addDirectoryLast(const char* dirName)
{
    if (!_directories)
        _directories = new Vector<String*>(1);
    _directories->append(new String(dirName));
}

// Add all of the directories in the environment variable to the beginning
// of the load path.  The names are either separated by spaces or colons
void Input::addEnvDirectoriesFirst(const char* envVarName)
{
    // get the environment variable, return if non-existent
    const char* dir_name;
    const char* ptr = getenv(envVarName);
    if (!ptr)
        return;

    Vector<char> dir_string(30);
    Vector<String*> temp_list(3);

    // parse the environment variable
    while (*ptr) {
        while ((isspace(*ptr) || (*ptr == ':')) && *ptr) ptr++;
        if (!*ptr)
            break;
        dir_name = ptr;
        while (!isspace(*ptr) && (*ptr != ':') && *ptr) ptr++;
        dir_string.setValue(ptr-dir_name, '\0');
        memcpy(dir_string.getData(), dir_name, ptr-dir_name);
        temp_list.append(new String(dir_string.getData()));
    }

    // append environment variables in reverse order (so that they end up
    // in the proper order in the load path)
    int i;
    for (i=temp_list.numElems()-1;i>=0;i--) {
        addDirectoryFirst(temp_list[i]->getString());
        delete temp_list[i];
    }
}
            
// Add all of the directories in the environment variable to the end
// of the load path.  The names are either separated by spaces or colons
void Input::addEnvDirectoriesLast(const char* envVarName)
{
    // get the environment variable, return if non-existent
    const char* dir_name;
    const char* ptr = getenv(envVarName);
    if (!ptr)
        return;

    Vector<char>* dir_string = new Vector<char>(30);

    // parse the directories from the environment variable and add them to
    // the end of the load path
    while (*ptr) {
        while ((isspace(*ptr) || (*ptr == ':')) && *ptr) ptr++;
        if (!*ptr)
            break;
        dir_name = ptr;
        while (!isspace(*ptr) && (*ptr != ':') && *ptr) ptr++;
        dir_string->setValue(ptr-dir_name, '\0');
        memcpy(dir_string->getData(), dir_name, ptr-dir_name);
        addDirectoryLast(dir_string->getData());
    }
    delete dir_string;
}

// remove a directory by name from the load path            
void Input::removeDirectory(const char* dirName)
{
    if (!_directories)
        return;

    int i;
    for (i=0;i<_directories->numElems();i++) {
        if ((*(*_directories)[i]) == dirName) {
            delete (*_directories)[i];
            _directories->remove(i);
            return;
        }
    }
}

void Input::popDirectory()
{
    if (!_directories)
        return;

    if (_directories->numElems()) {
        delete (*_directories)[0];
        _directories->remove(0);
    }
}

void Input::pushDirectoryOfFile(const char* file_name)
{
    String file_dir;
    const char* slash = strrchr(file_name, '/');
    if (slash) {
        file_dir = String(file_name, 0, slash-file_name-1);
    } else {
        file_dir = ".";
    }
    addDirectoryFirst(file_dir.getString());
}

// clear the load path
void Input::clearDirectories()
{
    if (!_directories)
        return;

    int i;
    // delete each entry
    for (i=0;i<_directories->numElems();i++)
        delete (*_directories)[i];
    // and remove the list
    delete _directories;
    _directories = 0L;
}

// print out the load path to the file fp
void Input::printDirectories(FILE* fp) 
{
    if (!_directories) {
        fprintf(fp, "( )");
        return;
    }

    fprintf(fp, "( ");
    int i;
    for (i=0;i<_directories->numElems();i++)
        fprintf(fp, "%s ", (*_directories)[i]->getString());
    fprintf(fp, ")");
}

// get the load path as a string list
const Vector<String*> &Input::getDirectories() 
{
    if (!_directories)
        _directories = new Vector<String*>(1);

    return *_directories;
}

// create an Input.  Initializes the backup buffer to have a maximum length
// of 12 characters, and the temporary string storage has a maximum length of
// 20 characters.  Sets the _cur_file member to be an empty source, so the
// Input has no source initially
Input::Input() : _back_buf(12), _temp_store(20)
{
    _cur_file = new File;
}

// destroy an Input
Input::~Input()
{
    delete _cur_file;
}

// set the input to be the file with structure newFP
void Input::setFilePointer(FILE* newFP)
{
    initFile(newFP, "", (String*) NULL, false);
}

// open the file fileName.  If fileName is not absolute, search for the file 
// in every directory in the load path.  If okIfNotFound (default is false), 
// don't print out anything if the open fails, otherwise, print out that there
// was a problem.  If the file is opened successfully, return true regardless
// of the value of okIfNotFound
bool Input::openFile(const char* fileName, bool okIfNotFound)
{
    // skip whitespace in file name
    while (isspace(*fileName)) fileName++;

    if (*fileName == '/') {   // is file name absolute?
        // yes, just try and open it
        FILE* fp = fopen(fileName, "rb");   
        if (!fp) {
            if (!okIfNotFound) {
                perror("Input");
                fprintf(stderr, "Input: File %s not found\n", fileName);
            }
            return false;
        }
        initFile(fp, fileName, (String*) NULL, true);
    } else {
        // no, try and find file along the load path and open it
        String full_name;
        FILE* fp = findFile(fileName, full_name);
        if (!fp) {
            if (!okIfNotFound) {
                perror("Input");
                fprintf(stderr, "Input: File %s not found in path ",
                        fileName);
                printDirectories(stderr);
                fprintf(stderr, "\n");
            }
            return false;
        }
        initFile(fp, fileName, &full_name, true);
    }

    return true;
}

// Given a file pointer newFP with file name fileName (and expanded full name
// fullName, a boolean indicating whether Input opened it or not, 
// openedHere, set up the File member.  
void Input::initFile(FILE* newFP, const char* fileName, String* fullName,
                     bool openedHere)
{
    // if file is already open, close it
    if (_cur_file->fp)
        closeFile();

    // set up file source
    _cur_file->fp = newFP;
    _cur_file->name = fileName;
    if (fullName)
        _cur_file->fullName = *fullName;
    else
        _cur_file->fullName = fileName;
    _cur_file->openedHere = openedHere;

    // zero out all of the necessary buffers and indices to indicate that
    // this is not a buffer source
    _back_buf.clear();
    _cur_file->buffer = NULL;
    _cur_file->curBuf = 0L;
    _cur_file->bufSize = 0;
    _cur_file->lineNum = 1;
}

// Tries to use the load path to open fileName.  If there are no entries on
// the load path, just use the original fileName.  If it can successfully
// open a file, fullName is set to the full name of the file and the FILE
// structure is returned, if not, NULL is returned
FILE* Input::findFile(const char *fileName, String &fullName) const
{
    FILE* fp = (FILE*) NULL;
    if (!_directories || _directories->numElems() == 0) {
        fp = fopen(fileName, "rb");
        fullName = fileName;
    } else {
        int i;
        for (i=0;i<_directories->numElems();i++) {
            fullName = *(*_directories)[i];
            fullName += "/";
            fullName += fileName;
            fp = fopen(fullName.getString(), "rb");
            if (fp)
                break;
        }
    }
    return fp;
}

// close the current source, whether it be a file or a buffer
void Input::closeFile()
{
    if (_cur_file->fp) {
        fclose(_cur_file->fp);
        _cur_file->fp = (FILE*) NULL;
        _back_buf.clear();
        _cur_file->name = _cur_file->fullName = "";
    }
}

void Input::releaseFile()
{
  if (_cur_file->fp) {
    _cur_file->fp = (FILE*) NULL;
    _back_buf.clear();
    _cur_file->name = _cur_file->fullName = "";
  }
}


// Get the current FILE pointer
FILE* Input::getCurFile() const 
{
    return _cur_file->fp;
}

// get the full name of the current file.  If there is no file, it will
// return ""
const char* Input::getCurFileName() const
{
    return _cur_file->fullName.getString();
}

int Input::getLineNum() const
{
    return _cur_file->lineNum;
}

void Input::setLineNum(int num) 
{
    _cur_file->lineNum = num;
}

// get the original name of the current file.  If there is no file, it will
// return ""
const char* Input::getOrigFileName() const
{
    return _cur_file->name.getString();
}

// Set the input source to be a buffer, bufPointer, with maximum size
// bufSize
void Input::setBuffer(void *bufPointer, size_t bufSize)
{
    closeFile();
    _cur_file->buffer = bufPointer;
    _cur_file->curBuf = (char*) bufPointer;
    _cur_file->bufSize = bufSize;
    _cur_file->lineNum = 1;
}

// get the number of bytes read from a buffer source.  Returns 0 if not
// a buffer source
size_t Input::getNumBytesRead() const
{
    return _cur_file->curBuf - (char*) _cur_file->buffer;
}

// get the next character from the source, whether it be buffer or file
bool Input::get(char& c)
{
    // report from the backup buffer first
    if (_back_buf.numElems()) {
        c = _back_buf[_back_buf.numElems()-1];
        _back_buf.remove(_back_buf.numElems()-1);
        return true;
    }

    // figure out what kind of source it is, and return a character
    bool res;
    if (fromBuffer())
        res = getASCIIBuffer(c);
    else
        res = getASCIIFile(c);
    if (res) {
        if (c == '\n')  // increment line number as necessary
            _cur_file->lineNum++;
        return true;
    } else
        return false;
}

// get a character from the buffer.  Return false if can't do it
bool Input::getASCIIBuffer(char& c)
{
    if (freeBytesInBuf() > 0) {
        c = *(_cur_file->curBuf)++;
        return true;
    } else
        return false;
}

// get a character from the FILE, Returns false if not a file or at EOF
bool Input::getASCIIFile(char& c)
{
    if (_cur_file->fp) {
        char tmp;
        tmp = fgetc(_cur_file->fp);
        if (tmp==(char) EOF)
            return false;
        c=tmp;
        return true;
    } else
        return false;
}

// returns if EOF found or read past the end of the buffer (depending on the 
// source)
bool Input::eof() const
{
    if (_back_buf.numElems())
        return true;

    if (fromBuffer()) 
        return freeBytesInBuf() <= 0;
    else
        return (!_cur_file->fp) || feof(_cur_file->fp) != 0;
}

// put a chararacter in the backup buffer
void Input::putBack(char c)
{
    _back_buf.append(c);
}

// put a whole string in the backup buffer
void Input::putBack(const char* string)
{
    int len;
    for (len = strlen(string); --len >= 0; ) 
        putBack(string[len]);
}

// Skip over spaces and comments.  Return false if at end of source
bool Input::skipWhiteSpace()
{
    char c = ' ';

    while (get(c) && (isspace(c) || c == '\0'));

    // check if no spaces to skip
    if (isspace(c) || c == '\0')
        return false;

    // check for comments
    if (c == '#') {
        // skip to end of line
        while (get(c) && c != '\n');
        if (eof())
            return false;
        return skipWhiteSpace();
    } else 
        putBack(c);   // make sure we don't munch anything important

    return true;
}

void Input::restore(char c)
{
    for (int i=0;i<_temp_store.numElems();i++)
        putBack(_temp_store[i]);
    if (c)
        putBack(c);
}

// Read a string of digits into temporary storage
void Input::readDigits()
{
    char c = ' ';
    while (get(c) && isdigit(c)) 
        _temp_store.append(c);

    if (!isspace(c))
        putBack(c);
}

// read in a signed integer.  Returns false on error.
bool Input::readInteger(int &l)
{
    skipWhiteSpace();     // skip leading spaces
    _temp_store.clear();  // clear temp storage

    // read in first character, which may be a - sign
    char c;
    if (!get(c)) {
        restore(c);
        return false;
    }
    if (c != '-' && !isdigit(c)) {
        restore(c);
        return false;
    }
    _temp_store.append(c);

    readDigits();  // read in rest of number

    // try and extract the number.
    _temp_store.append('\0');
    char* res_ptr;
    l = strtol(_temp_store.getData(), &res_ptr, 10);
    if (res_ptr == _temp_store.getData()) {
        restore();
        return false;
    } else
        return true;
}

// read in an unsigned integer.  Returns false on error.
bool Input::readUnsignedInteger(unsigned int &l)
{
    skipWhiteSpace();     // skip leading spaces
    _temp_store.clear();  // clear temp storage

    readDigits();    // read in the number

    // try and extract it
    _temp_store.append('\0');
    char* res_ptr;
    l = strtoul(_temp_store.getData(), &res_ptr, 10);
    if (res_ptr == _temp_store.getData()) {
        restore();
        return false;
    } else
        return true;
}

// read in a real, i.e., floating point, number.  Returns false on error
bool Input::readReal(double& d)
{
    skipWhiteSpace();      // skip leading spaces
    _temp_store.clear();   // clear temp storage

    // read in - sign, if there is one
    char c;
    if (!get(c)) {
        return false;
    }
    if (c != '-' && !isdigit(c)) {
        restore(c);
        return false;
    }
    _temp_store.append(c);

    // read in whole number portion of the mantissa
    readDigits();

    if (get(c)) {
        if (c == '.') { // read in optional fractional part of mantissa
            _temp_store.append(c);
            readDigits();
        } else
            putBack(c);

        if (get(c)) {  // read in optional exponent part
            if (c == 'E' || c == 'e') {
                _temp_store.append(c);
                if (!get(c) && c!='-' && !isdigit(c)) {
                    restore(c);
                    return false;
                }
                _temp_store.append(c);
                readDigits();
            } else
                putBack(c);
        }
    }
        
    // extract the number
    _temp_store.append('\0');
    char* res_ptr;
    d = strtod(_temp_store.getData(), &res_ptr);
    if (res_ptr == _temp_store.getData()) {
        restore();
        return false;
    } else
        return true;
}

// read in the next non-whitespace character
bool Input::read(char& c)
{
    if (!skipWhiteSpace())
        return false;

    return get(c);
}

// read in a string, either started and terminated by whitespace or contained
// in quotes
bool Input::read(String& s, char delimeter)
{
    if (!skipWhiteSpace())
        return false;

    _temp_store.clear();
    char c;
    if (!get(c))
        return false;

    if (c == '"') {   // read in string enclosed by quotes
        while (get(c) && c != '"')
            _temp_store.append(c);
    } else {  // read in string enclosed by whitespace 
        _temp_store.append(c);
        while (get(c) && !isspace(c) && c!=delimeter && c!='\0') 
            _temp_store.append(c);
    }
    _temp_store.append('\0');
    
    s = _temp_store.getData();
    return true;
}

// read in a name enclosed by whitespace.  Names have more specific structure, 
// so if validIdent is true, reject anything that is not valid name syntax
bool Input::read(Name& n, bool validIdent)
{
    if (!skipWhiteSpace())
        return false;

    char c;
    _temp_store.clear();
    if (validIdent) {  // verify is proper name syntax
        if (!get(c) || !Name::isIdentStartChar(c))
            return false;
        _temp_store.append(c);
        while (get(c) && !isspace(c)) {
            if (!Name::isIdentChar(c))
                return false;
            _temp_store.append(c);
        }
    } else {    // just read anything 
        while (get(c) && !isspace(c)) 
            _temp_store.append(c);
    }

    _temp_store.append('\0');
    n = _temp_store.getData();

    return true;
}

// make sure Inputs copy correctly
Input& Input::operator=(const Input& src)
{
    *_cur_file = *src._cur_file;
    _back_buf = src._back_buf;
    _temp_store = src._temp_store;

    return *this;
}

// routines that use the utilities to do the reading

bool Input::read(int& i)
{
    if (!readInteger(i)) {
        i = 0;
        return false;
    } else {
        return true;
    }
}

bool Input::read(short& i)
{
    int l;

    if (!readInteger(l)) {
        i = 0;
        return false;
    } else {
        i = l;
        return true;
    }
}

bool Input::read(unsigned int& i)
{
    unsigned int l;

    if (!readUnsignedInteger(l)) {
        i = 0;
        return false;
    } else {
        i = l;
        return true;
    }
}

bool Input::read(unsigned short& i)
{
    unsigned int l;

    if (!readUnsignedInteger(l)) {
        i = 0;
        return false;
    } else {
        i = l;
        return true;
    }
}

bool Input::read(double& d)
{
    if (!readReal(d)) {
        d = 0;
        return false;
    } else 
        return true;
}

bool Input::read(float& f)
{
    double d;

    if (!readReal(d)) {
        f = 0;
        return false;
    } else {
        f = d;
        return true;
    }
}

#define CHUNK_SIZE 1024

bool Input::readAll(unsigned char*& bytes, int& num_elems)
{
    FILE* fp = getCurFile();
    if (!fp)
        return false;
    int max_size = num_elems;
    int size; 
    char buffer[CHUNK_SIZE];
    while ((size=fread(buffer, sizeof(unsigned char), CHUNK_SIZE, fp))>0) {
        if (size + num_elems >= max_size) {
            int new_max_size = size+num_elems+1;
            if (max_size) {
                unsigned char* new_bytes = new unsigned char[new_max_size];
                memcpy((char*) new_bytes, (char*) bytes, max_size);
                delete bytes;
                bytes = new_bytes;
            } else 
                bytes = new unsigned char[new_max_size];
            max_size = new_max_size;
        }
                
        memcpy((char*) bytes+num_elems, (char*) buffer, 
               size*sizeof(unsigned char));
        num_elems += size;
    }
    if (num_elems < max_size) {
        bytes[num_elems] = '\0';
        num_elems++;
    }
    return true;
}

bool Input::readWord(String& s, const char* word_chars)
{
    if (!skipWhiteSpace())
        return false;

    _temp_store.clear();
    char c;
    if (!get(c))
        return false;
    
    char end;
    if (c == '"' || c == '\'') {   // read in string enclosed by quotes
        end = c;
        while (get(c) && c != end)
            _temp_store.append(c);
    } else { // read in a string enclosed by non-alphanumeric stuff
        _temp_store.append(c);
        while (1) {
            if (get(c)) {
                if (isalnum(c) || strchr(word_chars, c))
                    _temp_store.append(c);
                else {
                    putBack(c);
                    break;
                }
            } else
                break;
        }
    }

    _temp_store.append('\0');
    
    s = _temp_store.getData();
    return true;
}

int Input::read(unsigned char* data, int size)
{
    int res = 0;
    for (;size && _back_buf.numElems();size--) {
        *data++ = _back_buf[_back_buf.numElems()-1];
        _back_buf.remove(_back_buf.numElems()-1);
        res++;
    }

    if (!size)
        return res;

    if (fromBuffer()) {
        char c;
        for (;size;size--) {
            if (!getASCIIBuffer(c))
                return res;
            *data++ = c;
            res++;
        }
        return res;
    } else {
        int size_read =  fread(data, 1, size, _cur_file->fp);
        if (size_read < 0)
            return res;
        return res+size_read;
    }
}

off_t Input::getPos(void)
{
	off_t retval;
	if (fromBuffer()) {
		retval = (off_t)(_cur_file->curBuf - (char *)_cur_file->buffer);
	} else {
		retval = (off_t)ftell(_cur_file->fp);
	}
	return retval;
}

off_t Input::getSize(void)
{
	off_t retval;
	if (fromBuffer()) {
		retval = _cur_file->bufSize;
	} else {
		struct stat stat;
		if (fstat(fileno(_cur_file->fp), &stat) == 0) {
			retval = stat.st_size;
		} else {
			perror("Input");
			fprintf(stderr, "Input: fstat failed!\n");
			retval = 0;
		}
	}
	return retval;
}

int Input::seek(off_t offset, int whence)
{
	int retval = 0;
	if (fromBuffer()) {
		switch(whence) {
		case SEEK_SET:
			_cur_file->curBuf = ((char *)_cur_file->buffer + offset);
			break;
		case SEEK_END:
			_cur_file->curBuf = ((char *)_cur_file->buffer
					     + _cur_file->bufSize + offset);
			break;
		case SEEK_CUR:
			_cur_file->curBuf += offset;
			break;
		default:
			errno = EINVAL;
			retval = -1;
		}
	} else {
		retval = fseek(_cur_file->fp, offset, whence);
	}
	return retval;
}


__UTILS_END_NAMESPACE
