///////////////////////////////////////////////////////////////////////////////
//
//                               Format.cc
//
// Implements standard format specifiers.  All standard format specifiers have
// corresponding member functions of FormatAction to process them 
// appropriately for different subclasses of FormatAction.  Custom format
// specifiers should use the facilities defined by CustomSpec
//
// Classes implemented for export:
//   FormatSpec - the format specifier base class
//   IntSpec, ShortSpec, LongSpec, FloatSpec, CharSpec, DoubleSpec,
//     StringSpec - primitive, standard format specifiers
//   StructSpec - a collection of data types, i.e. a struct
//   FixedArraySpec - a fixed number of a data type
//   VarArraySpec - a variable number of a data type
//   PtrSpec - specifies pointer to data
//   SelfPtrSpec - pointer to the StructSpec that contains it
//   LengthSpec - specified an unstructured run of bytes with a fixed length
//   
///////////////////////////////////////////////////////////////////////////////

#include <utils/formatting/Format.h>
#include <utils/formatting/FlatAction.h>
#include <utils/formatting/DataSizeAction.h>
#include <utils/Vector.h>

__UTILS_BEGIN_NAMESPACE

// define convenience macros for specifying primitive formats
#define UTILS__IMPL_SPEC(name) \
UTILS_BASE_SOURCE(UTILS__SPEC_NAME(name)); \
void UTILS__SPEC_NAME(name)::initClass() { \
    UTILS_BASE_INIT_CLASS(UTILS__SPEC_NAME(name), \
                       build_name(UTILS__QUOTE(name), "Spec").getString(), \
                       "FormatSpec"); \
} \
UTILS__SPEC_NAME(name)::UTILS__SPEC_NAME(name)() { \
    UTILS_BASE_CONSTRUCTOR(UTILS__SPEC_NAME(name)); \
} \
bool UTILS__SPEC_NAME(name)::act(FormatAction* action) \
{ return action->UTILS__CONCAT(act,name)(); }

// used in the convenience macro to build the primitive format specifier name
static String build_name(const char* str1, const char* str2)
{                                                         
    String storage;
    storage = str1;
    storage += str2;

    return storage;
}

// implement the primitive format specifiers
UTILS__IMPL_SPEC(Int);       //  IntSpec
UTILS__IMPL_SPEC(Short);     //  ShortSpec
UTILS__IMPL_SPEC(Long);      //  LongSpec
UTILS__IMPL_SPEC(Float);     //  FloatSpec
UTILS__IMPL_SPEC(Char);      //  CharSpec
UTILS__IMPL_SPEC(Double);    //  DoubleSpec
UTILS__IMPL_SPEC(String);    //  LongSpec


///////////////////////////////////////////////////////////////////////////
// FormatSpec - the format specifier base class
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_ABSTRACT_SOURCE(FormatSpec);

void FormatSpec::initClass()
{
    UTILS_BASE_INIT_ABSTRACT_CLASS(FormatSpec, "FormatSpec", "Base");
}

FormatSpec::FormatSpec()
{
    UTILS_BASE_CONSTRUCTOR(FormatSpec);

    _size = -1;
    _flat = -1;
}

// convenience routine to find if a format specifier is flat
bool FormatSpec::isFlatType() const
{
    if (_flat != -1)   // flatness is cached
        return (_flat == 1);
    
    // just appy the flat query action and return the result
    FlatAction action;
    ((FormatSpec*) this)->act(&action);
    if (action.isFlat()) // cache flatness
        ((FormatSpec*) this)->_flat = 1;
    else
        ((FormatSpec*) this)->_flat = 0;

    return _flat == 1;
}

// convenience routine to find the size of the data structure defined by the
// format specifier.  
int FormatSpec::dataSize() const
{
    // The size is cached so we only have to apply the DataSize action once
    if (_size == -1) {
        DataSizeAction action;
        ((FormatSpec*) this)->act(&action);
        ((FormatSpec*) this)->_size = action.getSize();
        ((FormatSpec*) this)->_packed = action.isPacked();
    }
    return _size;
}

// returns TRUE if the format specified will be "prepacked," i.e., there is
// no alignment padding in the data
bool FormatSpec::isPackedType() const
{
    if (_size == -1)
        dataSize();
    return _packed;
}


///////////////////////////////////////////////////////////////////////////
// StructSpec - Specifies a flat collection of data types, i.e., a C struct
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(StructSpec);

// A StructSpec is created with no contents        
StructSpec::StructSpec()
{
    UTILS_BASE_CONSTRUCTOR(StructSpec);

    _size = 0;
    _fmts = 0L;
}

// specify the structure as the list of formats, fmts
void StructSpec::setStruct(const Vector<FormatSpec*>& fmts)
{
    FormatSpec** array = new FormatSpec*[fmts.numElems()];
    memcpy((char*) array, (char*) fmts.getData(),
           sizeof(FormatSpec*)*fmts.numElems());
    
    _size = fmts.numElems();
    _fmts = array;

    for (int i = 0; i<_size; i++) 
        if (_fmts[i])
            _fmts[i]->ref();
}

void StructSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(StructSpec, "StructSpec", "FormatSpec");
}

// unreference all of the formats specified by the structure
StructSpec::~StructSpec()
{
    if (!_fmts)
        return;
    for (int i = 0; i<_size; i++) 
        if (_fmts[i])
            _fmts[i]->unref();
    delete [] _fmts;
}    

// Since StructSpec is a standard, there is a action member function to
// deal with it
bool StructSpec::act(FormatAction* action)
{
    return action->actStruct(_size,_fmts);
}


///////////////////////////////////////////////////////////////////////////
// FixedArraySpec - Specifies a fixed size array of a data type
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(FixedArraySpec);

// initialize with an array of dimensions and the format which specifies
// the data type of the array.
FixedArraySpec::FixedArraySpec(Vector<int>* sizes, FormatSpec* fmt)
{
    UTILS_BASE_CONSTRUCTOR(FixedArraySpec);
    _sizes = sizes;
    _fmt = fmt;
    if (_fmt)
      _fmt->ref();
}

FixedArraySpec::FixedArraySpec()
{
    UTILS_BASE_CONSTRUCTOR(FixedArraySpec);
    _sizes = 0L;
    _fmt = (FormatSpec*) NULL;
}

FixedArraySpec::~FixedArraySpec()
{
    if (_fmt) {
        _fmt->unref();
        delete _sizes;
    }
}

void FixedArraySpec::initClass()
{
    UTILS_BASE_INIT_CLASS(FixedArraySpec, "FixedArraySpec", "FormatSpec");
}

// Since FixedArraySpec is a standard, there is a action member function to
// deal with it
bool FixedArraySpec::act(FormatAction* action)
{
    return action->actFixedArray(*_sizes, _fmt);
}


///////////////////////////////////////////////////////////////////////////
// FixedArraySpec - Specifies a fixed size array of a data type
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(VarArraySpec);

// create a variable sized array, which exists within the data structure
// specifed by parent, which has dimensions specifed by the integers at
// the indices specified in the indices array, and whose data type is 
// specified by fmt
VarArraySpec::VarArraySpec(StructSpec* parent, Vector<int>* indices,
                               FormatSpec* fmt)
{
    UTILS_BASE_CONSTRUCTOR(VarArraySpec);

    _parent = parent;
    _indices = indices;
    _fmt = fmt;
    _fmt->ref();
}

VarArraySpec::VarArraySpec()
{
    UTILS_BASE_CONSTRUCTOR(VarArraySpec);
    _parent = 0L;
    _indices = 0L;
    _fmt = 0L;
}

VarArraySpec::~VarArraySpec()
{
    if (_fmt) {
        _fmt->unref();
        delete _indices;
    }
}

void VarArraySpec::initClass()
{
    UTILS_BASE_INIT_CLASS(VarArraySpec, "VarArraySpec", "FormatSpec");
}

// Since VarArraySpec is a standard, there is a action member function to
// deal with it
bool VarArraySpec::act(FormatAction* action)
{
    return action->actVarArray(_parent, *_indices, _fmt);
}


///////////////////////////////////////////////////////////////////////////
// PtrSpec - Specifies pointer to another data type
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PtrSpec);

PtrSpec::PtrSpec(FormatSpec* fmt)
{
    UTILS_BASE_CONSTRUCTOR(PtrSpec);
    _fmt = fmt;
    _fmt->ref();
}

PtrSpec::PtrSpec()
{
    UTILS_BASE_CONSTRUCTOR(PtrSpec);
    _fmt = 0L;
}

PtrSpec::~PtrSpec()
{
    if (_fmt) _fmt->unref();
}

void PtrSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(PtrSpec, "PtrSpec", "FormatSpec");
}

// Since PtrSpec is a standard, there is a action member function to
// deal with it
bool PtrSpec::act(FormatAction* action)
{
    return action->actPtr(_fmt);
}


///////////////////////////////////////////////////////////////////////////
// SelfPtrSpec - Specifies a pointer to the structure in which it resides
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(SelfPtrSpec);

SelfPtrSpec::SelfPtrSpec(StructSpec* parent)
{
    UTILS_BASE_CONSTRUCTOR(SelfPtrSpec);
    _parent = parent; 
}

SelfPtrSpec::SelfPtrSpec()
{
    UTILS_BASE_CONSTRUCTOR(SelfPtrSpec);
    _parent = 0L;
}

void SelfPtrSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(SelfPtrSpec, "SelfPtrSpec", "FormatSpec");
}

// Since SelfPtrSpec is a standard, there is a action member function to
// deal with it
bool SelfPtrSpec::act(FormatAction* action)
{
    return action->actSelfPtr(_parent);
}


///////////////////////////////////////////////////////////////////////////
// LengthSpec - Specifies an array of bytes of a fixed length
///////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(LengthSpec);

LengthSpec::LengthSpec(int n)
{
    UTILS_BASE_CONSTRUCTOR(LengthSpec);

    _length = n;
}

void LengthSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(LengthSpec, "LengthSpec", "FormatSpec");
}

// Since LengthSpec is a standard, there is a action member function to
// deal with it
bool LengthSpec::act(FormatAction* action)
{
    return action->actLength(_length);
}

__UTILS_END_NAMESPACE
