///////////////////////////////////////////////////////////////////////////////
//
//                               Dict.cc
//
// Implements class which is a dictionary of void* indexed by unsigned long 
// keys.  This is used as a base class of a dictionary template which goes
// from unsigned longs to <T*>
//
// Classes implemented for export:
//     DictBase - the base dictionary class
//     DictEntry - an entry in a base dictionary
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <ctype.h>

#include <utils/Dict.h>

__UTILS_BEGIN_NAMESPACE

DictBase::DictBase( int entries )
{
    _table_size=entries;
    _buckets = new DictEntry *[_table_size];

    for (int i = 0; i < _table_size; i++)
        _buckets[i] = (DictEntry*) NULL;
}

DictBase::~DictBase()
{
    clear();
    if ( _buckets != NULL )
        delete [] _buckets;
}

void DictBase::clear()
{
    int		i;
    DictEntry	*entry, *nextEntry;

    for (i = 0; i < _table_size; i++) {
        for (entry = _buckets[i]; entry != NULL; entry = nextEntry) {
            nextEntry = entry->next;
            delete entry;
        }
        _buckets[i] = (DictEntry*) NULL;
    }
}

bool DictBase::enter(unsigned long key, void *value)
{
    DictEntry		*&entry = findEntry(key);

    if (entry == NULL) {
        entry = new DictEntry(key, value);
        
        entry->next = (DictEntry*) NULL;
        return true;
    } else {
        entry->value = value;
        return false;
    }
}

bool DictBase::find(unsigned long key, void *&value) const
{
    DictEntry		*&entry = findEntry(key);

    if (entry == NULL) {
        value = NULL;
        return false;
    } else {
        value = entry->value;
        return true;
    }
}

DictEntry *&DictBase::findEntry(unsigned long key) const
{
    DictEntry		**entry;

    entry = &_buckets[key % _table_size];

    while (*entry != NULL) {
        if ((*entry)->key == key)
            break;
        entry = &(*entry)->next;
    }
    return *entry;
}

bool DictBase::remove(unsigned long key)
{
    DictEntry		*&entry = findEntry(key);
    DictEntry		*tmp;

    if (entry == NULL)
        return false;
    else {
        tmp = entry;
        entry = entry->next;
        delete tmp;
        return true;
    }
}

bool DictBase::isEmpty()
{
    for ( int i = 0; i < _table_size; i++ ) {
        if ( _buckets[i] != NULL )
            return false;
    }
    return true;
}

void DictBase::applyToAll(void (*rtn)(unsigned long key, void *value))
{
    for (int i=0;i<_table_size;i++) 
        for (DictEntry* buf = _buckets[i];buf;buf = buf->next) 
            (*rtn)(buf->key, buf->value);
}

void DictBase::applyToAll(void (*rtn)(unsigned long key, void *value,
                                               void *data), void *data )
{
    for (int i=0;i<_table_size;i++) 
        for (DictEntry* buf = _buckets[i];buf;buf = buf->next) 
            (*rtn)(buf->key, buf->value, data);
}

DictIteratorBase::DictIteratorBase(DictBase& dict) : _dict(dict)
{
  _index = -1;
  _next = NULL;
}

void* DictIteratorBase::first()
{
  _index = -1;
  _next = NULL;
  return next();
}

void* DictIteratorBase::next()
{
  while (1) {
    if (!_next) {
      while (1) {
        _index++;
        if (_index >= _dict._table_size)
          return NULL;
        _next = _dict._buckets[_index];
        if (_next)
          return _next->value;
      }
    }
    _next = _next->next;
    if (_next)
      return _next->value;
  }
}

__UTILS_END_NAMESPACE
