%{
#include <ctype.h>

#include <utils/Type.h>
#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/Vector.h>
#include <utils/ConfigElem.h>
#include <utils/StructElem.h>
#include <utils/UnknownElem.h>
#include <utils/ConfigFile.h>

%}

%header{
#include <utils/String.h>
#include <utils/List.h>
__UTILS_BEGIN_NAMESPACE
class ConfigElem; 
class StructElem;
class Input;
struct WorkingStruct {
  StructElem* context;
  ConfigElem* elem;
  char* name;
};
%}

%name ConfigParser

%union {
  int num;
  short type;
  String *str;
  ConfigElem* elem;
  StructElem* structure;
  WorkingStruct working;
}

%define CONSTRUCTOR_PARAM Input& input, StructElem* elem
%define MEMBERS Input& _input; StructElem* _root; String _input_word; \
	bool errorPrinted; \
	int _first_eof; ConfigElem* _working; bool _reading_value; \
	StructElem* _context; Type _expected_type; \
        char* _tmp_buf; int _size_tmp_buf; \
	List<StructElem> _context_stack; \
	ConfigElem* make_elem(const char*, int); \
	ConfigElem* make_merge_elem(const char*); \
	StructElem* get_struct(const char*, int); \
	bool get_struct(String*); bool read_value(String*); \
	bool read_symbol(ConfigElem*); \
        bool prepare_subcontext(); void cleanup_subcontext(); \
	bool make_tag(StructElem*, String*); \
	int eof(); \
	ConfigElem* input_file(const char*); \
        void add_working(WorkingStruct&, ConfigElem*); \
        virtual ~ConfigParser() { delete [] _tmp_buf; } 
%define CONSTRUCTOR_INIT : _input(input), _root(elem),  errorPrinted(false), \
	_first_eof(false), _working(NULL), \
	_reading_value(false), _context(elem), \
        _tmp_buf(NULL), _size_tmp_buf(0)
%define YYERROR_VERBOSE 1

%token <type> TYPE
%token <str> STRING
%token <num> NUMBER
%token INCLUDE
%token FIRST_EOF
%token <str> MERGE_STRUCT
%token <elem> SYMBOL
%type <working> name
%type <working> merge_name
%type <type> type;
%type <structure> close_bracket;
%type <elem> open_bracket;

%%

body: /* empty */ { _working = _root; }
    | STRING { if (!make_tag(_context, $1)) YYERROR; }
    | STRING ':'  decl_list { if (!make_tag(_context, $1)) YYERROR; }
    | STRING ':' { if (!make_tag(_context, $1)) YYERROR; }
    | decl_list  { }
    | body FIRST_EOF
    | '{' body '}'
;

decl_list:  decl  { }
    | decl decl_list { }
;

decl: ';' {}
      | type name equals value_list { add_working($2,_working); }
      | name equals value_list { add_working($1,_working); }
      | type name open_bracket body close_bracket { add_working($2,$3);}
      | name open_bracket body close_bracket { add_working($1,$2);}
      | merge_struct merge_name open_bracket body close_bracket {
            if ($2.context) add_working($2, $3);
            else delete [] $2.name;
         }
      | INCLUDE { _reading_value = true; if(!_working) _working = _root; }
        STRING {
          if (!input_file($3->getString())) {
            YYERROR;
          }
          delete $3; _reading_value=false;
        }
;

type:  TYPE { _expected_type = $1; }
;

merge_struct: MERGE_STRUCT { _expected_type = StructElem::getClassTypeId(); }

open_bracket: '{' 
  {
    Type expected_type = _working->getTypeId();
    if (!expected_type.isDerivedFrom(StructElem::getClassTypeId())) {
      if (!expected_type.isDerivedFrom(UnknownElem::getClassTypeId())) {
        yyerror("Structure needs structure type");
        YYERROR;
      }
      delete _working;
      _working = new StructElem;
    }
  
    _expected_type = Type(); 
    _context_stack.prepend(_context); 
    _context = (StructElem*) _working;
    $$ = _working;
  }
;

close_bracket: '}' {  $$ = _context;  _context = _context_stack.pop();  }
;

value_list: value_terminator 
         {
           _reading_value = false;
           if (_working->getExpectedSize() != -1 &&
               _working->getExpectedSize() != _working->numValues()) {
             yyerror("Number of elements does not match expected size");
             YYERROR;
           }
         }
      | value value_list 
;

value:  STRING { if (!read_value($1)) YYERROR; }
      | SYMBOL { if (!read_symbol($1)) YYERROR; }
      | '{' { _reading_value = false; if (!prepare_subcontext()) YYERROR;
              $<elem>1 = _working;
            } 
        body '}' 
          { _reading_value = true; cleanup_subcontext(); 
            _working = $<elem>1; }
;

equals:  '=' { _reading_value = true; _expected_type = Type(); }
;
      
value_terminator: ';' 
      | '}' {  _input.putBack('}'); }
      | FIRST_EOF
;

name:   STRING { _working = make_elem($1->getString(), -1); 
                 if (!_working) YYERROR;
                 $$.elem = _working; $$.name = String::copy($1->getString());
                 $$.context = _context;
                 delete $1; }
      | STRING '[' NUMBER ']' 
         { 
           _working = make_elem($1->getString(), $3); 
           if (!_working) YYERROR;
           $$.elem = _working; $$.name = String::copy($1->getString());
           $$.context = _context;
           delete $1; 
         }
      | STRING '.' 
         { $<structure>2 = _context; _context = get_struct($1->getString(),-1); 
           if (!_context) YYERROR; }
        name 
         { _context = $<structure>2; delete $1; $$ = $4; 
           if (!_working) YYERROR; }
      | STRING '[' NUMBER ']' '.' 
         { $<structure>2 = _context; _context=get_struct($1->getString(),$3); 
           if (!_context) YYERROR; }
        name 
         { _context = $<structure>2; delete $1; $$ = $7; 
           if (!_working) YYERROR; }
;

merge_name: STRING { _working = _context->lookup($1->getString());
                     if (!_working) {
                       _working = make_elem($1->getString(), -1);
                       if (!_working) YYERROR;
                       $$.context = _context;
                     } else {
                       $$.context = NULL;
                     }
                     $$.elem = _working; 
                     $$.name = String::copy($1->getString());
                     delete $1; }

%%

//#define DEBUG_LEX 1

int ConfigParser::eof()
{
  if (!_first_eof) {
    _first_eof = true;
#ifdef DEBUG_LEX
    printf("FIRST_EOF\n");
#endif      
    return FIRST_EOF;
  }
  return 0;
}

int ConfigParser::yylex()
{
  static char valid_when_reading_value[] = "_%/.-[]:+";
  static char valid_when_not_reading_value[] = "_-%+";
  static char tokens_when_reading_value[] = "{};$";
  static char tokens_when_not_reading_value[] = "{};=[].:";
    
  char c;
  // backslash means interpret next character literally, note
  // mainly used for $ and the \ character itself
  bool literal = false;
  for (int i=0;i<2;i++) {
    if (!_input.read(c)) {
      return eof();
    }
    if (c == '\\') 
      literal = true;
    else
      break;
  }

  // try to decode symbols here, they start with '$'
  if (!literal) {
    if (c == '$') {
      // read in symbol name, can include underbars
      if (!_input.readWord(_input_word, "_.[]")) {
        return eof();
      }

      ConfigElem* source=_root->makeElem(_input_word.getString(), true, false);
      if (!source) {
        source = _context->makeElem(_input_word.getString(), true, false);
        if (!source) {
          char err_buf[200];
          snprintf(err_buf, 200, "Cannot find '%s' in local or global context",
                   _input_word.getString());
          yyerror(err_buf);
          return 0;
        }
      }
      yylval.elem = source;
      return SYMBOL;
    }
  }
    
  const char* tokens = 
    (_reading_value ? tokens_when_reading_value:tokens_when_not_reading_value);

  if (strchr(tokens, c)) {
#ifdef DEBUG_LEX
    printf("CHAR '%c'\n", c);
#endif      
    return c;
  }
  _input.putBack(c);

  const char* valid =
    (_reading_value ? valid_when_reading_value : valid_when_not_reading_value);
  if (!_input.readWord(_input_word, valid)) {
    return eof();
  }
  if (!_reading_value) {
    if (_input_word.getString()[0] == '%') {
      if (_input_word != "%include")
        return 0;
#ifdef DEBUG_LEX
      printf("INCLUDE\n");
#endif      
      return INCLUDE;
    }
    if (isdigit(_input_word.getString()[0])) {
      _input.putBack(_input_word.getString());
      int val;
      if (!_input.read(val)) {
        return 0;
      }
      yylval.num = val;
#ifdef DEBUG_LEX
      printf("NUMBER %d\n", val);
#endif      
      return NUMBER;
    }

    Type t = Type::fromName(_input_word.getString());
    if (t.isDerivedFrom(ConfigElem::getClassTypeId())) {
      yylval.type = t.getKey();
#ifdef DEBUG_LEX
      printf("TYPE '%s'\n", t.getName().getString());
#endif      
      return TYPE;
    }
  }

  if (_input_word == "mergeStruct")
    return MERGE_STRUCT;
#ifdef DEBUG_LEX
  printf("STRING '%s'\n", _input_word.getString());
#endif
  yylval.str = new String(_input_word);
  return STRING;
}

void ConfigParser::yyerror(char* s)
{
  if (_input.fromBuffer()) {
    void* buffer; int bufSize;
    _input.getBuffer(buffer, bufSize);
    printf("Line %d, error '%s', buffer '%s'\n", _input.getLineNum(), s,
           (char*) buffer);
  } else
    printf("File %s, line %d, error '%s'\n", _input.getCurFileName(),
           _input.getLineNum(), s);
  errorPrinted = true;
}

//#define DEBUG_PARSE 1

ConfigElem* ConfigParser::make_elem(const char* name, int expected_size)
{
#ifdef DEBUG_PARSE
  printf("Look up %s[%d]\n", name, expected_size);
#endif
  char err_buf[200];
  ConfigElem* elem = _context->lookup(name);
  if (!elem) {
    if (_expected_type.isBad()) {
      elem = new UnknownElem(_context, name);
    } else {
      elem =  (ConfigElem*) _expected_type.createInstance();
    }
    if (!elem) {
      snprintf(err_buf, 200, "Problem creating type for %s, %s", name,
               _expected_type.getName().getString());
      yyerror(err_buf);
      return NULL;
    }
  } else {
    if (!_expected_type.isDerivedFrom(elem->getTypeId())) {
      if (elem->getTypeId() == UnknownElem::getClassTypeId()) {
        UnknownElem* unknown = (UnknownElem*) elem;
        if (!_expected_type.isBad()) {
          elem = (ConfigElem*) _expected_type.createInstance();
          if (!elem) {
            snprintf(err_buf, 200, "Cannot create element of type '%s'",
                     _expected_type.getName().getString());
            yyerror(err_buf);
            return NULL;
          }
        }
        unknown->ref();
        elem->setWatcher(unknown->watcher());
        unknown->getContext()->set(unknown->getName(), elem);
        unknown->unref();
      } else {
        snprintf(err_buf, 200,
                 "Cannot override '%s', formerly of type '%s', with type '%s'",
                name, elem->getTypeId().getName().getString(), 
                _expected_type.getName().getString());
        yyerror(err_buf);
        return NULL;
      }
    }
    if (elem->getExpectedSize() != expected_size &&
        elem->getExpectedSize() != -1) {
      snprintf(err_buf, 200, "Cannot resize '%s' from %d to %d",
              name, elem->getExpectedSize(), expected_size);
      yyerror(err_buf);
      return NULL;
    }
    elem->clear();
  }

  if (expected_size >= 0) {
    if (!elem->setExpectedSize(expected_size)) {
      snprintf(err_buf, 200, "Could not set expected size of '%s' to %d", name,
              expected_size);
      yyerror(err_buf);
      return NULL;
    }
  }
  return elem;
}

ConfigElem* ConfigParser::make_merge_elem(const char* name)
{
  ConfigElem* elem = _context->lookup(name);
  if (!elem)
    return make_elem(name, -1);

  char err_buf[200];
  if (!elem->getTypeId().isDerivedFrom(StructElem::getClassTypeId())) {
    snprintf(err_buf, 200, "Cannot merge a structure with non-structure '%s'",
             name);
    yyerror(err_buf);
    return NULL;
  }

  return elem;
}

StructElem* ConfigParser::get_struct(const char* name, int index)
{
  char err_buf[200];
  ConfigElem* elem = _context->lookup(name);
  if (!elem) {
    snprintf(err_buf, 200, "Can not find structure '%s'", name);
    yyerror(err_buf);
    return NULL;
  }
  if (index >= 0) {
    if (!elem->getTypeId().isDerivedFrom(StructArrayElem::getClassTypeId())) {
      snprintf(err_buf, 200, "'%s' must be a structure array, not a '%s'\n\tP.S. you probably have an '=' between your variable name and the '{'\n",
              name, elem->getTypeId().getName().getString());
      yyerror(err_buf);
      return NULL;
    }
    elem = ((StructArrayElem*) elem)->get(index);
    if (!elem) {
      yyerror("Invalid structure array index");
      return NULL;
    }
  } else {
    if (!elem->getTypeId().isDerivedFrom(StructElem::getClassTypeId())) {
      snprintf(err_buf, 200, "'%s' must be a structure, not a '%s'",
              name, elem->getTypeId().getName().getString());
      yyerror(err_buf);
      return NULL;
    }
  }
  return (StructElem*) elem;
}

bool ConfigParser::read_value(String* val)
{
  bool res = _working->parseVal(val->getString());
  delete val;
  return res;
}

bool ConfigParser::read_symbol(ConfigElem* source)
{
  // If the first element,  or a structure,
  if (_working->getTypeId().isDerivedFrom(StructElem::getClassTypeId()) ||
      _working->numValues() == 0) {
   // just set from the source without printing/parsing
    bool dest_is_unknown = 
      _working->getTypeId().isDerivedFrom(UnknownElem::getClassTypeId());
    bool source_is_unknown =
      source->getTypeId().isDerivedFrom(UnknownElem::getClassTypeId());

    bool res;
    if (dest_is_unknown && !source_is_unknown) {
      ConfigElem* elem = source->clone();
      if (elem) {
        _working->ref();
        _working->unref();
        _working = elem;
        res = true;
      } else {
        char err_buf[200];
        snprintf(err_buf, 200, "Cannot make a copy of a '%s'",
                 source->getTypeId().getName().getString());
        yyerror(err_buf);
        res = false;
      }
    } if (!dest_is_unknown && source_is_unknown) {
      UnknownElem* unknown = (UnknownElem*) source;
      _working->clear();
      res = true;
      for (int i=0;i<unknown->numValues();i++) {
        if (!_working->parseVal(unknown->get(i))) {
          char err_buf[200];
          snprintf(err_buf, 200, "Error parsing '%s'", unknown->get(i));
          yyerror(err_buf);
          res = false;
          break;
        }
      }
    } else {
      if (!_working->copy(source)) {
        char err_buf[200];
        snprintf(err_buf, 200, "Cannot set %s with %s",
                 _working->getTypeId().getName().getString(),
                 source->getTypeId().getName().getString());
        yyerror(err_buf);
        res = false;
      } else
        res = true;
    }
    _working->invokeSetWatch();
    return res;
  }

  // print out source value, parse into destdestination element
  utils::Output output;
  output.setBuffer(_tmp_buf, _size_tmp_buf, utils::Output::standardResize);
  source->writeData(output);
  output.write('\0');
  void* bufptr;
  int size;
  output.getBuffer(bufptr, size);
  _tmp_buf = (char*) bufptr;
  _size_tmp_buf = output.getBufferSize();
  return _working->parseVal(_tmp_buf);
}

bool ConfigParser::prepare_subcontext()
{
  if (!_working->getTypeId().isDerivedFrom(StructArrayElem::getClassTypeId())){
    if (!_working->getTypeId().isDerivedFrom(UnknownElem::getClassTypeId())) {
      yyerror("Cannot override non-structure context with a structure array\n\tP.S. you probably have an '=' between your variable name and the '{'\n");
      return false;
    }
    StructArrayElem* new_working = new StructArrayElem;
    UnknownElem* unknown = (UnknownElem*) _working;
    unknown->ref();
    new_working->setWatcher(unknown->watcher());
    unknown->getContext()->set(unknown->getName(), new_working);
    unknown->unref();
    _working = new_working;
  }
    
  _context_stack.prepend(_context);
  _context = new StructElem();
  ((StructArrayElem*) _working)->add(_context);

  return true;
}

void ConfigParser::cleanup_subcontext()
{
  _context = _context_stack.pop();
}

bool ConfigParser::make_tag(StructElem* structure, String* tag_string)
{
#ifdef DEBUG_PARSE
  printf("Making tag %s\n", tag_string->getString());
#endif  
  ConfigElem* tag = new StringElem();
  structure->set("tag", tag);
  if (!tag->parseVal(tag_string->getString())) {
    yyerror("Error reading tag");
    return false;
  }
  delete tag_string;
  return true;
}

void ConfigParser::add_working(WorkingStruct& working, ConfigElem* elem)
{
  working.context->set(working.name, elem);
  delete [] working.name;
}

ConfigElem* ConfigParser::input_file(const char* name)
{
  static int include_level = 0;

  char err_buf[200];
  Input input;
  include_level++;
  if (!input.openFile(name)) {
    snprintf(err_buf, 200, "Cannot open '%s'", name);
    yyerror(err_buf);
    return NULL;
  }
  
  if (ConfigFile::verbosity() > 0) {
    for (int i=0;i<include_level;i++)
      printf("\t");
    printf("Including %s\n", input.getCurFileName());
  }

  ConfigParser include_parser(input, _context);
  if (include_parser.yyparse() == 0) {
    include_level--;
    return include_parser._working;
  } else {
    if (!include_parser.errorPrinted) {
      printf("File %s, line %d, syntax error\n", input.getCurFileName(),
             input.getLineNum());
    }
    return NULL;
  }
}



%header{
__UTILS_END_NAMESPACE
%}
