#include <utils/CannedDataAccess.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/time.h>

__UTILS_BEGIN_NAMESPACE

enum CannedDataStateEnum {
	CDT_STATE_CLOSED = 0,
	CDT_STATE_OPENED
};

/* Bits for the _headers_written field */
#define _HW_TYPE           0x00000001
#define _HW_VERSION        0x00000002
#define _HW_DESCRIPTION    0x00000004
#define _HW_HEADER         0x00000008

CannedDataWriter::CannedDataWriter()
{
	_throttle_size = 0;
	_throttle_window = 0;
	_init();
}

CannedDataWriter::~CannedDataWriter()
{
  close();
}

void CannedDataWriter::_doFwrite(const void *buf, size_t size, FILE *file)
{
        if (size) {
                if (!fwrite(buf, size, 1, file)) {
                        _cleanup();
                        throw CannedDataError("fwrite failed: %s", strerror(errno));
                }
        }
}

void CannedDataWriter::_writeIdxRecord(CannedDataFieldID id, const void *buf, uint16_t size)
{
	uint16_t tmp;
	uint16_t tmp_size;
	tmp = host_to_le_uint16((uint16_t)id);
	tmp_size = host_to_le_uint16(size);
	_doFwrite(&tmp, 2, _idx_file);
	_doFwrite(&tmp_size, 2, _idx_file);
	if (size) {
		_doFwrite(buf, size, _idx_file);
	}
}

void CannedDataWriter::_init()
{
	_state = CDT_STATE_CLOSED;
	_idx_file = 0;
	_data_file = 0;
	_data_file_num = 0;
	_mode = 0;
	_headers_written = 0;
	_base_name = "";
	_throttle_data_in_flight = 0;
        _last_secs_stamp = 0;
        _last_usecs_stamp = 0;
	_throttle_queue.clear();
}

void CannedDataWriter::_cleanup()
{
	if (_data_file) {
		fclose(_data_file);
	}
	if (_idx_file) {
		fclose(_idx_file);
	}
	_init();
        while (CDWThrottleQueueElt* elt = _throttle_queue.pop())
          delete elt;
}

void CannedDataWriter::_newDataFile()
{
	char tmp[16];
	if (_data_file) {
		fclose(_data_file);
	}
	_data_file_num++;
	snprintf(tmp, 15, "%i", _data_file_num - 1);
	tmp[15] = 0;
	String str = _base_name;
        str += ".data.";
        str += tmp;
	_data_file = fopen(str.getString(), "w");
	if (!_data_file) {
		_cleanup();
		throw CannedDataError("Couldn't open %s for output: %s\n", 
				      str.getString(), strerror(errno));
	}
	_doFwrite(&_set_id, 4, _data_file);
	_data_file_size = 4;
}

void CannedDataWriter::_checkClobber(const String &filename)
{
	struct stat buf;
	if ((_mode & O_EXCL)
	    && !stat(filename.getString(), &buf)) {
		_cleanup();
		CannedDataError err("Can't clobber file %s", filename.getString());
		throw err;
	}
}

/*
 * Open a file for writing.  Create the
 * file if it doesn't exist.  If it does exist, 
 * clobber it iff mode doesn't include O_EXCL.  Set 
 * file type and version data accordingly Ignore
 * the other mode bits for now. 
 */
void CannedDataWriter::open(const char *filename, mode_t mode)
{
	char buf[4] = { 'D', 'W', 'H', 'O' };
	struct timeval tv;
	uint32_t cd_major_version = host_to_le_uint32(CANNED_DATA_MAJOR_VERSION);
	uint32_t cd_minor_version = host_to_le_uint32(CANNED_DATA_MINOR_VERSION);
	uint64_t tmp64;
	String idx_filename, data_filename;
	if (_state != CDT_STATE_CLOSED) {
		_cleanup();
		throw CannedDataError("Open called when already open");
	}
	_set_id = random();
	_base_name = filename;
	_mode = mode;
	idx_filename = _base_name;
	_checkClobber(idx_filename);
	_idx_file = fopen(idx_filename.getString(), "w");
	if (!_idx_file) {
		_cleanup();
		throw CannedDataError("Couldn't open %s for output: %s\n", 
				      idx_filename.getString(), strerror(errno));
	}
	if (gettimeofday(&tv, 0)) {
		_cleanup();
		throw CannedDataError("gettimeofday failed: %s\n", strerror(errno));
	}
	_newDataFile();
	_doFwrite(buf, 4, _idx_file);
	_doFwrite(&cd_major_version, 4, _idx_file);
	_doFwrite(&cd_minor_version, 4, _idx_file);
	_writeIdxRecord(CDID_ID, &_set_id, 4);
	tmp64 = tv.tv_sec;
	tmp64 = host_to_le_uint64(tmp64);
	_writeIdxRecord(CDID_CREATION_TIME, &tmp64, 8);	
	_state = CDT_STATE_OPENED;
}


/* 
 * Write all metadata to disk, and finish out the file, closing it.
 * No further write* calls will be valid after close() is invoked
 */
void CannedDataWriter::close()
{
  if (_state == CDT_STATE_OPENED) {
    _writeIdxRecord(CDID_END, 0, 0);
    _cleanup();
  }
}



/*
 * Set the file type. 
 */
void CannedDataWriter::writeType(CannedDataType type)
{
	uint32_t tmp;
	if (_headers_written & _HW_TYPE) {
		_cleanup();
		throw CannedDataError("Tried to write type field twice");
	}
	_headers_written |= _HW_TYPE;
	tmp = host_to_le_uint32((uint32_t)type);
	_writeIdxRecord(CDID_TYPE, &tmp, 4);
}

/* 
 * Set the version information for this file.  
 */

void CannedDataWriter::writeVersion(uint32_t major, uint32_t minor)
{
	CannedDataVersionRecord tmp;
	if (_headers_written & _HW_VERSION) {
		_cleanup();
		throw CannedDataError("Tried to write version field twice");
	}
	_headers_written |= _HW_VERSION;
	tmp.major = host_to_le_uint32(major);
	tmp.minor = host_to_le_uint32(minor);
	_writeIdxRecord(CDID_VERSION, &tmp, 8);
}

/*
 * Add a description of this data.  The string should be null
 * terminated.
 */
void CannedDataWriter::writeDescription(const char *desc)
{
	if (_headers_written & _HW_DESCRIPTION) {
		_cleanup();
		throw CannedDataError("Tried to write description field twice");
	}
	_headers_written |= _HW_DESCRIPTION;
	_writeIdxRecord(CDID_DESCRIPTION, desc, strlen(desc));
}


/*
 * Write some custom header information for this file.
 */
void CannedDataWriter::writeHeader(void *buf, size_t size)
{
	if (size > 0xffff) {
		throw CannedDataError("Header too large!\n");
	}
	if (_headers_written & _HW_HEADER) {
		_cleanup();
		throw CannedDataError("Tried to write header field twice");
	}
	_headers_written |= _HW_HEADER;
	_writeIdxRecord(CDID_HEADER, buf, size);
}

void CannedDataWriter::_writeDataRecord(void *buf, size_t size)
{
	_doFwrite(buf, size, _data_file);
	fflush(_data_file);
	_data_file_size += size;
	/* start the next file with a little headroom under 2GB */
	if (_data_file_size > 2000000000) {
		_newDataFile();
	}
}


int CannedDataWriter::_checkThrottle(size_t size)
{
	CDWThrottleQueueElt* elt;
	struct timeval now;
	int done = 0;
	long secs_elapsed;

	/* If there's no window, we're not throttling at all */
	if (!_throttle_window) {
		return 0;
	}

	if (gettimeofday(&now, 0)) {
		throw CannedDataError("gettimeofday failed: %s\n", strerror(errno));
	}
	while (!(_throttle_queue.empty() || done)) {
		elt = _throttle_queue.first();
		secs_elapsed = now.tv_sec - elt->tv.tv_sec;
		if (elt->tv.tv_usec > now.tv_usec) {
			secs_elapsed--;
		}
		if (secs_elapsed > _throttle_window) {
			_throttle_data_in_flight -= elt->size;
                        _throttle_queue.pop();
                        delete elt;
		} else {
			done = 1;
		}
	}
	
	/* Sanity check...if the queue is empty, we shouldn't have any
	   data in flight 
	*/
	assert(!(_throttle_queue.empty() && _throttle_data_in_flight));
	if ((_throttle_data_in_flight + size) < _throttle_size) {
		_throttle_data_in_flight += size;
                elt = new CDWThrottleQueueElt;
		elt->tv = now;
		elt->size = size;
		_throttle_queue.append(elt);
		printf("+");
		fflush(stdout);
		return 0;
	}
	printf("-");
	fflush(stdout);
	return 1;
}

/*
 * Write out a data record to the file with the given time stamp 
 */
void CannedDataWriter::writeData(void *buf, size_t size, 
				 uint64_t secs, uint64_t usecs)
{
	uint16_t tmp16;
	uint32_t tmp32;
	uint64_t tmp64;
	uint8_t tmp8;
	
        if (_last_secs_stamp > secs
            || ((_last_secs_stamp == secs)
                && (_last_usecs_stamp >= usecs))) {
                /*
                 * This record is out of order.  Probably got an NTP update or somesuch.  Complain, and
                 * skip it.
                 */
                return;
        }
        _last_secs_stamp = secs;
        _last_usecs_stamp = usecs;

	if (_checkThrottle(size)) {
		return;
	}
	
	tmp16 = host_to_le_uint16((uint16_t)CDID_IDX);
	_doFwrite(&tmp16, 2, _idx_file);
	tmp16 = host_to_le_uint16((uint16_t)25);
	_doFwrite(&tmp16, 2, _idx_file);
	tmp64 = host_to_le_uint64(secs);
	_doFwrite(&tmp64, 8, _idx_file);
	tmp64 = host_to_le_uint64(usecs);
	_doFwrite(&tmp64, 8, _idx_file);
	tmp32 = host_to_le_uint32(_data_file_size);
	_doFwrite(&tmp32, 4, _idx_file);
	tmp32 = host_to_le_uint32(size);
	_doFwrite(&tmp32, 4, _idx_file);
	tmp8 = _data_file_num;
	_doFwrite(&tmp8, 1, _idx_file);
	_writeDataRecord(buf, size);
	fflush(_idx_file);
}

void CannedDataWriter::setThrottle(size_t size, long window)
{
	_throttle_size = size;
	_throttle_window = window;
}

__UTILS_END_NAMESPACE
