/***************************************************************************

                                BinPlayer.cc

  Implements a class can be used to read values from a binary file 
  with tags and times.  It attempts to return the variables at the
  correct time.  

  Classes implemented for export:
    BinPlayer - the playback class

***************************************************************************/

#include <math.h>
#include <ctype.h>

#include <utils/BinPlayer.h>
#include <utils/Vector.h>
#include <utils/List.h>
#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>
#include <utils/Generator.h>
#include <utils/formatting/FormConfig.h>

__UTILS_BEGIN_NAMESPACE

class BinPlayerElem;
class BinPlayerGenerator : public Generator<BinPlayerElem>
{
public:
  BinPlayerGenerator(bool old_version) { _old_version = old_version; }

  virtual BinPlayerElem* interface(const char* spec_string,
                                   SymbolTable* globals) {
    _tmp = spec_string;
    if (_old_version) {
      char* c_ptr = (char*) _tmp.getString();
      while (*c_ptr && !isalpha(*c_ptr)) 
        c_ptr++;
      if (!c_ptr)
        return NULL;
      *c_ptr = toupper(*c_ptr);
    }
    return Generator<BinPlayerElem>::interface(_tmp.getString(), globals);
  }    
  virtual BinPlayerElem* interface(ConfigFile* config_spec,
				   SymbolTable* globals)
    { return (BinPlayerElem*) GeneratorBase::interface(config_spec, globals); }

private:
  bool _old_version;
  String _tmp;
};
static BinPlayerGenerator* PlayElemGenerator = NULL;


static bool read_word(utils::Input& input, unsigned char* dest, int byte_order)
{
  if (byte_order == BYTE_ORDER)
    return input.read(dest, sizeof(int));
  else {
    int tmp;
    unsigned char* src = (unsigned char*) &tmp;
    bool res = input.read(src, sizeof(int));
    if (!res)
      return false;
    dest[0] = src[3];
    dest[1] = src[2];
    dest[2] = src[1];
    dest[3] = src[0];
    return true;
  }
}

static bool read_double(utils::Input& input, unsigned char* dest,
                       int byte_order)
{
  if (byte_order == BYTE_ORDER)
    return input.read(dest, sizeof(double));
  else {
    return read_word(input, dest, byte_order) &&
      read_word(input, dest+sizeof(int), byte_order);
  }
}  

static bool read_short(utils::Input& input, unsigned char* dest,
                       int byte_order)
{
  if (byte_order == BYTE_ORDER)
    return input.read(dest, sizeof(short));
  else {
    int tmp;
    unsigned char* src = (unsigned char*) &tmp;
    bool res = input.read(src, sizeof(short));
    if (!res)
      return false;
    dest[0] = src[1];
    dest[1] = src[0];
    return true;
  }
}

static bool read_char(utils::Input& input, unsigned char* dest)
{
  return input.read(dest, sizeof(char));
}

class BinPlayerElem {
public:
  BinPlayerElem();
  virtual ~BinPlayerElem() {};

  const char* getFormat() const { return _format.getString(); }
  void setFormat(const char* format) { _format = format; }
  const char* getName() const { return _name.getString(); }
  void setName(const char* name) { _name = name; }
  void* getAddress() const { return _address; }
  void setAddress(void* address) { _address = address; }

  virtual bool read(Input& input, void* address, int, int) = 0;
  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) = 0;
  virtual int size() const = 0;
  virtual bool isInterpolating() const = 0;
  virtual void setInterpolating(bool) {}
  virtual void alloc(void* address) {}
  virtual void dealloc(void* address) {}

  static BinPlayerGenerator* generator(bool);

protected:
  String _format;
  String _name;
  void* _address;
};

class FloatBinPlayerElem : public BinPlayerElem {
public:
  FloatBinPlayerElem(bool use_min, float min, bool use_max, float max,
                     bool periodical, bool interpolate)
    : BinPlayerElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period = 0;
    _interpolate = interpolate;
  }

  virtual bool read(Input& input, void* address, int byte_order, int) {
    float val;
    
    // put byte swapping stuff here, if necessary
    if (read_word(input, (unsigned char*) &val, byte_order)) {
      if (_period) {
        while (val < _min) 
          val += _period;
        while (val > _max)
          val -= _period;
      } else {
        if (_use_min && val < _min)
          val = _min;
        if (_use_max && val > _max)
          val = _max;
      }

      if (address)
        memcpy((char*) address, (char*) &val, sizeof(float));
      return true;
    } else
      return false;
  }
      
  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (!address)
      return true;

    if (!_interpolate || dist == 1.0 || !prev_ptr) {
      memcpy(address, cur_ptr, sizeof(float));
      return true;
    }
    if (dist == 0.0) {
      memcpy(address, cur_ptr, sizeof(float));
      return true;
    }      

    float cur, prev;
    memcpy(&cur, cur_ptr, sizeof(float));
    memcpy(&prev, prev_ptr, sizeof(float));
    float delta = cur-prev;
    if (!_period) {
      *(float*) address = prev + delta*dist;
    } else {
      // assume we go the shortest way around
      if (delta > _period/2.0) { 
        cur -= _period;
        delta -= _period;
      }
      else if (delta < -_period/2.0) {
        cur += _period;
        delta += _period;
      }

      float res = prev + delta*dist;

      // make sure still in range
      while (res < _min)
        res += _period;
      while (res > _max)
        res -= _period;
      *(float*) address = res;
    }

    return true;
  }

  virtual int size() const { return sizeof(float); }
  virtual bool isInterpolating() const { return _interpolate; }
  virtual void setInterpolating(bool i) { _interpolate = i; }

private:
  bool _use_min;
  float _min;
  bool _use_max;
  float _max;
  float _period;
  bool _interpolate;
};

class DoubleBinPlayerElem : public BinPlayerElem {
public:
  DoubleBinPlayerElem(bool use_min, double min, bool use_max, double max,
                     bool periodical, bool interpolate)
    : BinPlayerElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period = 0;
    _interpolate = interpolate;
  }

  virtual bool read(Input& input, void* address, int byte_order, int) {
    double val;
    
    // put byte swapping stuff here, if necessary
    if (read_double(input, (unsigned char*) &val, byte_order)) {
      if (_period) {
        while (val < _min) 
          val += _period;
        while (val > _max)
          val -= _period;
      } else {
        if (_use_min && val < _min)
          val = _min;
        if (_use_max && val > _max)
          val = _max;
      }

      if (address)
        memcpy((char*) address, (char*) &val, sizeof(double));
      return true;
    } else
      return false;
  }
      
  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (!address)
      return true;

    if (!_interpolate || dist == 1.0 || !prev_ptr) {
      memcpy(address, cur_ptr, sizeof(double));
      return true;
    }
    if (dist == 0.0) {
      memcpy(address, cur_ptr, sizeof(double));
      return true;
    }      

    double cur, prev;
    memcpy(&cur, cur_ptr, sizeof(double));
    memcpy(&prev, prev_ptr, sizeof(double));
    double delta = cur-prev;
    if (!_period) {
      *(double*) address = prev + delta*dist;
    } else {
      // assume we go the shortest way around
      if (delta > _period/2.0) { 
        cur -= _period;
        delta -= _period;
      }
      else if (delta < -_period/2.0) {
        cur += _period;
        delta += _period;
      }

      double res = prev + delta*dist;

      // make sure still in range
      while (res < _min)
        res += _period;
      while (res > _max)
        res -= _period;
      *(double*) address = res;
    }

    return true;
  }

  virtual int size() const { return sizeof(double); }
  virtual bool isInterpolating() const { return _interpolate; }
  virtual void setInterpolating(bool i) { _interpolate = i; }

private:
  bool _use_min;
  double _min;
  bool _use_max;
  double _max;
  double _period;
  bool _interpolate;
};

class ShortBinPlayerElem : public BinPlayerElem {
public:
  ShortBinPlayerElem(bool use_min, short min, bool use_max, short max,
                     bool periodical)
    : BinPlayerElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period = 0;
  }

  virtual bool read(Input& input, void* address, int byte_order, int) {
    short val;
    
    // put byte swapping stuff here, if necessary
    if (read_short(input, (unsigned char*) &val, byte_order)) {
      if (_period) {
        while (val < _min) 
          val += _period;
        while (val > _max)
          val -= _period;
      } else {
        if (_use_min && val < _min)
          val = _min;
        if (_use_max && val > _max)
          val = _max;
      }

      if (address)
        memcpy((char*) address, (const char*) &val, sizeof(short));
      return true;
    } else
      return false;
  }
      
  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (address)
      memcpy(address, cur_ptr, sizeof(short));
    return true;
  }

  virtual int size() const { return sizeof(short); }
  virtual bool isInterpolating() const { return false; }

private:
  bool _use_min;
  short _min;
  bool _use_max;
  short _max;
  short _period;
};

class ByteBinPlayerElem : public BinPlayerElem {
public:
  ByteBinPlayerElem(bool use_min, char min, bool use_max, char max,
                    bool periodical)
    : BinPlayerElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      if (_min > _max)
        _period = 0;
      else
        _period = _max - _min;
    } else
      _period = 0;
  }

  virtual bool read(Input& input, void* address, int byte_order, int) {
    char val;
    
    // put byte swapping stuff here, if necessary
    if (read_char(input, (unsigned char*) &val)) {
      if (_period) {
        while (val < _min) 
          val += _period;
        while (val > _max)
          val -= _period;
      } else {
        if (_use_min && val < _min)
          val = _min;
        if (_use_max && val > _max)
          val = _max;
      }

      if (address)
        memcpy((char*) address, (const char*) &val, sizeof(char));
      return true;
    } else
      return false;
  }
      
  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (address)
      *(char*) address = *(char*) cur_ptr;
    return true;
  }

  virtual int size() const { return sizeof(char); }
  virtual bool isInterpolating() const { return false; }

private:
  bool _use_min;
  char _min;
  bool _use_max;
  char _max;
  char _period;
};

class ArrayBinPlayerElem : public BinPlayerElem {
public:
  ArrayBinPlayerElem(int size, const char* elem_format, bool interpolating) 
    : BinPlayerElem()
  {
    _elem = PlayElemGenerator->interface(elem_format, NULL);
    _num_elems = size;
  }
  virtual ~ArrayBinPlayerElem() { if (_elem) delete _elem; }

  virtual int size() const { return _elem->size()*_num_elems; }
  virtual bool isInterpolating() const { return _elem->isInterpolating(); }

  virtual bool read(Input& input, void* address, int byte_order, int align) {
    if (!_elem)
      return false;
    bool res = true;
    for (int i=0;i<_num_elems;i++) {
      if (address) {
        if (!_elem->read(input, ((unsigned char*) address)+i*_elem->size(),
                         byte_order, align))
        res = false;
      } else {
        if (!_elem->read(input, NULL, byte_order, align))
          res = false;
      }
    }
    return res;
  }

  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (!address)
      return true;
    if (!_elem)
      return false;
    bool res = true;
    unsigned char* cp = (unsigned char*) cur_ptr;
    unsigned char* pp = (unsigned char*) prev_ptr;
    for (int i=0;i<_num_elems;i++) {
      if (!_elem->get(time, ((unsigned char*) address)+i*_elem->size(),
                      cp, pp, dist))
        res = false;
      cp += _elem->size();
      pp += _elem->size();
    }
    return res;
  }

private:
  BinPlayerElem* _elem;
  int _num_elems;
};

class VectorBinPlayerElem : public BinPlayerElem {
public:
  VectorBinPlayerElem(const char* elem_format, bool interpolating) 
    : BinPlayerElem()
  {
    _elem = PlayElemGenerator->interface(elem_format, NULL);
    _elem->setInterpolating(interpolating);
  }
  virtual ~VectorBinPlayerElem() { if (_elem) delete _elem; }

  virtual int size() const { return sizeof(PlayVectorSpec); }
  virtual bool isInterpolating() const { return _elem->isInterpolating(); }

  virtual void alloc(void* address) {
    PlayVectorSpec* spec = (PlayVectorSpec*) getAddress();
    PlayVectorSpec* targ = (PlayVectorSpec*) address;
    if (!spec || !targ)
      return;
    targ->max_size = spec->max_size;
    targ->data = new unsigned char[spec->max_size*_elem->size()];
    targ->size = 0;
  }

  virtual void dealloc(void* address) {
    PlayVectorSpec* targ = (PlayVectorSpec*) address;
    unsigned char* doomed = (unsigned char*) targ->data;
    delete [] doomed;
  }

  virtual bool read(Input& input, void* address, int byte_order, int align) {
    PlayVectorSpec* spec = (PlayVectorSpec*) address;
    if (!_elem)
      return false;
    bool res = true;
    if (read_word(input, (unsigned char*) &spec->size, byte_order)) {
      if (spec->size < 0 || spec->size > 100000) {
        printf("Bad spec size %d\n", spec->size);
        return false;
      }
      if (spec->size <= spec->max_size) {
        for (int i=0;i<spec->size;i++) {
          if (!_elem->read(input,
                           ((unsigned char*) spec->data)+i*_elem->size(),
                           byte_order, align))
            res = false;
        }
      } else {
        int i;
        for (i=0;i<spec->max_size;i++) {
          if (!_elem->read(input,
                           ((unsigned char*) spec->data)+i*_elem->size(),
                           byte_order, align)) {
            res = false;
            break;
          }
        }
        for (i=spec->max_size; i<spec->size;i++) {
          if (!_elem->read(input, NULL, byte_order, align)) {
            res = false;
            break;
          }
        }
        spec->size = spec->max_size;
      }
    } else 
      res = false;
    return res;
  }

  virtual bool get(Time time, void* address, void* cur_ptr, 
                   void* prev_ptr, double dist) {
    if (!address)
      return true;
    if (!_elem)
      return false;
    bool res = true;
    PlayVectorSpec* out_spec = (PlayVectorSpec*) address;
    PlayVectorSpec cur_spec, prev_spec; 
    memcpy(&cur_spec, cur_ptr, sizeof(PlayVectorSpec));
    memcpy(&prev_spec, prev_ptr, sizeof(PlayVectorSpec));
    unsigned char* cp = (unsigned char*) cur_spec.data;
    unsigned char* pp = (unsigned char*) prev_spec.data;
    for (int i=0;i<cur_spec.size;i++) {
      if (i >= prev_spec.size)
        pp = NULL;
      if (!_elem->get(time,
                      ((unsigned char*) out_spec->data)+i*_elem->size(),
                      cp, pp, dist))
        res = false;
      cp += _elem->size();
      pp += _elem->size();
    }
    out_spec->size = cur_spec.size;
    return res;
  }

private:
  BinPlayerElem* _elem;
};

class BinPlayerTag {
public:
  BinPlayerTag();
  ~BinPlayerTag();

  void setAttributes(int bo, int al) { _byte_order = bo; _alignment = al; }

  bool expect(const char* name, const char* format, void* address);
  bool get(Time& play_time);
  bool getPrevious(Time& play_time);
  bool read_line(Time time, Input& input);
  bool declare(const char* name, const char* format);

  Time cur_time() const { return _cur_time; }

private:
  List<BinPlayerElem> _input;
  StringDict<BinPlayerElem*> _elems;
  unsigned char* _cur_data;
  Time _cur_time, _prev_time;
  unsigned char* _prev_data;
  int _size_data;
  int _byte_order, _alignment;
};

struct BinPlayerElemInfo {
    char* tag;
    BinPlayerElem* (*create_interface)(Generator<BinPlayerElem>*, ConfigFile*,
                                       SymbolTable*);
};

static BinPlayerElem* create_float_play_elem(Generator<BinPlayerElem>*,
                                             ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  float min=100, max=-100;
  if (params->getFloats("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getFloats("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new FloatBinPlayerElem(use_min, min, use_max, max, periodical,
                                params->getBool("interpolate", true));
}

static BinPlayerElem* create_short_play_elem(Generator<BinPlayerElem>*,
                                             ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  int min=100, max=-100;
  if (params->getInts("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getInts("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new ShortBinPlayerElem(use_min, min, use_max, max, periodical);
}

static BinPlayerElem* create_byte_play_elem(Generator<BinPlayerElem>*,
                                             ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  int min=100, max=-100;
  if (params->getInts("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getInts("max", &max, 1, NULL, 255))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new ByteBinPlayerElem(use_min, min, use_max, max, periodical);
}

static BinPlayerElem* create_double_play_elem(Generator<BinPlayerElem>*,
                                             ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  double min=100, max=-100;
  if (params->getDoubles("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getDoubles("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new DoubleBinPlayerElem(use_min, min, use_max, max, periodical,
                                 params->getBool("interpolate", true));
}

static BinPlayerElem* create_angle_play_elem(Generator<BinPlayerElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  return new FloatBinPlayerElem(true, -M_PI, true, M_PI, true,
                                params->getBool("interpolate", true));
}

static BinPlayerElem* create_array_play_elem(Generator<BinPlayerElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  int size_array;
  if (!params->getInts("size", &size_array, 1, NULL, 0)) {
    printf("Array element requires a size\n");
    return NULL;
  }
  
  return new ArrayBinPlayerElem(size_array,
                                params->getString("format", "float"),
                                params->getBool("interpolating", false));
}

static BinPlayerElem* create_vector_play_elem(Generator<BinPlayerElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  return new VectorBinPlayerElem(params->getString("format", "float"),
                                 params->getBool("interpolating", false));
}

static struct BinPlayerElemInfo BinPlayerElem_interfaces[] = {
	{"Byte", create_byte_play_elem},
	{"Short", create_short_play_elem},
	{"Float", create_float_play_elem},
	{"Double", create_double_play_elem},
	{"Angle", create_angle_play_elem},
	{"Array", create_array_play_elem},
	{"Vector", create_vector_play_elem},
	{0, 0}
};

// get the generator in case someone wants to generate an instance of BinPlayerElem
// or add to the possible configurations
BinPlayerGenerator* BinPlayerElem::generator(bool old_version)
{
    int i;
    BinPlayerGenerator* gen = new BinPlayerGenerator(old_version);
    for (i=0;BinPlayerElem_interfaces[i].tag; i++) {
        gen->registerInterface(BinPlayerElem_interfaces[i].tag,
                               BinPlayerElem_interfaces[i].create_interface,
                               0L);
    }

    return gen;
}    

BinPlayerElem::BinPlayerElem()
{
  _address = NULL;
}

BinPlayerTag::BinPlayerTag()
{
  _cur_data = _prev_data = NULL;
  _byte_order = BYTE_ORDER;
  _alignment = ALIGN;
}

static void delete_player_elem(const char*, BinPlayerElem* elem)
{
  delete elem;
}

BinPlayerTag::~BinPlayerTag()
{
  if (_cur_data) {
    ListIterator<BinPlayerElem> iter(_input);
    unsigned char* cur = _cur_data;
    unsigned char* prev = _prev_data;
    for (BinPlayerElem* elem = iter.first(); elem; elem=iter.next()) {
      elem->dealloc(cur);
      cur += elem->size();        
      elem->dealloc(prev);
      prev += elem->size();
    }
    delete [] _cur_data;
    delete [] _prev_data;
  }
  _elems.applyToAll(delete_player_elem);
}

bool BinPlayerTag::read_line(Time time, Input& input)
{
  ListIterator<BinPlayerElem> iter(_input);
  bool first_time = false;
  if (!_cur_data) {
    BinPlayerElem* elem;
    _size_data = 0;
    for (elem = iter.first(); elem; elem=iter.next()) 
      _size_data += elem->size();
    if (!_size_data)
      return false;
    _cur_data = new unsigned char[_size_data];
    memset((char*) _cur_data, 0, _size_data);
    _prev_data = new unsigned char[_size_data];
    memset((char*) _prev_data, 0, _size_data);
    first_time = true;
    unsigned char* cur = _cur_data;
    unsigned char* prev = _prev_data;
    for (elem = iter.first(); elem; elem=iter.next()) {
      elem->alloc(cur);
      cur += elem->size();        
      elem->alloc(prev);
      prev += elem->size();
    }
  }

  _prev_time = _cur_time;
  unsigned char* tmp = _prev_data;
  _prev_data = _cur_data;
  _cur_data = tmp;

  _cur_time = time;

  bool res = true;
  unsigned char* ptr = _cur_data;
  for (BinPlayerElem* elem = iter.first(); elem; elem=iter.next()) {
    if (!elem->read(input, ptr, _byte_order, _alignment)) {
      res = false;
      break;
    }
    ptr += elem->size();
  }

  if (first_time) {
    _prev_time = _cur_time;
  }

  return res;
}

bool BinPlayerTag::get(Time& play_time)
{
  bool res = true;
  unsigned char* cur_ptr = _cur_data;
  unsigned char* prev_ptr = _prev_data;
  ListIterator<BinPlayerElem> iter(_input);
  double dist;
  if (_cur_time == _prev_time || play_time == _cur_time) {
    dist = 1;
    play_time = _cur_time;
  } else
    dist =
      (play_time-_prev_time).getValue()/(_cur_time-_prev_time).getValue();
  if (dist < 0) {
    dist = 0;
    play_time = _prev_time;
  }
  if (dist > 1) {
    dist = 1;
    play_time = _cur_time;
  }

  for (BinPlayerElem* elem = iter.first(); elem; elem=iter.next()) {
    if (!elem->get(play_time, elem->getAddress(),cur_ptr, prev_ptr, dist))
      res = false;
    cur_ptr += elem->size();
    prev_ptr += elem->size();
  }

  return res;
}

bool BinPlayerTag::getPrevious(Time& play_time)
{
  play_time = _prev_time;
  unsigned char* prev_ptr = _prev_data;

  bool res = true;
  ListIterator<BinPlayerElem> iter(_input);
  for (BinPlayerElem* elem = iter.first(); elem; elem=iter.next()) {
    if (!elem->get(play_time, elem->getAddress(),prev_ptr, 0, 1.0))
      res = false;
    prev_ptr += elem->size();
  }

  return res;
}

bool BinPlayerTag::expect(const char* name, const char* format,
                          void* address)
{
  BinPlayerElem* elem = PlayElemGenerator->interface(format, NULL);
  if (!elem)
    return false;

  elem->setFormat(format);
  elem->setName(name);
  elem->setAddress(address);
  _elems.enter(name, elem);

  return true;
}

bool BinPlayerTag::declare(const char* name, const char* format)
{
  BinPlayerElem* elem;
  if (!_elems.find(name, elem)) {
    elem = PlayElemGenerator->interface(format, NULL);
    if (!elem)
      return false;
    
    elem->setFormat(format);
    elem->setName(name);
    elem->setAddress(NULL);
    _elems.enter(name, elem);
  } else {
    if (strcasecmp(format, elem->getFormat()) != 0) {
      printf("Expected format %s does not match incoming format %s\n",
             elem->getFormat(), format);
      _elems.remove(name);
      delete elem;
      return declare(name, format);
    }
  }

  _input.append(elem);

  return true;
}

// helper function used in destuctor
static void delete_value(const char*, BinPlayerTag* value)
{
  delete value;
}

BinPlayer::BinPlayer()
{
  if (!PlayElemGenerator)
    PlayElemGenerator = BinPlayerElem::generator(_old_version);

  _close_input = false;
  _read_time = false;
  _zero_time.setValue(0,0);
}

BinPlayer::~BinPlayer()
{
  _values.applyToAll(delete_value);
    close();
}
        
// set the zero time for the player to zero time
// (adjust the offset if we have read a logged time already)
void BinPlayer::setZero(Time zero_time)
{
  _zero_time = zero_time;
  if (_read_time)
    _offset = _zero_time - _first_time;
}

bool BinPlayer::expect(const char* tag, const char* name, const char* format,
                       void* address)
{
  BinPlayerTag* tagged;
  if (!_values.find(tag, tagged)) {
    tagged = new BinPlayerTag();
    _values.enter(tag, tagged);
  }

  return tagged->expect(name, format, address);
}

static void set_attributes(const char*, BinPlayerTag* value, void* data)
{
  BinPlayer* player = (BinPlayer*) data;
  value->setAttributes(player->getByteOrder(), player->getAlignment());
}

bool BinPlayer::open(const char* file_name)
{
  if (!_input.openFile(file_name))
    return false;
  _close_input = true;

  String title;
  float version = -1;
  _byte_order = _alignment = -1;
  if (!_input.read(title) || title!="BinLog" || !_input.read(version)
      || (version != 1.0 && version != 0.0) || !_input.read(_byte_order)
      || !_input.read(_alignment)) {
    printf("Invalid header %s %f %d %d\n", title.getString(), version,
           _byte_order, _alignment);
    return false;
  }
  _old_version = (version == 0.0);

  _values.applyToAll(set_attributes, this);

  return true;
}
  
// if necessary, close the input file
void BinPlayer::close()
{
  if (_close_input)
    _input.closeFile();
}

bool BinPlayer::next(const char* tag, Time& time)
{
  BinPlayerTag* value;
  Time tag_time;
  if (!_values.find(tag, value)) {
    while (1) {
      if (!read_line(tag_time))
        return false;
      time = value->cur_time();
      value->get(time);
      return true;
    }
  }

  time = value->cur_time();
  while (1) {
    if (!read_line(tag_time))
      return false;
    if (value->cur_time() != time) {
      time = value->cur_time();
      value->get(time);
      return true;
    }
  }
}

// get the values associated with tag associated with time "play_time"
// and fill the associated addresses.
bool BinPlayer::get(const char* tag, Time& play_time)
{
  Time cur_time;
  if (!_read_time) {
    if (!read_line(cur_time))
      return false;
    _first_time = cur_time;
    _read_time = true;
    if (!_zero_time.getValue()) 
      setZero(_first_time);
    else
      setZero(_zero_time);
  }

  BinPlayerTag* value;
  while (!_values.find(tag, value)) {
    if (!read_line(cur_time))
      return false;
  }

  // read through file until value's time value is greater than the current
  // time
  cur_time = play_time - _offset;
  Time tag_time;
  while (value->cur_time() < cur_time) {
    if (!read_line(tag_time))
      return false;
  }

  bool res = value->get(cur_time);
  play_time = cur_time + _offset;

  return res;
}

// get the values associated with tag associated with time closest to
// but not newer than "play_time" and fill the associated addresses.
bool BinPlayer::getPrevious(const char* tag, Time& play_time)
{
  Time cur_time;
  if (!_read_time) {
    if (!read_line(cur_time))
      return false;
    _first_time = cur_time;
    _read_time = true;
    if (!_zero_time.getValue()) 
      setZero(_first_time);
    else
      setZero(_zero_time);
  }

  BinPlayerTag* value;
  while (!_values.find(tag, value)) {
    if (!read_line(cur_time))
      return false;
  }

  // read through file until value's time value is greater than the current
  // time
  cur_time = play_time - _offset;
  Time tag_time;
  while (value->cur_time() < cur_time) {
    if (!read_line(tag_time))
      return false;
  }

  bool res = value->getPrevious(cur_time);
  play_time = cur_time + _offset;

  return res;
}

// Read a line from the file, updating the appropriate BinPlayerValue
bool BinPlayer::read_line(Time& time)
{
  String tag;
  if (!_input.read(tag))
    return false;

  BinPlayerTag* tagged;
  char c;
  String name, fmt;
  if (tag == "D") {
    if (!_input.read(tag))
      return false;

    if (!_values.find(tag.getString(), tagged)) {
      tagged = new BinPlayerTag();
      _values.enter(tag.getString(), tagged);
    }
    tagged->setAttributes(_byte_order, _alignment);

    while (1) {
      if (!_input.read(c))
        return false;
      if (c==';')
        return read_line(time);
      _input.putBack(c);
        
      if (!_input.read(name, '=') || !_input.read(fmt))
        return false;
      tagged->declare(name.getString(), fmt.getString());
    }
  }

  if (!_values.find(tag.getString(), tagged)) {
    return false;
  }

  unsigned long tsec, tusec;
  if (!read_word(_input, (unsigned char*) &tsec, _byte_order) ||
      !read_word(_input, (unsigned char*) &tusec, _byte_order))
    return false;

  time.setValue(tsec, tusec);

  return tagged->read_line(time, _input);
}

// Look in the params for the name of the player, look for that player
// in globals.  If it is there, return it, else create a new player
// and use params to get the file name, open it, and return it.
BinPlayer* BinPlayer::create(ConfigFile* params, SymbolTable* globals)
{
  const char* player_name =
    params->getString("playerName", "DefaultBinPlayer");
  BinPlayer* res = (BinPlayer*) globals->get(player_name);
  if (!res) {
    const char* file_name = params->getString("file", "test.log");
    res = new BinPlayer();
    if (!res->open(file_name)) {
      delete res;
      res = (BinPlayer*) NULL;
      return NULL;
    }
  }

  int zero_time[2];
  zero_time[0] = zero_time[1] = 0;
  Time* gzero_time;
  if (params->getInts("zero_time", zero_time, 2, zero_time, 0)) {
    Time zero(zero_time[0], zero_time[1]);
    res->setZero(zero);
  } else {
    if (params->getBool("real_time", false)) {
      if (!(gzero_time = (Time*) globals->get("player_zero_time"))) {
        gzero_time = new Time;
        Time now = Time::getTimeOfDay();
        long sec, usec;
        now.getValue(sec, usec);
        gzero_time->setValue(sec, usec);
        res->setZero(*gzero_time);
        globals->set("player_zero_time", gzero_time,
                     new SymbolDeleter<Time>);
      }
      res->setZero(*gzero_time);
    }
  }

  return res;
}

__UTILS_END_NAMESPACE
