#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/serial_reg.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/time.h>
#include <asm/io.h>
#include <asm/ioctl.h>
#include <asm/uaccess.h>
#include "bc635.h"  
#include "bc635_kern.h"
MODULE_LICENSE("GPL");
int master = 0;
MODULE_PARM(master,"i");
MODULE_PARM_DESC (master,"When 1, forces BC635 to generate an IRIG-B signal. Slave by default.");

#define MSECS_TO_JIFFIES(ms) (((ms)*HZ+999)/1000)

//number of milliseconds to wait for TFP response.
#define WAIT_LIMIT 128

//Number of seconds between updates of the clock to the system clock
//when the device is not in use.
#define UPDATE_INTERVAL_SECS 30*60

//Number of seconds of error before the update thread forces a correction
#define UPDATE_ERROR 5

struct pci_dev *bc635_dev = NULL;
int bc635_major = 0;
unsigned int  *iomem1;//device registers
unsigned char *iomem2;//dual port RAM interface
unsigned char *input;
unsigned char *output;
unsigned int *ack_reg;
unsigned int *mask_reg;
unsigned int *time_req;
unsigned int *event_req;
unsigned int *time_maj;
unsigned int *event_maj;
unsigned int *time_min;
unsigned int *event_min;
unsigned int *control;
unsigned int *unlock;
int wait_for_ack();
int dev_fs_init();
int software_reset();
int print_info();
int initialized = 0;
typedef struct bc635_thread_struct
{
  int tid; 
  int thread_die;
  struct completion thread_exited;
}bc635_thread_type;
bc635_thread_type bc635_update_thread_info;
bc635_thread_type bc635_sync_thread_info;
static int bc635_update_thread (void *data);
static int bc635_sync_thread (void *data);
static int cleanup_thread(bc635_thread_type *bc);
/**********************************************************/
/**********************************************************/
static int     bc635_open (struct inode* inode, struct file* file);
static int     bc635_close(struct inode* inode, struct file* file);
static ssize_t bc635_read (struct file *, char *, size_t, loff_t *);
static ssize_t bc635_write(struct file *, const char *, size_t, loff_t *);
static int     bc635_ioctl(struct inode* inode, struct file* file,
			   unsigned int cmd, unsigned long arg);
static struct file_operations bc635_fops = {
  NULL,         /* module owner       */
  NULL,		/* lseek	      */
  bc635_read,   /* read		      */
  bc635_write,  /* write	      */
  NULL,		/* readdir 	      */
  NULL,	        /* poll 	      */
  bc635_ioctl,	/* ioctl 	      */
  NULL,		/* mmap		      */
  bc635_open,	/* open 	      */
  NULL,		/* flush	      */
  bc635_close,  /* release 	      */
  NULL,		/* fsync 	      */
  NULL,		/* fasync 	      */
  NULL,		/* lock               */
  NULL,		/* readv              */
  NULL,		/* writev             */
  NULL,		/* sendpage           */
  NULL          /* get_unmapped_area  */
};
/**********************************************************/
/**********************************************************/
static int bc635_sync_thread(void *data)
{  
  struct bc635_thread_struct *bc = (struct bc635_thread_struct *)data;
  struct timeval tv;
  unsigned int temp;
  unsigned long secs; 
  unsigned long usecs;
  char nsecs;
  char status;
  int i;

  daemonize();
  reparent_to_init();
  strcpy (current->comm, DEVICE_NAME); 


  if (software_reset())
    {printk("BC635: Unable to software_reset().\n");
    goto exit_fail;}

  if (print_info())
    {printk("BC635: Unable to print_info().\n");
    goto exit_fail;}
    
  //set clock source to internal
  input[0] = 0x20;
  input[1] = 0x49;
  if (wait_for_ack())
    {printk("BC635: Unable to set internal time source.\n");
    goto exit_fail;}

  //set the time code generator to IRIG-B
  input[0] = 0x1B;
  input[1] = 0x42;
  if (wait_for_ack())
    {printk("BC635: Unable to set time code generator output.\n");
    goto exit_fail;}

  //set the time code receiver to IRIG-B AM
  input[0] = 0x15;
  input[1] = 0x42;
  if (wait_for_ack())
    {printk("BC635: Unable to set time code receiver to IRIG-B.\n");
    goto exit_fail;}
  
  input[0] = 0x16;
  input[1] = 0x4D;
  if (wait_for_ack())
    {printk("BC635: Unable to set time code receiver to AM mode.\n");
    goto exit_fail;}

  //set time format to unix binary
  input[0] = 0x11;
  input[1] = 0x01;
  if (wait_for_ack())
    {printk("BC635: Unable to set binary time format.\n");		
    goto exit_fail;}

  //configure slave clock to sync to time code
  if (master == 0)
    {
      printk("BC635: Configuring as slave.\n");
      input[0] = 0x10;
      input[1] = 0x00;
      if (wait_for_ack())
	{printk("BC635: Unable to set slave timing mode.\n");
	goto exit_fail;}
      //It takes a second to change from master to slave mode.
      //This is in here just in case
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(MSECS_TO_JIFFIES(2000));
    }

  if (master == 1)
    {
      printk("BC635: Configuring as master.\n");
      //configure master to flywheel
      input[0] = 0x10;
      input[1] = 0x01;
      if (wait_for_ack())
	{printk("BC635: Unable to set master timing mode.\n");
	goto exit_fail;} 
    };

  //create the time source monitoring thread - if master
  //Will set master time to system time immediately
  if (master == 1)
    {
      printk("BC635: Setting master clock.\n");
      //set Datum time to system time    
      do_gettimeofday(&tv);	 
      temp = tv.tv_sec;
      input[0] = 0x12;
      for (i = 1; i <= 4; i++)
	{
	  input[5-i] = temp & 0xFF;
	  temp = temp >> 8;
	}
      if (wait_for_ack())
	{printk("BC635; Unable to set Datum time.\n");
 	 goto exit_fail;
	}
    }

  //synchronize with timesource before proceeding.
  status = 0x01;
  printk("BC635: Synchronizing with timesource.");
  //wait five minutes for the master to come online.
  for (i = 0; (i < 5*60) && (status & 0x01); i++)  
    {    
      if (bc->thread_die)
	goto exit_fail;	
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(MSECS_TO_JIFFIES(1000));
      printk(".");
      bc635_read_time(&secs,&usecs,&nsecs,&status,&tv);
    }  
  printk("\n");
 
  if (status & 0x01)
    {printk("BC635: Unable to sync to time source.\n");
     goto exit_fail;
    }

  *mask_reg  = 0x00; //disable all interrupt sources
  *control   = 0x08 | 0x04 | 0x00 | 0x00; //Enable event capture, rising edge active, event input, lockout disabled.
  *unlock    = 0x00; //Disable the capture lockout, of necessary.

  if (master == 1)
    {
      init_completion(&bc635_update_thread_info.thread_exited);
      bc635_update_thread_info.tid = kernel_thread(bc635_update_thread, &bc635_update_thread_info, \
						   CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
      if (bc635_update_thread_info.tid < 0)
	{printk("BC635: Unable to create monitoring thread.\n");
	goto exit_fail;}
    }

  printk("BC635: Initialization complete.\n");

  initialized = 1;
  complete_and_exit(&(bc->thread_exited),0);
  return(0);
 exit_fail:
  initialized = -1;
  complete_and_exit(&(bc->thread_exited),0);
  return(0);
}
/**********************************************************/
/**********************************************************/
static int bc635_update_thread(void *data)
{
  struct bc635_thread_struct *bc = (struct bc635_thread_struct *)data;
  struct timeval tv;
  unsigned int temp;
  int i;
  int count=0;
  unsigned long secs;
  unsigned long usecs;
  unsigned char nsecs;
  unsigned char status;

  daemonize();
  reparent_to_init();
  strcpy (current->comm, DEVICE_NAME);

  do
    {
      if (bc->thread_die)
	break;
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(MSECS_TO_JIFFIES(1000));
      count = (count + 1) % UPDATE_INTERVAL_SECS;
      if (count == 0)
	{           
	  if (GET_USE_COUNT(THIS_MODULE) == 0)
	    {
	      if (!(bc635_read_time(&secs,&usecs,&nsecs,&status,&tv)) && (!(status & 0x01)))
		{
		  do_gettimeofday(&tv);	 
		  temp = tv.tv_sec;
		  if (abs(temp-secs) >= UPDATE_ERROR)
		    {
		      printk("BC635: Updating master clock.\n");
		      //set Datum time to system time    
		      
		      input[0] = 0x12;
		      for (i = 1; i <= 4; i++)
			{
			  input[5-i] = temp & 0xFF;
			  temp = temp >> 8;
			}
		      if (wait_for_ack())
			printk("BC635; Unable to set Datum time.\n");
		    }
		}	    
	    }
	}
    } while (!signal_pending(current));   
  complete_and_exit(&(bc->thread_exited),0);
}
/**********************************************************/
/**********************************************************/
static int cleanup_thread(struct bc635_thread_struct *bc)
{
  if (bc->tid >= 0) 
    {
      bc->thread_die = 1;
      wmb();
      wait_for_completion(&bc->thread_exited);      
      return(0);
    }
  printk("BC635: cleanup_thread attempted to delete nonexistant thread.\n");
  return(-1);
}
/**********************************************************/
/**********************************************************/
//We happily assume that the first client of the driver is the
//master, and has priviledges to slave the BC635 clock to system
//time.
static int bc635_open(struct inode* inode, struct file* fp)
{
  if (bc635_is_initialized() != 1)
    return(-1);
  MOD_INC_USE_COUNT;
  return(0);
} 
/**********************************************************/
/**********************************************************/
static int bc635_close(struct inode* inode, struct file* fp)
{ 
  if (bc635_is_initialized() != 1)
    return(-EINVAL);
  MOD_DEC_USE_COUNT;
  return(0);
}
/**********************************************************/
/**********************************************************/
//Use this to set the time. There is no privilige checking here.
static ssize_t bc635_write(struct file *fp, const char *buf, size_t n, loff_t *o)
{
 unsigned long secs;
 int i;

 if (bc635_is_initialized() != 1)
   return(-EINVAL);
 if (master != 1)
   return(-EINVAL);
 //Is the nth character always a null?
 if ((buf[n] != '\0') || (sscanf(buf,"%ld",&secs) != 1))
   {
     printk("BC635: bc635_write - Invalid date string.\n");
     return(-EINVAL);
   }
 
 printk("BC635: Updating master clock.\n");
 //set Datum time to system time  
 input[0] = 0x12;
 for (i = 1; i <= 4; i++)
   {
     input[5-i] = secs & 0xFF;
     secs = secs >> 8;
   }
 if (wait_for_ack())
   printk("BC635: Unable to set Datum time.\n");
 printk("BC635: Update successful.\n");
 return(n);
}
/**********************************************************/
//A very stupid read. Returns the time in ASCII in one gulp.
static ssize_t bc635_read(struct file *fp, char *buf, size_t n, loff_t *o)
{
 unsigned long secs,usecs;
 unsigned char nsecs,status;
 long long int a,b;
 struct timeval tv;
 char time[256];
 int i;
 if (bc635_is_initialized() != 1)
   return(-EINVAL);

 bc635_read_time(&secs,&usecs,&nsecs,&status,&tv);
 a = 1000000*((long long)(secs)  - (long long)(tv.tv_sec));
 b = (long long)(usecs) - (long long)(tv.tv_usec);
 a = a + b;
 
 sprintf(time,"Datum:%lu:%06lu.%u Stat: %x Sys:%lu:%06lu Diff: %06ld us\n",\
	 secs,usecs,nsecs,status,tv.tv_sec, tv.tv_usec,\
	 (long int)a);
 for (i = 0; i < n; i++)
   {
     if (time[i] == '\0') break;
     buf[i] = time[i];
   }
 return(i);
}
/**********************************************************/
/**********************************************************/
static int bc635_ioctl(struct inode* inode, struct file* file,
                       unsigned int cmd, unsigned long arg)
{
  unsigned long secs,usecs;
  unsigned char nsecs,status;
  struct timeval tv;
  BC635_IOCTL_STRUCT *i_struct = (BC635_IOCTL_STRUCT *)arg; 
  struct bc635_time_struct *temp1;

  if (bc635_is_initialized() != 1)
    return(-EINVAL);
  if (_IOC_TYPE(cmd) != BC635_IOC_MAGIC) return -EINVAL;
  if (_IOC_NR(cmd) > BC635_IOC_MAXNR) return -EINVAL;
  
  switch (cmd) 
    {
     case BC635_GETNEWTIME:
          if (bc635_read_time(&secs,&usecs,&nsecs,&status,&tv))
	    return(-EINVAL);
	  temp1 = &(i_struct->bc635.new_time_struct);
	  temp1->secs = secs;
	  temp1->usecs = usecs;
	  temp1->nsecs = nsecs;
	  temp1->status = status;
	  temp1->sys_time = tv;
          return(0);
    case BC635_GETOLDTIME:
          if (bc635_read_time(&secs,&usecs,&nsecs,&status,&tv))
	    return(-EINVAL);
	  i_struct->bc635.timeval_struct.tv_sec  = secs;
	  i_struct->bc635.timeval_struct.tv_usec = usecs; 
          return(0);
    case BC635_GETEVENTTIME:
          bc635_read_event_time(&secs,&usecs,&nsecs,&status);
	  temp1 = &(i_struct->bc635.new_time_struct);
	  temp1->secs = secs;
	  temp1->usecs = usecs;
	  temp1->nsecs = nsecs;
	  temp1->status = status;	 
          return(0);
     default:
          return -EINVAL;
    }
  return(-EINVAL);//should never get here
}
/**********************************************************/
/**********************************************************/
int dev_fs_init()
{
  bc635_major = register_chrdev(bc635_major, DEVICE_NAME, &bc635_fops);
  if (bc635_major < 0) 
    {printk("BC635: Unable to attatch to devfs. %d\n",bc635_major);
     return(-1);}
  return(0);
}
/**********************************************************/
/**********************************************************/
int wait_for_ack()
{
  int i;
  *ack_reg = 0x81; 
  for (i = 0; i < WAIT_LIMIT; i++)
    {  
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(MSECS_TO_JIFFIES(10));   
      if (*ack_reg & 0x01)
	break;
    }  
  if (i == WAIT_LIMIT)
     return(-1);
  return(0); 
}
/**********************************************************/
/**********************************************************/
//allows other modules in the kernel to know that the bc635
//is finished initializing.
// 1 = OK
// 0 = Unitialized
//-1 = Error
int bc635_is_initialized()
{
  return(initialized);
}
/**********************************************************/
/**********************************************************/
/*This function reads the event time register.
  There currently is no interrupt management procedure. This is good
  enough for the purposes of verifying the SICK timing.
*/
int bc635_read_event_time(unsigned long *secs, unsigned long *usecs, \
			  unsigned char *nsecs,unsigned char *status)
{
  unsigned long flags; 
  save_flags(flags); cli();
  *event_req = 0;
  *secs = *event_maj;
  *usecs = (*event_min & 0xFFFFF);
  *nsecs = (*event_min &  0xF00000) >> 20;
  *status = (*event_min & 0xF000000)>> 24;
  printk("%ld %ld\n",*secs,*usecs);
  restore_flags(flags);
  return(0);
}
/**********************************************************/
/**********************************************************/
int bc635_read_time(unsigned long *secs, unsigned long *usecs, \
		    unsigned char *nsecs,unsigned char *status,
		    struct timeval *tv)
{
  unsigned long flags; 
  if (tv == NULL)
    return(-1);
  save_flags(flags); cli();
  *time_req = 0;//latch time in registers
  do_gettimeofday(tv);
  *secs = *time_maj;
  *usecs = (*time_min & 0xFFFFF);
  *nsecs = (*time_min &  0xF00000) >> 20;
  *status = (*time_min & 0xF000000)>> 24;
  restore_flags(flags);
  return(0);
}
/**********************************************************/
/**********************************************************/
int print_info()
{ int i;
  int temp;
 
  printk("BC635: Board Information:\n");
  input[0] = REQUEST_TFP_DATA;
  input[1] = MODEL_ID;
  if (wait_for_ack())
    return(-1);
  printk("\tModel:                     ");
  for (i = 1; i <= 8; i++)
    printk("%c",output[i]);
  printk("\n");

  input[0] = REQUEST_TFP_DATA;
  input[1] = SERIAL_NUMBER;
  if (wait_for_ack())
    return(-1);
  printk("\tSerial Number:             ");
  temp = 0;
  for (i = 1; i <= 4; i++)
    temp = ((temp << 8) | output[i]);
  printk("%d",temp);
  printk("\n"); 
	   
  input[0] = REQUEST_TFP_DATA;
  input[1] = HARDWARE_FAB_NUM;
  if (wait_for_ack())
    return(-1);
  printk("\tHardware Fab Part Number:  ");
  temp = 0;
  for (i = 1; i <= 2; i++)
    temp = ((temp << 8) | output[i]);
  printk("%d",temp);
  printk("\n"); 
   
  input[0] = REQUEST_TFP_DATA;
  input[1] = ASSEMBLY_PART;
  if (wait_for_ack())
    return(-1);
  printk("\tAssembly Part Number:      ");
  temp = 0;
  for (i = 1; i <= 2; i++)
    temp = ((temp << 8) | output[i]);
  printk("%d",temp);
  printk("\n"); 

  input[0] = REQUEST_TFP_DATA;
  input[1] = PCI_FIRMWARE;
  if (wait_for_ack())
    return(-1);
  printk("\tPCI Firmware Number:       ");
  temp = 0;
  for (i = 1; i <= 11; i++)
    printk("%c",output[i]);
  printk("\n"); 
  return(0);
}
/**********************************************************/
/**********************************************************/
int software_reset()
{
  input[0] = 0x1A;
  input[1] = 0x01;
  if (wait_for_ack())
    return(-1);
  set_current_state(TASK_INTERRUPTIBLE);
  schedule_timeout(MSECS_TO_JIFFIES(1500));
  return(0);
}
/**********************************************************/
/**********************************************************/
char configure_bc635(struct pci_dev *dev)
{
  unsigned short sub_vendor_id, sub_id;
  unsigned int start, length;  

  pci_enable_device(dev);
  pci_read_config_word(dev, 0x2C, &sub_vendor_id);
  pci_read_config_word(dev, 0x2E, &sub_id);
  if (sub_vendor_id !=  SUBSYSTEM_VENDOR_ID || sub_id != SUBSYSTEM_ID )
    {printk(KERN_ALERT "BC635: Invalid subsystem or subvendor ID: %x %x",\
	    sub_vendor_id, sub_id);
    return(-1);}

  /* request associated memory regions */
  if (pci_request_regions(dev,"BC635"))
     {printk("BC635: Unable to request BC635 I/O regions.\n");
      return(-1);}   

  bc635_dev = dev;
  start = pci_resource_start(dev,0);    
  length= pci_resource_len(dev,0);    
  iomem1 = (unsigned int*)ioremap(start,length);  
  if (iomem1 == NULL)
    goto pci_error;
  printk("BC635: Remapping PCI memory at %X length %X to address %X.\n",\
	 start,length,(unsigned int)iomem1); 

  start = pci_resource_start(dev,1);    
  length= pci_resource_len(dev,1);    
  iomem2 = (unsigned char *)ioremap(start,length);  
  if (iomem2 == NULL)
    goto iomem1_error;
  printk("BC635: Remapping PCI memory at %X length %X to address %X.\n",\
	 start,length,(unsigned int)iomem2);

  //these are 32 bit PCI registers
  time_req   = iomem1+0x0;
  event_req  = iomem1+0x1;
  unlock     = iomem1+0x2;
  control    = iomem1+0x4;
  time_maj   = iomem1+0xD;
  time_min   = iomem1+0xC;
  event_maj  = iomem1+0xF;
  event_min  = iomem1+0xE;
  ack_reg    = iomem1+0x5;
  mask_reg   = iomem1+0x6;

  input  = iomem2+0x102;//magic constants
  output = iomem2+0x082;

  init_completion(&bc635_sync_thread_info.thread_exited);
  bc635_sync_thread_info.tid = kernel_thread(bc635_sync_thread, &bc635_sync_thread_info, \
					     CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
  if (bc635_sync_thread_info.tid < 0)
    {printk("BC635: Unable to create synchronization thread.\n");
     goto iomem2_error;}

  if (dev_fs_init())
    {      
      printk("BC635: Failed to initialize dev_fs.\n");
      goto sync_thread_error;
    }   
  return(0);

  //handle the various error conditions so that we exit gracefully
 sync_thread_error:
  cleanup_thread(&bc635_sync_thread_info);
  if (master == 1)
    cleanup_thread(&bc635_update_thread_info);    
 iomem2_error:
  iounmap((void*)iomem2);
 iomem1_error:
  iounmap((void*)iomem1);
 pci_error:
  pci_release_regions(bc635_dev); 
  return(-1);
}
/**********************************************************/
/**********************************************************/
int init_module(void)
{ 
  struct pci_dev *dev = NULL;
  int count = 0;

  while ((dev = pci_find_device(VENDOR_ID, DEVICE_ID, dev)))
    {if (configure_bc635(dev) == -1)
      return(-1);
      count++;
     if (count > 1)
      {printk("BC635: This driver can handle only one Datum card.\n");
       return(-1);}
    }
  if (count == 0)
     {printk("BC635: Unable to find BC635.\n");
      return(-1);}
  return(0);
}
/**********************************************************/
/**********************************************************/
void cleanup_module(void)
{  
  if (bc635_sync_thread_info.tid != 0)
    {
      printk("BC635: Killing synchronization thread.\n");
      cleanup_thread(&bc635_sync_thread_info);
    }
  if (bc635_update_thread_info.tid != 0)
    {printk("BC635: Killing update thread.\n");
     cleanup_thread(&bc635_update_thread_info);
    }
  if (bc635_major > 0)
    unregister_chrdev(bc635_major, DEVICE_NAME);
  pci_release_regions(bc635_dev);
  iounmap((void*)iomem1); 
  iounmap((void*)iomem2);
}
/**********************************************************/
/**********************************************************/
