#ifndef CONFIG_MODULE_H
#define CONFIG_MODULE_H

#include <utils/Time.h>
#include <utils/String.h>
#include <utils/List.h>

class ConfigSource;

namespace utils {
  class SymbolTable;
  class Managed;
  class ConfigFile;
};

class Module {
 public:
  Module(const char* mod_name, const char* spec_string);
  virtual ~Module();

  // you must define these
  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table)=0;
  virtual bool run() = 0;

  // you may define these
  virtual bool initializeDisplay(ConfigSource*) { return false; }
  virtual bool display() { return true; };
  virtual bool displayIdle() { return true; };

  void mainLoop();  // run a module until it is finished

  // run is built out of these two methds
  bool startup();  // start up the module
  bool cycle();  // run through one cycle

  template<class T> T* create(const char* spec) {
    T* res = T::create(spec, _symbol_table);
    if (res) {
      res->ref();
      _intfs.prepend(res);
      return res;
    } else
      return NULL;
  }
  template<class T> T* create(utils::ConfigFile& params) {
    T* res = T::create(params, _symbol_table);
    if (res) {
      res->ref();
      _intfs.prepend(res);
      return res;
    } else
      return NULL;
  }
    
  template <class T> void clear(T* intf) {
    unrefIntf(intf);
    T::clear(_symbol_table);
  }
  void unrefIntf(utils::Managed* intf);
    
  // convenience access functions
  ConfigSource* getConfigSource() const { return _config_src; }
  utils::SymbolTable* getSymbolTable() const { return _symbol_table; }
  const char* getModuleName() const { return _name.getString(); }
  int getCycleNumber() const { return _cycle_number; }
  bool isDisplaying() const;
  bool isRunning() const;
  bool isPaused() const;

 private:
  ConfigSource* _config_src;
  utils::SymbolTable* _symbol_table;
  utils::String _name;
  bool _display, _running, _paused;
  float _idle_time;
  int _cycle_number;
  bool _making_movie;
  float _movie_step;
  utils::String _movie_script;
  char* _movie_buf;
  int _movie_time_key;
  float _display_interval;
  utils::Time _last_display;
  utils::List<utils::Managed> _intfs;
  bool _timing;
  int _timing_periods;
  utils::Time _last;
  utils::Time _start;
  double _elapsed;
  bool _timing_details;
  float _elapsed_run;
  float _elapsed_display;
  int _n_run;
  int _n_display;
  float _cycle_time;
  utils::Time _last_cycle;
};

#ifdef NO_MAIN
#define MODULE_MAIN(nickname,ModuleClass)
#else
#define MODULE_MAIN(nickname,ModuleClass) \
int main(int argc, char** argv) {\
  const char* config_spec; \
  if (argc < 2) config_spec=UTILS__QUOTE(nickname) ".cfg"; \
  else config_spec=argv[1]; \
  ModuleClass* module = new ModuleClass(config_spec); module->mainLoop(); \
  delete module; \
}
#endif

#endif
