/***************************************************************************

                                ConfigSource.h

  Defines an API which access a hierarchical, typed configuration and and
  parameter source where parameters can be queried (static) or attached to a
  memory location or callback function (dynamic).  Static parameters stay the
  same for the duration of the program.  When an external agent causes a
  dynamic parameter to change, the appropriate memory location is changed or
  callback function is called as registered by the user of the configuration
  source.  This file also defines C cover functions for most configuration
  source capabilities

  ConfigSource recognizes a variety of data types:
    int, float, double, char, string, and bool
  are atomic types.  In addition there are complex primitives,
    struct, structArray, spec

  The configuration files themselves have a fairly flexible syntax.  

    <type decl> <name>[num_elems] = <list of types of length num elem>;

  If <type decl> is ommitted, everything after the equal sign up to the 
  first semicolon not inside a pair of brackets is stored in an UnknownElem,
  which will resolve to a type upon user querying the ConfigFile.

  If [num_elems] is present the list of data must be of length num elem,
  otherwise the list can be of any length:  num_elems is purely for redundancy.

  The atomic types are fairly straightforward, except for strings. A string
  is either a white-space delimited word, or enclosed in quotes or enclosed
  in matching curly brackets.  For example, the following is a valid string
  declaration
    string str_example = first "second string" { third { string { here }}};

  Structures can be declared using the "struct" type, and have a slightly
  different syntax:

    struct name {
       int example = 1;
       <other declarations>
    }

  Note no trailing semicolon is necessary, but stylistically is probably
  a good idea (and certainly would not cause any harm).

  Just for grins, structure values can be added and reset after the structure
  is declared.  For example, a config file which contained the above structure
  could have later in the file a statement

     name.example = 2;

  which would override the original definition of name.  You could also do

     int name.new_thing = 1;

  Which will add the integer element "new_thing" to the structure name.  If
  name does not exist, a structure element will be created.

  You can declare a named array of structures with the structArray type:

     structArray array {
       {
         int thing=1;
         < struct 1 declarations >
       }
       .
       .
       {
        < struct n declarations >
       }
     }

  A wrinkle is that you can refer later to individual elements of a structure
  array:

     array[0].thing = 2;

  will reset "thing" in the first element of array.  You cannot do this
  with arbitrary types, i.e., setting the nth element of an array of ints.

  The comment character in the parameter files is '#' (inherited from Input)

***************************************************************************/

#ifndef INTF_CONFIG_SOURCE
#define INTF_CONFIG_SOURCE

#include <utils/Interface.h>
#include <utils/Type.h>

typedef void (*ConfigCallback)(const char* name, utils::Type type,
                               void* data, int num_elems, void* cbdata);

#ifdef __cplusplus

namespace utils {
    class ConfigElem;
    class ConfigFile;
};

UTILS_INTERFACE(ConfigSource) {
public:
  enum ModuleState {
    NOT_RUNNING = 0,
    INITIALIZING,
    INITIALIZING_DISPLAY,
    INITIALIZED,
    RUNNING,
    PAUSED,
    BLOCKED,
    ERROR
  };

  struct ModuleStatus {
    float status;
    float confidence;
    float avg_cycle_time;
    const char* msg;
  };
  
  // work horse function for getting static parameters
  virtual int get(utils::Type, const char* name, void* data, int max_num=1,
                  void* default_data=NULL, int num_defaults=1) = 0;

  // work horse function for attaching dynamic parameters
  virtual bool attach(utils::Type, const char* name, void* data, int max_num=1,
                      ConfigCallback notify=NULL, void* cbdata = NULL,
                      void* default_data=NULL, int num_defaults=1) = 0;

  virtual int numValues(const char* name) = 0;

  // Resets a parameter
  virtual bool set(const char* name, const char* value) = 0;

  // convenience methods for getting singleton parameters
  virtual int getInt(const char* name, int def=0);
  virtual float getFloat(const char* name, float def=0.0);
  virtual double getDouble(const char* name, double def=0.0);
  virtual char getChar(const char* name, char def='\0');
  virtual const char* getString(const char* name, const char* def = "");
  virtual bool getBool(const char* name, bool def=false);

  // convenience methods for getting parameter arrays
  virtual int getInts(const char* name, int* data, int max_num=1,
                      int* default_data=NULL, int num_defaults=1);
  virtual int getFloats(const char* name, float* data, int max_num=1,
                        float* default_data=NULL, int num_defaults=1);
  virtual int getDoubles(const char* name, double* data, int max_num=1,
                         double* default_data=NULL, int num_defaults=1);
  virtual int getChars(const char* name, char* data, int max_num=1,
                       char* default_data=NULL, int num_defaults=1);
  virtual int getStrings(const char* name, const char** data, int max_num=1,
                         const char** default_data=NULL, int num_defaults=1);
  virtual int getBools(const char* name, bool* data, int max_num=1,
                       bool* default_data=NULL, int num_defaults=1);
  virtual int getStructs(const char* name, utils::ConfigElem** data,
                         int max_num=1);

  // downloads an entire substructure into a static configuration file
  virtual utils::ConfigFile* getSubFile(const char* name);
  virtual bool getStruct(const char* name, utils::ConfigFile&);

  // just attaches a callback, with no value setting
  virtual bool notify(const char* name,
                      ConfigCallback notify, void* cb_data=0L);

  // convenience function for attaching to singleton dynamic parameters
  virtual bool attachInt(const char* name, int* data, int def=0,
                         ConfigCallback notify=NULL, void* cb_data=0L);
  virtual bool attachFloat(const char* name, float* data, float def=0.0,
                           ConfigCallback notify=NULL, void* cb_data=0L);
  virtual bool attachDouble(const char* name, double* data, double def=0.0,
                            ConfigCallback notify=NULL, void* cb_data=0L);
  virtual bool attachChar(const char* name, char* data, char def='\0',
                          ConfigCallback notify=NULL, void* cb_data=0L);
  virtual bool attachString(const char* name, const char** data,
                            const char* def = "",
                            ConfigCallback notify=NULL, void* cb_data=0L);
  virtual bool attachBool(const char* name, bool* data, bool def=false,
                          ConfigCallback notify=NULL, void* cb_data=0L);

  // convenience function for attaching to dynamic parameter arrays
  virtual bool attachInts(const char* name, int* data, int max_num=1,
                          ConfigCallback notify=NULL, void* cb_data=0L,
                          int* default_data=NULL, int num_defaults=1);
  virtual bool attachFloats(const char* name, float* data, int max_num=1,
                            ConfigCallback notify=NULL, void* cb_data=0L,
                            float* default_data=NULL, int num_defaults=1);
  virtual bool attachDoubles(const char* name, double* data, int max_num=1,
                             ConfigCallback notify=NULL, void* cb_data=0L,
                             double* default_data=NULL, int num_defaults=1);
  virtual bool attachChars(const char* name, char* data, int max_num=1,
                           ConfigCallback notify=NULL, void* cb_data=0L,
                           char* default_data=NULL, int num_defaults=1);
  virtual bool attachStrings(const char* name,
                             const char** data, int max_num=1,
                             ConfigCallback notify=NULL, void* cb_data=0L,
                             const char** default_data=NULL,
                             int num_defaults=1);
  virtual bool attachBools(const char* name, bool* data, int max_num=1,
                           ConfigCallback notify=NULL, void* cb_data=0L,
                           bool* default_data=NULL, int num_defaults=1);

  virtual bool set(utils::Type, const char* name, void* data,
                   int num_values) = 0;
  virtual bool setInt(const char* name, int data);
  virtual bool setFloat(const char* name, float data);
  virtual bool setDouble(const char* name, double data);
  virtual bool setChar(const char* name, char data);
  virtual bool setString(const char* name, const char* data);
  virtual bool setBool(const char* name, bool data);
  virtual bool setInts(const char* name, int* data, int num_values);
  virtual bool setFloats(const char* name, float* data, int num_values);
  virtual bool setDoubles(const char* name, double* data, int num_values);
  virtual bool setChars(const char* name, char* data, int num_values);
  virtual bool setStrings(const char* name, const char** data, int num_values);
  virtual bool setBools(const char* name, bool* data, int num_values);

  // Do whatever work is necessary to do to process configuration source
  // events coming from the outside
  virtual bool processEvents(double timeout=-1) = 0;

  // sleep, processing events, for timeout seconds.  Return true if 
  // any events processed in that time
  virtual bool sleep(double timeout);

  // Mark a cycle
  virtual void reportState(ModuleState) {}
  virtual void setStatus(float status) {}
  virtual void setConfidence(float confidence) {}
  virtual void setStatusMessage(const char* msg) {}

  virtual bool refresh() { return false; }

  UTILS_INTF_DECL(ConfigSource);
};

typedef utils::SymbolTable *UTL_SymbolTable;
typedef ConfigSource *IF_ConfigSource;

extern "C" {

#else

/* C Cover function section */

#include <utils/SymbolTable.h>

typedef void *IF_ConfigSource;

#endif

extern IF_ConfigSource CS_create(const char* spec, UTL_SymbolTable);
bool CS_set(IF_ConfigSource cs, const char* name, const char* value);
int CS_get(IF_ConfigSource cs, int type, const char* name,
           void* data, int max_num, void* default_data, int num_defaults);
int CS_getInt(IF_ConfigSource cs, const char* name, int def);
float CS_getFloat(IF_ConfigSource cs, const char* name, float def);
double CS_getDouble(IF_ConfigSource cs, const char* name, double def);
char CS_getChar(IF_ConfigSource cs, const char* name, char def);
const char* CS_getString(IF_ConfigSource cs, const char* name,const char* def);
bool CS_getBool(IF_ConfigSource cs, const char* name, bool def);

int CS_getInts(IF_ConfigSource cs, const char* name, int* data, int max_num,
               int* default_data, int num_defaults);
int CS_getFloats(IF_ConfigSource cs, const char* name,
                 float* data, int max_num,
                 float* default_data, int num_defaults);
int CS_getDoubles(IF_ConfigSource cs, const char* name,
                  double* data, int max_num,
                  double* default_data, int num_defaults);
int CS_getChars(IF_ConfigSource cs, const char* name, char* data, int max_num,
                char* default_data, int num_defaults);
int CS_getStrings(IF_ConfigSource cs, const char* name,
                  const char** data, int max_num,
                  const char** default_data, int num_defaults);
int CS_getBools(IF_ConfigSource cs, const char* name, bool* data, int max_num,
                bool* default_data, int num_defaults);

bool CS_attachInt(IF_ConfigSource cs, const char* name, int* data, int def,
                  ConfigCallback notify, void* cb_data);
bool CS_attachFloat(IF_ConfigSource cs, const char* name, float* data,
                    float def, ConfigCallback notify, void* cb_data);
bool CS_attachDouble(IF_ConfigSource cs, const char* name, double* data,
                     double def, ConfigCallback notify, void* cb_data);
bool CS_attachChar(IF_ConfigSource cs, const char* name, char* data, char def,
                   ConfigCallback notify, void* cb_data);
bool CS_attachString(IF_ConfigSource cs, const char* name, const char** data,
                     const char* def, ConfigCallback notify, void* cb_data);
bool CS_attachBool(IF_ConfigSource cs, const char* name, bool* data, bool def,
                   ConfigCallback notify, void* cb_data);

bool CS_attachInts(IF_ConfigSource cs, const char* name,
                   int* data, int max_num, ConfigCallback notify,void* cb_data,
                   int* default_data, int num_defaults);
bool CS_attachFloats(IF_ConfigSource cs, const char* name,
                     float* data, int max_num,
                     ConfigCallback notify, void* cb_data,
                     float* default_data, int num_defaults);
bool CS_attachDoubles(IF_ConfigSource cs, const char* name,
                      double* data, int max_num,
                      ConfigCallback notify, void* cb_data,
                      double* default_data, int num_defaults);
bool CS_attachChars(IF_ConfigSource cs, const char* name,
                    char* data, int max_num,
                    ConfigCallback notify, void* cb_data,
                    char* default_data, int num_defaults);
bool CS_attachStrings(IF_ConfigSource cs, const char* name,
                      const char** data, int max_num,
                      ConfigCallback notify, void* cb_data,
                      const char** default_data, int num_defaults);
bool CS_attachBools(IF_ConfigSource cs, const char* name,
                    bool* data, int max_num,
                    ConfigCallback notify, void* cb_data,
                    bool* default_data, int num_defaults);

#ifdef __cplusplus
}
#endif

#endif
