#include <stdio.h>

#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <TimeSource/Shmem.h>
#include <TimeSource/shmem_format.h>

#include <utils/ConfigFile.h>
#include <utils/Generator.h>

TimeSource* create_TimeSource_shmem(TimeSourceGenerator*,
                                    utils::ConfigFile* params,
                                    utils::SymbolTable* globals)
{
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params->getString("ipt_spec",
                                                   "unix:port=0;"));
  if (!com)
    return NULL;

  const char* mem_name = params->getString("name", TIME_SHMEM_NAME);
  const char* machine = params->getString("machine");
  int port = params->getInt("port", 1389);
  char buffer[200];
  if (!*machine) {
    sprintf(buffer, "managed: name=%s;", mem_name);
  } else {
    sprintf(buffer, "managed: name='%s@%s|%d';", mem_name, machine, port);
  }
  const char* mem_spec = params->getString("mem", buffer);
  IPSharedMemory* shm = 
    com->OpenSharedMemory(mem_spec, TIME_SHMEM_FORMAT,
                          sizeof(ShmemTimeStruct));

  if (!shm) {
    printf("Problem opening shared memory %s\n", mem_spec);
    return NULL;
  }

  return new ShmemTimeSource(shm);
}

ShmemTimeSource::ShmemTimeSource(IPSharedMemory* shm)
{
  _shm = shm;
}

bool ShmemTimeSource::getTime(utils::Time& t)
{
  ShmemTimeStruct time;
  if (!_shm->FormattedData(&time))
    return false;
  t.setValue(time.secs, time.usecs);
  _shm->DeleteContents(&time);
  return true;
}

int ShmemTimeSource::getKey()
{
  return 0;
}

bool ShmemTimeSource::setTime(const utils::Time& t, int key)
{
  return false;
}

bool ShmemTimeSource::advanceTime(double step_time)
{
  return false;
}

TimeSource* create_TimeSource_shmemPublish(TimeSourceGenerator* gen,
                                           utils::ConfigFile* params,
                                           utils::SymbolTable* globals)
{
  TimeSource* contained = 
    gen->interface(params->getString("contained", "default"), globals);
  if (!contained) {
    return NULL;
  }
  
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params->getString("ipt_spec",
                                                   "unix: int port=0;"));
  if (!com)
    return NULL;

  const char* mem_name = params->getString("name", TIME_SHMEM_NAME);
  char buffer[200];
  sprintf(buffer, "managed: name=%s; owner=true;", mem_name);
  const char* mem_spec = params->getString("mem", buffer);
  IPSharedMemory* shm = 
    com->OpenSharedMemory(mem_spec, TIME_SHMEM_FORMAT,
                          sizeof(ShmemTimeStruct));
  if (!shm)
    return NULL;

  return new ShmemPublishTimeSource(contained, shm);
}

ShmemPublishTimeSource::ShmemPublishTimeSource(TimeSource* contained,
                                               IPSharedMemory* shm)
{
  _contained = contained;
  _contained->ref();
  _shm = shm;
}

ShmemPublishTimeSource::~ShmemPublishTimeSource()
{
  _contained->unref();
}

bool ShmemPublishTimeSource::getTime(utils::Time& t)
{
  if (!_contained)
    return false;
  if (!_contained->getTime(t))
    return false;

  ShmemTimeStruct tstruct;
  long secs, usecs;
  t.getValue(secs, usecs);
  tstruct.secs = secs;
  tstruct.usecs = usecs;
  _shm->PutFormattedData(&tstruct);

  return true;
}

int ShmemPublishTimeSource::getKey()
{
  if (!_contained)
    return -1;
  return _contained->getKey();
}

bool ShmemPublishTimeSource::setTime(const utils::Time& t, int key)
{
  if (!_contained)
    return false;
  if (!_contained->setTime(t, key))
    return false;

  ShmemTimeStruct tstruct;
  long secs, usecs;
  t.getValue(secs, usecs);
  tstruct.secs = secs;
  tstruct.usecs = usecs;
  _shm->PutFormattedData(&tstruct);

  return true;
}

void ShmemPublishTimeSource::setExtent(const utils::Time& beginning,
                                       const utils::Time& ending)
{
  if (_contained)
    _contained->setExtent(beginning, ending);
}

bool ShmemPublishTimeSource::getExtent(utils::Time& beginning,
                                       utils::Time& ending)
{
  if (_contained)
    return _contained->getExtent(beginning, ending);
  return false;
}

bool ShmemPublishTimeSource::advanceTime(double step_time)
{
  if (_contained)
    return _contained->advanceTime(step_time);
  return false;
}
