#include <time.h>
#include <string.h>

#include <TimeSource/GUI.h>

#include <utils/ConfigFile.h>

#include "TimeSourceIntf.h"

TimeSource* create_TimeSource_gui(TimeSourceGenerator*,
                                  utils::ConfigFile* params,
                                  utils::SymbolTable*)
{
  return new GUITimeSource(params->getBool("continuous", true),
                           params->getFloat("scale", 1.0),
                           params->getFloat("nudge", 1.0),
                           params->getFloat("step", 10),
                           params->getFloat("offset", 0.0),
                           params->getBool("process", true));

}

GUITimeSource::GUITimeSource(bool continuous, float scale, 
                             float nudge, float step, float offset,
                             bool process)
  : UtilsTimeSource(continuous, scale, nudge)
{
  _offset = offset;
  _intf = new TimeSourceIntf(this);
  _intf->nudge->value(nudge);
  _intf->step->value(step);
  _intf->time_slider->deactivate();
  _intf->main->show();
  _paused = false;
  _self = pthread_self();
  _process = process;
}

void GUITimeSource::setExtent(const utils::Time& beginning,
                              const utils::Time& ending)
{
  if (_beginning.isZero() || (beginning < _beginning))
    _beginning = beginning;
  if (_ending.isZero() || (ending > _ending))
    _ending = ending;
  if (!_beginning.isZero() && !_ending.isZero()) {
    _intf->time_slider->activate();
    
    long secs, usecs;
    beginning.getValue(secs, usecs);
    char* time_str = ctime(&secs);
    time_str[strlen(time_str)-1] = '\0';
    _intf->from_output->value(time_str);
    _intf->to_output->value((ending-beginning).getValue());
    _intf->elapsed_output->set_visible();
    utils::Time start = _beginning + utils::Time(_offset);
    setTime(start);
    setSlider(start);
  }
}

bool GUITimeSource::getExtent(utils::Time& beginning, utils::Time& ending)
{
  ending = _ending;
  beginning = _beginning;
  return true;
}

void GUITimeSource::setPercentage(double percentage)
{
  UtilsTimeSource::
    setTime(_beginning +
            utils::Time((_ending-_beginning).getValue()*percentage));
}

void GUITimeSource::setSlider(const utils::Time& time)
{
  if (_beginning.isZero() || _ending.isZero())
    return;
  _intf->time_slider->value((time-_beginning).getValue()/
                            (_ending-_beginning).getValue());

}

void GUITimeSource::setContinuous(bool continuous)
{
  if (continuous)
    utils::Time::useContinuousTime();
  else
    utils::Time::useDiscreteTime();
}

void GUITimeSource::output_time(const utils::Time& time)
{
  if (pthread_self() != _self)
    return;

  if (_process)
    while (Fl::wait(0.0) > 0);
  long secs, usecs;
  time.getValue(secs, usecs);
  char* time_str = ctime(&secs);
  time_str[strlen(time_str)-1] = '\0';
  _intf->time_output->value(time_str);
  _intf->elapsed_output->value((time-_beginning).getValue());
  setSlider(time);
}

bool GUITimeSource::getTime(utils::Time& time)
{
  bool res = UtilsTimeSource::getTime(time);
  if (!res)
    return false;
  output_time(time);
  return true;
}

void GUITimeSource::pause()
{
  _paused = !_paused;
  if (_paused) {
    utils::Time::pushState();
    utils::Time::setTimeConstant(0);
  } else {
    utils::Time::popState();
  }
}

void GUITimeSource::deltaTime(float delta)
{
  utils::Time time;
  UtilsTimeSource::getTime(time);
  time += utils::Time(delta);
  setTime(time);
  output_time(time);
}

void GUITimeSource::setScale(float scale)
{
  utils::Time::setTimeConstant(scale);
}

bool GUITimeSource::setScale(double scale, int)
{
  setScale(scale);
  _intf->scale->value(scale);
  return true;
}
