// ---------------------------------------------------------------
// SPU-Toolbox
//
//        File: pnm_helper.cc
// Description: Support PNM routines
// ---------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file may use Doxgen style comments to facilitate automatic
// documentation generation. More information on PERCEPS can be
// found at "http://www.stack.nl/~dimitri/doxygen/index.html".
//
// Author: Ross J. Micheals
//         rjm2@eecs.lehigh.edu
//
// (c) 1999-2001 Ross J. Micheals

/*!
  
  \file   pnm_helper.cc
  \brief  Support PNM routines
  \author Ross J. Micheals (rjm2@eecs.lehigh.edu)

*/

#include <SPU-Toolbox/pnm_helper.h>

bool
pgm_write_helper(const char* filename, const int& cols, const int& rows,
		 const int mode) 
{
  FILE* fd;
  //  char line[256];
  if ((fd = fopen(filename,"rw")) == NULL ) {
    fprintf(stderr, "Could not open file \"%s\".\n", filename);
    return(false);
  }

  fprintf(fd, "P%d\n\n%d %d\n\n255\n", mode, cols, rows);
  
  fprintf(stderr, "Not yet implemented.\n");

  switch (mode) {

  case 2:
    break;

  case 3:
    break;

  case 5:
    break;

  case 6:
    break;

  default:
    break;

  }
  return(false);
}

bool
pnm_read_helper(const char* filename, 
		int& in_cols, int& in_rows, int& in_mode, 
		int&            in_bytes_per_line, 
		unsigned char*& raw_mem,
		int&            raw_memsize,
		unsigned char*& aligned_mem,
		int&            aligned_memsize,
		const int       byte_alignment)
{
	   
  FILE* fd;
  char line[256];
  int maxval;
  int tmp_int, tmp_int1, tmp_int2, tmp_int3;
  unsigned char* data = 0;
  if ((fd = fopen(filename, "r")) == NULL) {
    fprintf(stderr, "Could not open file \"%s\".\n", filename);
    return(false);
  }
  fgets(line, 256, fd);

  char image_type[4];
  bool valid = false;

  if (strncmp(line, "P2", 2) == 0 ||
      strncmp(line, "P3", 2) == 0 ||
      strncmp(line, "P5", 2) == 0 ||
      strncmp(line, "P6", 2) == 0) {
    strcpy(image_type, line);
    valid = true;
  }  
  if (!valid) {
    fprintf(stderr, "\"%s\" is not a valid PNM file.\n", filename);
    return(false);
  }

  // Save row state before we change them.
  int old_rows = in_rows;
  int old_cols = in_cols;

  int num_read = 0;  
  fgets(line, 256, fd);
  while (line[0] == '#')	// Ignore comments.
    fgets(line, 256, fd);
  num_read = sscanf(line,"%d %d", &in_cols, &in_rows);
  if (num_read != 2) {
    fprintf(stderr, "File \"%s\" does not appear to be a PNM.\n", filename);
    return(false);
  }

  if(raw_mem && (old_rows != in_rows || old_cols != in_cols) ) {
    free(raw_mem);
    raw_mem     = 0; 
    aligned_mem = 0;
  }

  fgets(line, 256, fd);
  num_read = sscanf(line, "%d", &maxval);
  if (num_read != 1) {
    fprintf(stderr, "File \"%s\" does not appear to be a PNM.\n", filename);
    return(false);
  }

  char type = image_type[1];
  in_mode = (int) (type - '0');


  //
  // Allocate aligned memory
  //
  int bpl = in_cols;
  if (type == '3' || type == '6') bpl *= 3;
  
  // Make sure bytes-per-line is aligned
  if (bpl % byte_alignment) 
    bpl = bpl - (bpl % byte_alignment) + byte_alignment;
  in_bytes_per_line = bpl;

  if (!raw_mem) {
    aligned_memsize = in_rows * bpl;
    raw_memsize = aligned_memsize + byte_alignment;
    raw_mem     = (SPU_u8*) malloc( (size_t) raw_memsize );
    if (!raw_mem) {
      fprintf(stderr, "pnm_helper: Memory allocation failure.\n");
      return(false);
    }
    
    if ((unsigned int) raw_mem % byte_alignment) {
      aligned_mem = 
	(raw_mem + byte_alignment) - ((unsigned int) raw_mem % byte_alignment);
    } else {
      aligned_mem = raw_mem;
    }
    memset(raw_mem, 0, raw_memsize);
  } // end if (!raw_mem)
  
  data = aligned_mem;

  switch (type) {

  case '2':
    ////////////////////////////////////////////////////////////////
    //                                                            //
    // ASCII Greyscale                                            //
    //                                                            //
    ////////////////////////////////////////////////////////////////

    // For each row ... 
    for (int i=0; i < in_rows; i++) {
      // ... copy the valid image data, ...
      for (int j=0; j < in_cols; j++) {
	fscanf(fd, "%d", &tmp_int);
	data[in_bytes_per_line * i + j] = tmp_int;
      }
      // ... and pad the end for good alignment.
      for (int j=in_cols; j < in_bytes_per_line; j++) 
	data[in_bytes_per_line * i + j] = 0;
    }
    break;


  case '3':
    ////////////////////////////////////////////////////////////////
    //                                                            //
    // ASCII RGB-24                                               //
    //                                                            //
    ////////////////////////////////////////////////////////////////

    // For each row...
    for (int i=0; i < in_rows; i++) {
      // ... copy the valid image data ...
      for (int j=0; j < in_cols*3; j+=3) {
	int k = in_bytes_per_line * i + j;
	fscanf(fd, "%d %d %d", &tmp_int1, &tmp_int2, &tmp_int3);
	data[k  ]   = (SPU_u8) tmp_int1;
	data[k+1]   = (SPU_u8) tmp_int2;
	data[k+2]   = (SPU_u8) tmp_int3;
      }
      // ... and pad the end for good alignment.
      for (int j=in_cols*3; j < in_bytes_per_line; j++)
      	data[in_bytes_per_line * i + j] = 0;
    }
    break;


  case '5':
    ////////////////////////////////////////////////////////////////
    //                                                            //
    // Binary greyscale                                           //
    //                                                            //
    ////////////////////////////////////////////////////////////////

    // For each row...
    for (int i=0; i < in_rows; i++) {
      // ... copy in the valid image data (one row at a time).
      unsigned char *dest = data + (in_bytes_per_line * i);
      fread((void*) dest, sizeof(SPU_u8), in_cols, fd);
    }
    break;


  case '6':
    ////////////////////////////////////////////////////////////////
    //                                                            //
    // Binary 24-bit color                                        //
    //                                                            //
    ////////////////////////////////////////////////////////////////

    // For each row...
    for (int i=0; i < in_rows; i++) {
      // ... copy in the valid image data (one row at a time).
      unsigned char *dest = data + (in_bytes_per_line * i);
      fread((void*) dest, sizeof(SPU_u8), in_cols * 3, fd);
      
      // Why do we need to do this? (Otherwise, raw doesn't work --- why?)
      //      for (int j=0; j < in_bytes_per_line; j += 3) {
      //      	SPU_u8 tmp = dest[j];
      //      	dest[j]    = dest[j+2];
      //      	dest[j+2]  = tmp;
      //}
    }
    break;

  default:
    return(false);
    break;
  }
  fclose(fd);
  return(true);
}



