#include <SPU-Toolbox/color-utils.h>

#if SPU_HAVE_LIBIPL == 1
static char* ipl_colorModel_str_GRAY;
static char* ipl_colorModel_str_RGB;  
static char* ipl_colorModel_str_HLS;

static char* ipl_channelSeq_str_G;
static char* ipl_channelSeq_str_GRAY;
static char* ipl_channelSeq_str_BGR;
static char* ipl_channelSeq_str_BGRA;
static char* ipl_channelSeq_str_RGB;
static char* ipl_channelSeq_str_RGBA;
static char* ipl_channelSeq_str_HSV;
static char* ipl_channelSeq_str_HLS;
static char* ipl_channelSeq_str_XYZ;
static char* ipl_channelSeq_str_YUV;
static char* ipl_channelSeq_str_YCr;
static char* ipl_channelSeq_str_YCC;
static char* ipl_channelSeq_str_LUV;
#endif

bool color_utils_init_statics()
{
  if (color_utils_static_initialization_hack)
    return(true);

#if SPU_HAVE_LIBIPL == 1
  ipl_colorModel_str_GRAY = "GRAY";
  ipl_colorModel_str_RGB  = "RGB";
  ipl_colorModel_str_HLS  = "HLS";

  ipl_channelSeq_str_G    = "G";
  ipl_channelSeq_str_GRAY = "GRAY";
  ipl_channelSeq_str_BGR  = "BGR";
  ipl_channelSeq_str_BGRA = "BGRA";
  ipl_channelSeq_str_RGB  = "RGB";
  ipl_channelSeq_str_RGBA = "RGBA";
  ipl_channelSeq_str_HSV  = "HSV";
  ipl_channelSeq_str_HLS  = "HLS";
  ipl_channelSeq_str_XYZ  = "XYZ";
  ipl_channelSeq_str_YUV  = "YUV";
  ipl_channelSeq_str_YCr  = "YCr";
  ipl_channelSeq_str_YCC  = "YCC";
  ipl_channelSeq_str_LUV  = "LUV";
#endif

  return(true);
}

int bytes_per_pixel(const SPU_Color_Format& cf) 
{
  int rv = 0;
  switch (cf) {

  case SPU_Color_Format_Grey8:         
    rv = 1; 
    break;

  case SPU_Color_Format_Grey16: 
  case SPU_Color_Format_RGB1555:
  case SPU_Color_Format_RGB5155:
  case SPU_Color_Format_RGB5515:
  case SPU_Color_Format_RGB5551:
  case SPU_Color_Format_BGR1555:
  case SPU_Color_Format_BGR5155:
  case SPU_Color_Format_BGR5515:
  case SPU_Color_Format_BGR5551:
  case SPU_Color_Format_RGB565:
  case SPU_Color_Format_BGR565:
    rv = 2;
    break;
    
  case SPU_Color_Format_RGB24:
  case SPU_Color_Format_BGR24:
    rv = 3;
    break;

  case SPU_Color_Format_Grey32:
  case SPU_Color_Format_MSBPadded_RGB32:
  case SPU_Color_Format_LSBPadded_RGB32:
  case SPU_Color_Format_MSBPadded_BGR32:
  case SPU_Color_Format_LSBPadded_BGR32:
    rv = 4;
    break;

    // rv = 2, Thomas Howard 9/28/2004
  case SPU_Color_Format_YUV422_Planar:
  case SPU_Color_Format_YUV422_Packed:
    rv = 2;
    break;

  case SPU_Color_Format_COUNT:
  case SPU_Color_Format_Unknown:
    rv = 0;
    break;

  }
  return rv;
}

const char* const 
SPU_Color_Format_to_str(const SPU_Color_Format& cf)
{
  return(short_SPU_Color_Format_strings[cf]);
}

const char* const 
SPU_Color_Format_to_long_str(const SPU_Color_Format& cf)
{
  return(long_SPU_Color_Format_strings[cf]);
}

SPU_Color_Format
str_to_SPU_Color_Format(const char* const& str)
{
  if (strcasecmp(str, "grey8") == 0 ||
      strcasecmp(str, "gray8") == 0)
    return SPU_Color_Format_Grey8;
  
  else if (strcasecmp(str, "grey16") == 0 ||
	   strcasecmp(str, "gray16") == 0)
    return SPU_Color_Format_Grey16;

  else if (strcasecmp(str, "grey32") == 0 ||
      strcasecmp(str, "gray32") == 0)
    return SPU_Color_Format_Grey32;

  else if (strcasecmp(str, "rgb1555") == 0)
    return SPU_Color_Format_RGB1555;

  else if (strcasecmp(str, "rgb5155") == 0)
    return SPU_Color_Format_RGB5155;

  else if (strcasecmp(str, "rgb5515") == 0)
    return SPU_Color_Format_RGB5515;

  else if (strcasecmp(str, "rgb5551") == 0)
    return SPU_Color_Format_RGB5551;

  else if (strcasecmp(str, "bgr1555") == 0)
    return SPU_Color_Format_BGR1555;

  else if (strcasecmp(str, "bgr5155") == 0)
    return SPU_Color_Format_BGR5155;

  else if (strcasecmp(str, "bgr5515") == 0)
    return SPU_Color_Format_BGR5515;

  else if (strcasecmp(str, "bgr5551") == 0)
    return SPU_Color_Format_BGR5551;

  else if (strcasecmp(str, "rgb565") == 0 ||
      strcasecmp(str, "rgb16") == 0)
    return SPU_Color_Format_RGB565;

  else if (strcasecmp(str, "bgr565") == 0 ||
      strcasecmp(str, "bgr16") == 0)
    return SPU_Color_Format_BGR565;

  else if (strcasecmp(str, "rgb888") == 0 ||
      strcasecmp(str, "rgb24") == 0)
    return SPU_Color_Format_RGB24;

  else if (strcasecmp(str, "bgr888") == 0 ||
      strcasecmp(str, "bgr24") == 0)
    return SPU_Color_Format_BGR24;

  else if (strcasecmp(str, "msb_rgb32") == 0)
    return SPU_Color_Format_MSBPadded_RGB32;
  
  else if (strcasecmp(str, "lsb_rgb32") == 0)
    return SPU_Color_Format_LSBPadded_RGB32;

  else if (strcasecmp(str, "msb_bgr32") == 0)
    return SPU_Color_Format_MSBPadded_BGR32;
  
  else if (strcasecmp(str, "lsb_bgr32") == 0)
    return SPU_Color_Format_LSBPadded_BGR32;

  // case "yuv422_packed" instituted for SONY DFW-VL500 camera, Thomas Howard 9/28/2004
  else if (strcasecmp(str, "yuv422_packed") == 0){
    cout << "Value is " << SPU_Color_Format_YUV422_Packed << endl;
    return SPU_Color_Format_YUV422_Packed;
  }

  else {
    cerr << "Warning: Color format \"" << str << "\" unknown." << endl;
    return(SPU_Color_Format_Unknown);
  }
}

#if SPU_HAVE_LIBIPL == 1

// Handly little macro for simplifying the switch statement
#define IPL_TO_SPU_CF_CASE(ICF, CHAN, DEPTH, COLMOD, CHANSEQ, DATAORD) \
 case SPU_Color_Format_ ## ICF : \
  nChannels = CHAN ;\
  depth = IPL_DEPTH_ ## DEPTH ; \
  colorModel = COLMOD ; \
  channelSeq = CHANSEQ ; \
  dataOrder = IPL_DATA_ORDER_ ## DATAORD ; \
  return(true); \
  break;

bool
SPU_Color_Format_to_ipl_Color_Format(const SPU_Color_Format cf,
				     int& nChannels,
				     int& depth,
				     char*& colorModel,
				     char*& channelSeq,
				     int& dataOrder)
{

  switch(cf) {

  IPL_TO_SPU_CF_CASE( Grey8,             1,  8U, "GRAY", "GRAY", PIXEL);
  IPL_TO_SPU_CF_CASE( Grey16,            1, 16U, "GRAY", "GRAY", PIXEL);
  IPL_TO_SPU_CF_CASE( RGB24,             3,  8U, "RGB",  "RGB",  PIXEL);
  IPL_TO_SPU_CF_CASE( BGR24,             3,  8U, "RGB",  "BGR",  PIXEL);
  IPL_TO_SPU_CF_CASE( LSBPadded_RGB32,   4,  8U, "RGB",  "RGBA", PIXEL);
  IPL_TO_SPU_CF_CASE( LSBPadded_BGR32,   4,  8U, "RGB",  "BGRA", PIXEL);

  case SPU_Color_Format_Grey32:
  case SPU_Color_Format_RGB1555:
  case SPU_Color_Format_RGB5155:
  case SPU_Color_Format_RGB5515:
  case SPU_Color_Format_RGB5551:
  case SPU_Color_Format_BGR1555:
  case SPU_Color_Format_BGR5155:
  case SPU_Color_Format_BGR5515:
  case SPU_Color_Format_BGR5551:
  case SPU_Color_Format_RGB565:
  case SPU_Color_Format_BGR565:
  case SPU_Color_Format_MSBPadded_RGB32:
  case SPU_Color_Format_MSBPadded_BGR32:
  case SPU_Color_Format_YUV422_Planar:
  case SPU_Color_Format_YUV422_Packed:
  case SPU_Color_Format_COUNT:
  case SPU_Color_Format_Unknown:
    return(false);
  }
  return(false);
}

#define SPU_TO_IPL_CF_IF(C, D, CM, CS, DO, OCF) \
 if (nChannels == C && \
     depth == IPL_DEPTH_ ## D && \
     strcasecmp(colorModel, CM) == 0 && \
     strcasecmp(channelSeq, CS) == 0 && \
     dataOrder == IPL_DATA_ORDER_ ## DO ) \
  return(SPU_Color_Format_ ## OCF )
  
     


SPU_Color_Format
ipl_Color_Format_to_SPU_Color_Format(int nChannels,
				     int depth, 
				     char* colorModel,
				     char* channelSeq,
				     int dataOrder)
{
  SPU_TO_IPL_CF_IF(1, 8U, "GRAY", "GRAY", PIXEL, Grey8 );
  SPU_TO_IPL_CF_IF(1,16U, "GRAY", "GRAY", PIXEL, Grey16);
  SPU_TO_IPL_CF_IF(3, 8U,  "RGB",  "RGB", PIXEL, RGB24 );
  SPU_TO_IPL_CF_IF(3, 8U,  "RGB",  "BGR", PIXEL, BGR24 );
  SPU_TO_IPL_CF_IF(4, 8U,  "RGB", "RGBA", PIXEL, LSBPadded_RGB32);
  SPU_TO_IPL_CF_IF(4, 8U,  "RGB", "BGRA", PIXEL, LSBPadded_BGR32);

  return(SPU_Color_Format_Unknown);
}

#endif
