//
// $Id: color-convert.cc,v 1.2 2004/10/09 15:14:59 jayg Exp $
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file uses PERCEPS style comments to facilitate automatic
// documentation generation. More information on PERCEPS can be
// found at "http://starship.python.net/crew/tbryan/PERCEPS".
//
// Author: Ross J. Micheals
//         rmicheals@lehigh.edu
//
// (c) 1999-2000 Ross J. Micheals

#include <SPU-Toolbox/color-convert.h>

static SPU_color_convert_FP 
SPU_colconv_table[SPU_Color_Format_COUNT][SPU_Color_Format_COUNT];

bool SPU_init_colconv_called = SPU_init_colconv();

bool
SPU_init_colconv(void) 
{

  SPU_color_convert_FP* t;

  // Table is arranged as [icf][ocf].
  
  t = SPU_colconv_table[ SPU_Color_Format_Grey8 ];
  t[ SPU_Color_Format_Grey8           ] = color_convert_raw;
  t[ SPU_Color_Format_RGB565          ] = Grey8_to_RGB565;
  t[ SPU_Color_Format_BGR565          ] = Grey8_to_RGB565;
  t[ SPU_Color_Format_RGB24           ] = Grey8_to_RGB24;
  t[ SPU_Color_Format_BGR24           ] = Grey8_to_RGB24;
  t[ SPU_Color_Format_MSBPadded_RGB32 ] = Grey8_to_MSBPadded_RGB32;
  t[ SPU_Color_Format_LSBPadded_RGB32 ] = Grey8_to_MSBPadded_RGB32;
  t[ SPU_Color_Format_MSBPadded_BGR32 ] = Grey8_to_MSBPadded_RGB32;
  t[ SPU_Color_Format_LSBPadded_BGR32 ] = Grey8_to_MSBPadded_RGB32;

  t = SPU_colconv_table[ SPU_Color_Format_RGB1555 ];
  t[ SPU_Color_Format_RGB565          ] = RGB1555_to_RGB565;
		       

  t = SPU_colconv_table[ SPU_Color_Format_RGB565 ];
  t[ SPU_Color_Format_RGB565          ] = color_convert_raw;

  t = SPU_colconv_table[ SPU_Color_Format_RGB24  ];
  t[ SPU_Color_Format_Grey8           ] = RGB24_to_Grey8;
  t[ SPU_Color_Format_RGB565          ] = RGB24_to_RGB565;
  t[ SPU_Color_Format_BGR565          ] = RGB24_to_BGR565;
  t[ SPU_Color_Format_RGB24           ] = color_convert_raw;
  t[ SPU_Color_Format_BGR24           ] = RGB24_to_BGR24;
  t[ SPU_Color_Format_MSBPadded_RGB32 ] = RGB24_to_MSBPadded_RGB32;
  t[ SPU_Color_Format_LSBPadded_RGB32 ] = RGB24_to_LSBPadded_RGB32;
  t[ SPU_Color_Format_MSBPadded_BGR32 ] = RGB24_to_MSBPadded_BGR32;
  t[ SPU_Color_Format_LSBPadded_BGR32 ] = RGB24_to_LSBPadded_BGR32;

  t = SPU_colconv_table[ SPU_Color_Format_BGR24  ];
  t[ SPU_Color_Format_Grey8           ] = BGR24_to_Grey8;
  t[ SPU_Color_Format_RGB565          ] = BGR24_to_RGB565;
  t[ SPU_Color_Format_BGR565          ] = BGR24_to_BGR565;
  t[ SPU_Color_Format_RGB24           ] = BGR24_to_RGB24;
  t[ SPU_Color_Format_BGR24           ] = color_convert_raw;
  t[ SPU_Color_Format_MSBPadded_RGB32 ] = BGR24_to_MSBPadded_RGB32;
  t[ SPU_Color_Format_LSBPadded_RGB32 ] = BGR24_to_LSBPadded_RGB32;
  t[ SPU_Color_Format_MSBPadded_BGR32 ] = BGR24_to_MSBPadded_BGR32;
  t[ SPU_Color_Format_LSBPadded_BGR32 ] = BGR24_to_LSBPadded_BGR32;

  t = SPU_colconv_table[ SPU_Color_Format_MSBPadded_RGB32];
  t[ SPU_Color_Format_MSBPadded_RGB32 ] = color_convert_raw;
  t[ SPU_Color_Format_RGB565          ] = MSBPadded_RGB32_to_RGB565;

  t = SPU_colconv_table [ SPU_Color_Format_YUV422_Packed ];
  t[ SPU_Color_Format_Grey8           ] = YUV422_Packed_to_Grey8;
  //  t[ SPU_Color_Format_RGB565          ] = YUV422_Packed_to_RGB565;
  //  t[ SPU_Color_Format_BGR565          ] = YUV422_Packed_to_BGR565;
  //  t[ SPU_Color_Format_RGB24           ] = YUV422_Packed_to_RGB24;
  //  t[ SPU_Color_Format_BGR24           ] = YUV422_Packed_to_BGR24;
  t[ SPU_Color_Format_YUV422_Packed   ] = color_convert_raw;
  //  t[ SPU_Color_Format_MSBPadded_RGB32 ] = YUV422_Packed_to_MSBPadded_RGB32;
  //  t[ SPU_Color_Format_LSBPadded_RGB32 ] = YUV422_Packed_to_LSBPadded_RGB32;
  //  t[ SPU_Color_Format_MSBPadded_BGR32 ] = YUV422_Packed_to_MSBPadded_BGR32;
  t[ SPU_Color_Format_LSBPadded_BGR32 ] = YUV422_Packed_to_LSBPadded_BGR32;

  return(true);
}

SPU_color_convert_FP
SPU_get_colconv_fp(const SPU_Color_Format& icf,
		   const SPU_Color_Format& ocf) 
{
  return SPU_colconv_table[icf][ocf];
}

// Thomas Howard's Modified Code (START) 9/28/2004

// Chris Urmson's YUV2RGB Function
  
//   This function takes the 4:2:2 YUV (luminance & chrominance) components of the signal coming from the camera.

#define YUV2RGB(y, u, v, r, g, b)\
  r = y + ((v*1436) >>10);\
  g = y - ((u*352 + v*731) >> 10);\
  b = y + ((u*1814) >> 10);\
  r = r < 0 ? 0 : r;\
  g = g < 0 ? 0 : g;\
  b = b < 0 ? 0 : b;\
  r = r > 255 ? 255 : r;\
  g = g > 255 ? 255 : g;\
  b = b > 255 ? 255 : b

// This code converts a 4:2:2 YUV array into a 32-bit BGR image for viewing in X-Windows.

void
YUV422_Packed_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl,
                                 SPU_u8* to_p, int to_bpl,
		                 int width, int height) 
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  register int y0, y1, u, v;
  register int r, g, b;
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width*2; i+=4) {
      y0 = (unsigned char) src_p[i+1];
      y1  = (unsigned char) src_p[i+3];    
      u  = (unsigned char) src_p[i+0] - 128;   
      v  = (unsigned char) src_p[i+2] - 128; 
 
      // for four bytes of YUV data, two pixels (eight bytes) can be constructed.

      YUV2RGB (y0, u, v, r, g, b); // the first pixel holds the first luminance value and the two chrominance values
      *dest_p++ = b; // the first byte holds the b component in BGR format.
      *dest_p++ = g; // the second byte holds the g component in BGR format.
      *dest_p++ = r; // the third byte holds the r component in BGR format.
      *dest_p++ = 0; // the last byte holds 0.
      YUV2RGB (y1, u, v, r, g, b); // the second pixel holds the second luminance value and the two chrominance value
      *dest_p++ = b; // same comments as above for all cases.
      *dest_p++ = g;
      *dest_p++ = r;
      *dest_p++ = 0;
    }
  }
 return;
}

// This code converts a 4:2:2 YUV image into a 8-bit Greyscale image for viewing in X-Windows.
// Modifed by Kai 09/28/2004

void 
YUV422_Packed_to_Grey8(SPU_u8* from_p, int from_bpl,
                                 SPU_u8* to_p, int to_bpl,
		                 int width, int height) 
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  register int y0, y1;
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width*2; i+=4) {
      // Byte format of YUV422Packed: UYVY
      y0 = (unsigned char) src_p[i+1];
      y1  = (unsigned char) src_p[i+3];
      // Y component represents the luminance of a pixel, i.e. the greyscale value
      *dest_p++ = y0;
      *dest_p++ = y1;
    }
  }
 return;
}

// Thomas Howard's Modified Code (END) 9/28/2004

void 
color_convert_raw(SPU_u8* from_p, int from_bpl,
		  SPU_u8* to_p, int to_bpl,
		  int width, int height) 
{
  if (from_bpl == to_bpl)
    memcpy(to_p, from_p, height * from_bpl);
  
  else {
    register SPU_u8* dest_p;
    register SPU_u8* src_p;
    int memcpy_bpl = from_bpl < to_bpl? from_bpl:to_bpl;
    for (int i=0; i < height; i++) {
      src_p  = from_p + i * from_bpl;
      dest_p = to_p   + i * to_bpl;
      memcpy(dest_p, src_p, memcpy_bpl);
    }
  }

  return;
}

void 
Grey8_to_RGB565(SPU_u8* from_p, int from_bpl, 
		SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16 y5;
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  for (int j=0; j < height ; j++) {
    dest_p = dest_base_p + j * to_bpl / sizeof(SPU_u16);
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width; i++) {
      y5 = (*src_p++ >> 3) & (0x1f);
      *dest_p++ = (y5 << 11) | (y5 << 6) | y5;
    }
  }
  return;
}

void 
Grey8_to_RGB24(SPU_u8* from_p, int from_bpl, 
		      SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width; i++) {
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i];
    }
  }
 return;
}

void 
Grey8_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
		      SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width; i++) {
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i];
    }
  }
 return;
}

void 
RGB565_to_Grey8(SPU_u8* from_p, int from_bpl, 
		SPU_u8* to_p, int to_bpl, int width, int height) {

  cerr << "Color conversion RGB565 to Grey8 unsupported." << endl;
 return;
}

void 
RGB565_to_BGR_RAW_565(SPU_u8* from_p, int from_bpl, 
		      SPU_u8* to_p, int to_bpl, int width, int height) {
  cerr << "Color conversion RGB565 to BGR unsupported." << endl;
 return;
}

void 
RGB565_to_RGB24(SPU_u8* from_p, int from_bpl, 
		SPU_u8* to_p, int to_bpl, int width, int height) {

  cerr << "Color conversion RGB565 to RGB24 unsupported." << endl;
 return;
}

void 
RGB565_to_BGR24(SPU_u8* from_p, int from_bpl, 
		SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion RGB565 to BGR24 unsupported." << endl;
 return;
}

void 
RGB565_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion RGB565 to MSB Padded RGB32 unsupported." << endl;
 return;
}

void 
RGB565_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion RGB565 to LSB Padded RGB32 unsupported." << endl;
 return;
}

void 
RGB565_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion RGB565 to MSB Padded BGR32 unsupported." << endl;
 return;
}

void 
RGB565_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion RGB565 to LSB padded BGR32 unsupported." << endl;
 return;
}

void 
BGR_RAW_565_to_Grey8     (SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion BGR565 to Grey8 unsupported." << endl;
 return;
}

void 
BGR_RAW_565_to_RGB565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  cerr << "Color conversion BGR565 to RGB565 unsupported." << endl;
 return;
}

void 
BGR_RAW_565_to_BGR_RAW_565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR_RAW_565_to_RGB24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR_RAW_565_to_BGR24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR_RAW_565_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR_RAW_565_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
BGR_RAW_565_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
BGR_RAW_565_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
RGB24_to_Grey8     (SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
RGB24_to_RGB565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16 r, g, b;
  
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + (j * to_bpl / sizeof(SPU_u16));
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      r = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      b = static_cast<SPU_u16>(*src_p++);
      *dest_p++ = 
	((r << 8) & 0xf800) | 
	((g << 3) & 0x07e0) | 
	((b >> 3) & 0x001f);
    }
  }
 return;
}

void 
RGB24_to_BGR565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {
  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16 r, g, b;
  
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + (j * to_bpl / sizeof(SPU_u16));
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
#ifdef SPU_LITTLE_ENDIAN
      r = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      b = static_cast<SPU_u16>(*src_p++);
#else
      b = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      r = static_cast<SPU_u16>(*src_p++);
#endif
      *dest_p++ = 
	((b << 8) & 0xf800) | 
	((g << 3) & 0x07e0) | 
	((r >> 3) & 0x001f);
    }
  }
 return;
}

void 
RGB24_to_BGR24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i+2];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i];
    }
  }
 return;
}

void 
RGB24_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {
  //  cerr << "!";
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i+2];
      *dest_p++ = 0;
    }
  }
  return;
}

void 
RGB24_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i+2];
      *dest_p++ = 0;
    }
  }
  return;
}

void 
RGB24_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = 0;
      *dest_p++ = src_p[i+2];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i];
    }
  }
  return;
}

void 
RGB24_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i+2];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i];
      *dest_p++ = 0;
    }
  }
 return;
}

void 
BGR24_to_Grey8     (SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR24_to_RGB565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16 r, g, b;
  
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + (j * to_bpl / sizeof(SPU_u16));
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      b = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      r = static_cast<SPU_u16>(*src_p++);
      *dest_p++ = 
	((r << 8) & 0xf800) | 
	((g << 3) & 0x07e0) | 
	((b >> 3) & 0x001f);
    }
  }
  return;
}

void 
BGR24_to_BGR565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {
  

  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16 r, g, b;
  
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + (j * to_bpl / sizeof(SPU_u16));
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      b = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      r = static_cast<SPU_u16>(*src_p++);
      *dest_p++ = 
	((b << 8) & 0xf800) | 
	((g << 3) & 0x07e0) | 
	((r >> 3) & 0x001f);
    }
  }
 return;
}

void 
BGR24_to_RGB24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i+2];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i];
    }
  }
 return;
}

void 
BGR24_to_BGR24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
BGR24_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
BGR24_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  
 return;
}

void 
BGR24_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++;
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i+2];
    }
  }
 return;
}

void 
BGR24_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 3; i+=3) {
      *dest_p++ = src_p[i];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i+2];
      *dest_p++;
    }
  }
 return;
}

void 
MSBPadded_RGB32_to_Grey8     (SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
MSBPadded_RGB32_to_RGB565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height) 
{

  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u8  *src_p,  *src_base_p;
  register SPU_u16  r, g, b;
  
  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = from_p;
  
  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + (j * to_bpl / sizeof(SPU_u16));
    src_p  = src_base_p  + (j * from_bpl);
    for (int i=0; i < width * 4; i+=4) {

      b = static_cast<SPU_u16>(*src_p++);
      g = static_cast<SPU_u16>(*src_p++);
      r = static_cast<SPU_u16>(*src_p++);
      src_p++;
      
      *dest_p++ = 
	((r << 8) & 0xf800) | 
	((g << 3) & 0x07e0) | 
	((b >> 3) & 0x001f);
    }
  }
 return;
}

void 
MSBPadded_RGB32_to_BGR565(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl, int width, int height)
{

 return;
}

void 
MSBPadded_RGB32_to_RGB24(SPU_u8* from_p, int from_bpl, 
			   SPU_u8* to_p, int to_bpl,
			   int width, int height)
{
  register SPU_u8 *src_p, *src_base_p;
  register SPU_u8 *dest_p, *dest_base_p;
  dest_p = dest_base_p = to_p;
  src_p  = src_base_p  = from_p;

  for (int j=0; j < height; j++) {
    dest_p = dest_base_p + j * to_bpl;
    src_p  = src_base_p  + j * from_bpl;
    for (int i=0; i < width * 4; i+=4) {
      *dest_p++ = src_p[i+2];
      *dest_p++ = src_p[i+1];
      *dest_p++ = src_p[i];
    }
  }
 return;
}

void 
MSBPadded_RGB32_to_BGR24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height)
{

 return;
}

void 
MSBPadded_RGB32_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_RGB32_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_RGB32_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
				   SPU_u8* to_p, int to_bpl, 
				   int width, int height) 
{

 return;
}

void 
MSBPadded_RGB32_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
				   SPU_u8* to_p, int to_bpl, 
				   int width, int height) 
{

 return;
}

void 
LSBPadded_RGB32_to_Grey8(SPU_u8* from_p, int from_bpl, 
			 SPU_u8* to_p, int to_bpl, int width, int height) 
{

 return;
}

void 
LSBPadded_RGB32_to_RGB565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_BGR565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_RGB24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_BGR24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_RGB32_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_Grey8     (SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_RGB565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_BGR565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_RGB24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_BGR24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
MSBPadded_BGR32_to_LSBPadded_BGR32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_Grey8     (SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_RGB565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_BGR565(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_RGB24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_BGR24(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_MSBPadded_RGB32(SPU_u8* from_p, int from_bpl, SPU_u8* to_p, int to_bpl, int width, int height) {

 return;
}

void 
LSBPadded_BGR32_to_LSBPadded_RGB32(SPU_u8* from_p, int from_bpl, 
				   SPU_u8* to_p, int to_bpl, 
				   int width, int height) 
{

 return;
}

void 
LSBPadded_BGR32_to_MSBPadded_BGR32(SPU_u8* from_p, int from_bpl, 
				   SPU_u8* to_p, int to_bpl, 
				   int width, int height) 
{

 return;
}

void 
LSBPadded_BGR32_to_BGR_LSB_888(SPU_u8* from_p, int from_bpl, 
			       SPU_u8* to_p, int to_bpl, 
			       int width, int height) 
{

 return;
}


void 
RGB1555_to_RGB565(SPU_u8* from_p, int from_bpl, 
		  SPU_u8* to_p, int to_bpl, 
		  int width, int height) 
{
  register SPU_u16 *dest_p, *dest_base_p;
  register SPU_u16 *src_p,  *src_base_p;
  register SPU_u16 r, g, b;
  register SPU_u16 y;

  dest_p = dest_base_p = (SPU_u16*) to_p;
  src_p  = src_base_p  = (SPU_u16*) from_p;

  for (int j=0; j < height ; j++) {
    dest_p = dest_base_p + j * to_bpl / sizeof(SPU_u16);
    src_p  = src_base_p  + j * from_bpl / sizeof(SPU_u16);
    for (int i=0; i < width; i++) {

      r = (*src_p) << 1;
      g = (*src_p) << 1;
      b = (*src_p);
      
      y = (r & 0xf800) | (g & 0x07e0) | (b & 0x001f);

      *dest_p++ = y;
      ++src_p;
    }
  }
  return;
}

