#include <stdio.h>
#include <pthread.h>
#include <SPU-Toolbox/XIH.h>

#include <ImgDisplay/ImgDisplay.h>
#include <ConfigSource/ConfigSource.h>

#include <utils/Output.h>
#include <utils/Input.h>
#include <utils/Generator.h>
#include <utils/Interface.h>
#include <utils/Type.h>
#include <utils/ConfigFile.h>
#include <utils/List.h>
#include <utils/Dict.h>

static const char* XIH_color_IDs[ImgDisplay::LastColor] = {
  "black", "white", "red", "orange", "yellow", "green", "blue", "purple",
  "pink", "light blue"
};

//////////////////////////////
void stop_callback(void) {
  XIH::stop_all();
}

class XIHWindow;

#define FONT_SPEC "-misc-fixed-medium-r-normal-*-%d-*-*-*-*-*-*-*"

class XIHDisplay: public ImgDisplay
{
public:
  XIHDisplay(int debug);
  virtual ~XIHDisplay();

  virtual Window* createWindow(const char* title, int width, int height,
                               ImgDisplay::ColorType);
  virtual void destroyWindow(Window*);
  virtual void destroyAllWindows();

private:
  int _debug;
  utils::List<XIHWindow> _windows;
  int _numWin;
};

class XIHWindow : public ImgDisplay::Window {
public:
  XIHWindow(ImgDisplay* display, int width, int height, XIH* xih);
  virtual ~XIHWindow();

  virtual int width() const { return _xih->width(); }
  virtual int height() const { return _xih->height(); };
  virtual ImgDisplay::ColorType colorType() const { return _type; }
  virtual int bytesPerPixel() const { return _xih->bytes_per_pixel(); }
  virtual int bytesPerLine() const { return _xih->get_bytes_per_line(); }

  virtual void pixel(int x, int y, ImgDisplay::Color colorID);
  virtual void fill(unsigned char* src);
  virtual void clear();
  virtual void line(int x1, int y1, int x2, int y2,
                        ImgDisplay::Color color);
  virtual void fillBox(int x1, int y1, int x2, int y2,
                       ImgDisplay::Color color);
  virtual void circle(int x, int y, double radius, ImgDisplay::Color color);
  virtual bool begin();
  virtual void end();
  virtual void setDefaultColor(ImgDisplay::Color color);
  virtual void setLineWidth(unsigned int width);

  void setColorType(ImgDisplay::ColorType type) { _type = type; }

  virtual void setSelectable(bool selectable);
  virtual void acquireSelectionLock();
  virtual void releaseSelectionLock();
  virtual bool waitForSelection(int& x, int& y, unsigned int& modifiers);

private:
  void set_color(ImgDisplay::Color color) {
    if (_last_color != color) {
      _xih->set_draw_color(XIH_color_IDs[color]);
      _last_color = color;
    }
  }

private:
  XIH* _xih;
  ImgDisplay::Color _last_color;
  ImgDisplay::ColorType _type;
  pthread_mutex_t _select_mutex;
};

XIHDisplay::XIHDisplay(int debug)
{
  _debug=debug;
  _numWin = 0;
}

XIHDisplay::~XIHDisplay()
{
  destroyAllWindows();
  XIH::wait();
}

ImgDisplay::Window*
XIHDisplay::createWindow(const char* title, int width, int height,
                         ImgDisplay::ColorType src_type)
{
  Display* disp;

  const char* fmt_string = ImgDisplay::colorTypeToString(src_type);
  if (!fmt_string) {
    fprintf(stderr, "Illegal color type %d\n", src_type);
    return NULL;
  }
		
  //// Check screen depth (aka dest_depth)
  const char* disp_name = getenv("DISPLAY");
  if ( (disp = XOpenDisplay(getenv("DISPLAY"))) == NULL ) {
    fprintf(stderr, "Cannot connect to X server.\n");
    return NULL;
  }
  if (_debug>=1)
    printf("Opened X display.\n");
  int screen_bpp = DefaultDepth(disp, DefaultScreen(disp));
  XCloseDisplay(disp);
  if (screen_bpp==15) {
    fprintf(stderr, "XIH does not support screen depth of 15.\n");
    return NULL;
  }

  //// Add a new window
  if (_debug>=1)
    printf("Beginning XIH initialization...\n");

  XIH* xih = new XIH();
  if (xih==NULL) {
    fprintf(stderr, "Failed to create new XIH class instance.\n");
    return NULL;
  }
  // printf("Creating %d %d\n", width, height);
  xih->init(width, height, fmt_string, title, disp_name);

  XIHWindow* win = new XIHWindow(this, width, height, xih);
  win->setColorType(src_type);
  _windows.append(win);

  if (_numWin==0)
    {
      XIH_set_finished_callback(stop_callback);
      if (xih->start()==false) {
        delete xih;
        fprintf(stderr, "Could not start XIH.\n");
        return NULL;
      }
    }
  _numWin++;

  return win;
}

void XIHDisplay::destroyWindow(ImgDisplay::Window* win)
{
  _windows.remove((XIHWindow*) win);
  delete win;
  _numWin--;

  if (_numWin==0) 
    XIH::stop_all();
}

void XIHDisplay::destroyAllWindows(void)
{
  while (XIHWindow* win = _windows.pop()) 
    delete win;
  if (_numWin != 0) {
    _numWin=0;
    XIH::stop_all();
  }
}

XIHWindow::XIHWindow(ImgDisplay* disp, int width, int height, XIH* xih)
  : ImgDisplay::Window(disp, width, height)
{
  pthread_mutex_init(&_select_mutex, NULL);
  _xih = xih;
  if (_xih->lock_and_draw())
    {
      _xih->set_line_width(1);
      _xih->unlock();
    }
  setLineWidth(1);
  setDefaultColor(ImgDisplay::White);
  _last_color = ImgDisplay::UnknownColor;
}

XIHWindow::~XIHWindow()
{
  _xih->close();
  delete _xih;
  pthread_mutex_destroy(&_select_mutex);
}

void XIHWindow::setDefaultColor(ImgDisplay::Color color)

{
  ImgDisplay::Window::setDefaultColor(color);
  set_color(color);
}
	
void XIHWindow::setLineWidth(unsigned int thickness)

{
  ImgDisplay::Window::setLineWidth(thickness);
  _xih->set_line_width(thickness);
}

void XIHWindow::fill(unsigned char* src)
{
  if (!src)
    fprintf(stderr, "XIHWindow::fill: NULL pointer passed as src!\n");
  else
    _xih->fill(src);
}
	
void XIHWindow::pixel(int x, int y, ImgDisplay::Color color)
{
  set_color(color);
  _xih->draw_point(x, y);
}

void XIHWindow::line(int x1, int y1, int x2, int y2,
                         ImgDisplay::Color color)
{
  set_color(color);
  _xih->draw_line(x1, y1, x2, y2);
}

bool XIHWindow::begin()
{
  if (_xih->lock_and_draw()) {
    return true;
  } else {
    fprintf(stderr,
            "XIHWindow::begin: lock_and_draw failed. Could not lock window\n");
    return false;
  }
}

void XIHWindow::end()
{
  _xih->refresh();
  _xih->unlock();
}

void XIHWindow::clear()
{
  _xih->clear();
}

void XIHWindow::fillBox(int x1, int y1, int x2, int y2,
                        ImgDisplay::Color color)
{
  set_color(color);
  _xih->draw_filled_rectangle(x1, y1, x2-x1+1, y2-y1+1);
}

void XIHWindow::circle(int x, int y, double radius, ImgDisplay::Color color)
{
  set_color(color);
  _xih->draw_circle(x, y, (int) (radius+0.5));
}

static void press_button(XIH* xih, const XEvent* xev, void* data)
{
  XIHWindow* window = (XIHWindow*) data;
  unsigned int flags = ImgDisplay::NoModifiers;
  if (xih->pressed_Control(xev)) 
    flags |= ImgDisplay::Ctrl;
  if (xih->pressed_Shift(xev)) 
    flags |= ImgDisplay::Shift;
  if (xih->pressed_Mod1(xev) || xih->pressed_Mod2(xev) ||
      xih->pressed_Mod3(xev) || xih->pressed_Mod4(xev) ||
      xih->pressed_Mod5(xev))
    flags |= ImgDisplay::Alt;

  pair<int,int> pos = xih->extract_position(xev);
  window->select(pos.first, pos.second, flags);
}

void XIHWindow::setSelectable(bool selectable)
{
  ImgDisplay::Window::setSelectable(selectable);

  if (selectable) {
    _xih->register_callback(XIH::CB_ButtonPress,
                            press_button, (void*) this);
  } else
    _xih->register_callback(XIH::CB_ButtonPress,
                            NULL, (void*) this);
}

void XIHWindow::acquireSelectionLock()
{
  pthread_mutex_lock(&_select_mutex);
}

void XIHWindow::releaseSelectionLock()
{
  pthread_mutex_unlock(&_select_mutex);
}

bool XIHWindow::waitForSelection(int& x, int& y, unsigned int& modifiers)
{
  bool old_selectable = getSelectable();
  if (!old_selectable)
    setSelectable(true);
  clearSelections();
  utils::Vector<ImgDisplay::Selection> v;
  while (!popSelections(v)) 
    utils::Time::sleep(0.02);

  if (!old_selectable)
    setSelectable(false);
  ImgDisplay::Selection& s = v[v.numElems()-1];
  x = s.x;
  y = s.y;
  modifiers = s.modifiers;

  return true;
}

ImgDisplay* create_ImgDisplay_XIH(utils::Generator<ImgDisplay>*,
                                  utils::ConfigFile* params,
                                  utils::SymbolTable*)
{
  return new XIHDisplay(params->getInt("debug", 2));
}
