//
// XIH.cc
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file uses PERCEPS style comments to facilitate automatic
// documentation generation. More information on PERCEPS can be
// found at "http://starship.python.net/crew/tbryan/PERCEPS".
//
// Author: Ross J. Micheals
//         rmicheals@lehigh.edu
//
// (c) 1999-2000 Ross J. Micheals

#include <SPU-Toolbox/XIH.h>

////////////////////////////////////////////////////////////////
//                                                            //
// Statics initialization                                     //
//                                                            //
////////////////////////////////////////////////////////////////

// List of current valid XIH instances

const int XIH::sm_default_valid_list_size =  128;
const int XIH::scm_interevent_usleep_time = 3000;

std::vector<XIH*>  XIH::sm_valid_list((vector<XIH*>::size_type) 
				      sm_default_valid_list_size, 0);
std::map<SPU_u64, XIH::per_disp_info, less<SPU_u64> > XIH::sm_per_disp_map;


// Allocate static member storage.
// struct sigaction   XIH::sm_sigaction; Don't need a sig handler for me jayg
pthread_t          XIH::sm_xih_control_thread;
pthread_mutex_t    XIH::sm_critical_mutex;
pthread_mutex_t    XIH::sm_start_stop_count_mutex;
int                XIH::sm_start_stop_count;
pthread_mutex_t    XIH::sm_is_running_mutex;
pthread_cond_t     XIH::sm_is_running_cv;
SPU_PFv            XIH::sm_finished_cb;
SPU_PFvp           XIH::sm_finished_cb_with_user_data;
void*              XIH::sm_finished_cb_user_data;
bool               XIH::sm_is_running;
bool               XIH::sm_is_shutdown;
bool               XIH::sm_kill;
bool               XIH::sm_die_with_last;
int                XIH::sm_instance_count;
bool               XIH::sm_sighandler_installed;
bool               XIH::sm_called_XInitThreads; 
bool               XIH::sm_executing_callback;
XIH::Callback_Type XIH::sm_currently_executing_callback_type;
int                XIH::sm_default_x_pos;
int                XIH::sm_default_y_pos;
unsigned int       XIH::sm_default_border_width;
unsigned long      XIH::sm_default_border_value;
unsigned long      XIH::sm_default_background_value;
char*              XIH::sm_default_font_name;        
#if SPU_HAVE_XEXT_SHM == 1
bool               XIH::sm_disable_shm_for_all;
#endif
#if SPU_HAVE_XEXT_DBE == 1
bool               XIH::sm_disable_dbe_for_all;
#endif

const char* XIH::scm_Callback_Type_strings[] = 
{
  "close window",
  "expose window",
  "button press",
  "button release",
  "key press",
  "key release",
  "",
  "none"
};

bool
XIH_start()
{
  return(XIH::start());
}

bool 
XIH::start() 
{
  // The asynchronous nature of XIH makes starting up and stopping
  // quite complicated. Lots of error checking is required. :(

  // Make sure the start/stop attempts are balanced.
  //
  SPU_mutex_lock(&XIH::sm_start_stop_count_mutex);
  ++XIH::sm_start_stop_count;
  if (XIH::sm_start_stop_count != 1) {
    cerr << "XIH: Unbalanced start/stop noticied in XIH_start()." << endl;
    SPU_mutex_unlock(&XIH::sm_start_stop_count_mutex);
    return(false);
  }

  // Have we been killed already?
  //
  if (XIH::sm_kill) {
    cerr << "XIH: XIH has already been killed. Ignoring XIH_start() request." 
	 << endl;
    SPU_mutex_unlock(&XIH::sm_start_stop_count_mutex);
    return(false);
  }

  // Are we running already?
  //
  if (XIH::sm_is_running) {
    cerr << "XIH: XIH is already running. Ignoring XIH_start() request." 
	 << endl;
    SPU_mutex_unlock(&XIH::sm_start_stop_count_mutex);
    return(false);
  }

  SPU_mutex_unlock(&XIH::sm_start_stop_count_mutex);

  // Create the thread.
  //
  if (pthread_create(&XIH::sm_xih_control_thread, 0, XIH_true_start, 0)) {
    cerr << "XIH: Failed to start main control thread.\n";
    return(false);
  }

  // Wait for the thread to be created, and signal to us that it is ready.
  //
  SPU_mutex_lock(&XIH::sm_is_running_mutex);
  while (XIH::sm_is_running == false) {
    pthread_cond_wait(&XIH::sm_is_running_cv, 
		      &XIH::sm_is_running_mutex);
  }
  SPU_mutex_unlock(&XIH::sm_is_running_mutex);

  return(true);
}

bool
XIH_stop_all() 
{
  return(XIH::stop_all());
}


bool 
XIH::stop_all() 
{
  if (!XIH::sm_kill)
    XIH::sm_kill = true;
  return(XIH::sm_kill);
}

void* 
XIH_true_start(void*) 
{
  if (XIH::sm_kill) pthread_exit((void*) 1300);
  
  int last_type = 0;


  bool done = false;

  SPU_mutex_lock(&XIH::sm_is_running_mutex);
  XIH::sm_is_running = true;
  pthread_cond_signal(&XIH::sm_is_running_cv);
  SPU_mutex_unlock(&XIH::sm_is_running_mutex);

  pthread_setcanceltype  (PTHREAD_CANCEL_DISABLE, &last_type);
  while (!done) {

    // Enter critical section.
    SPU_mutex_lock(&XIH::sm_critical_mutex);

    // For each valid window 
    //
    // FIXME: This is not broken, but should be made faster. That is,
    // we should keep track of the instance with the highest index and
    // count up to that, instead of checking the whole thing. I wonder
    // how much time that would save?
    //
    for (unsigned int i=0; i < XIH::sm_valid_list.size(); i++) {
      XIH* xih_p = XIH::sm_valid_list[i]; 

      // Is there a window at all?
      if (!xih_p) continue; 

      // Handle all events for this window.
      xih_p->handle_events();
      XSync(xih_p->m_display_p, False);
      
      // Sanity check.
      if (!xih_p) continue; 

      // Should this window be destroyed?
      if (xih_p->m_destroy_me) {
	XIH::sm_valid_list[i] = 0; // Remove window from valid list.
	if (!xih_p->m_destroyed) 
	  xih_p->destroy(); // Destroy the window
      }
    }

    // Should we die?
    if (XIH::sm_kill) {
      done = true;
    }

    // Exit critical section.
    SPU_mutex_unlock(&XIH::sm_critical_mutex);

    if (XIH::sm_kill) 
      continue;

    // Wait about a video frame.
    usleep(XIH::scm_interevent_usleep_time);
  }

  // Die.
  XIH::true_stop((void*) 2);
  pthread_setcanceltype  (PTHREAD_CANCEL_ENABLE, &last_type);
  pthread_testcancel();
  pthread_exit(0);

  return(0);
}

void 
XIH::true_stop(void* data) 
{
  static int num_calls = 0;
  
  if (num_calls > 0) return;
  ++num_calls;

  // If we are not running, then we were (probably) called from the
  // signal handler.
  //
  SPU_mutex_lock(&XIH::sm_critical_mutex);

  // Call the user's XIH finished callback
  //
  if (XIH::sm_finished_cb) {
    XIH::sm_finished_cb();
    XIH::sm_finished_cb = 0;
  }
  else if (XIH::sm_finished_cb_with_user_data) {
    XIH::sm_finished_cb_with_user_data(XIH::sm_finished_cb_user_data);
    XIH::sm_finished_cb_with_user_data = 0;
    XIH::sm_finished_cb_user_data      = 0;
  }

  // Destroy each window and close X connections
  for (unsigned int i=0; i < XIH::sm_valid_list.size(); i++) {
    XIH* xih_p = XIH::sm_valid_list[i];
    if (xih_p) {
      if (!xih_p->m_destroyed) xih_p->destroy();
      XCloseDisplay(xih_p->m_display_p);
    }
  }

  XIH::sm_is_running = false;
  SPU_mutex_unlock(&XIH::sm_critical_mutex);

  return;
}

bool
XIH::lock(void)
{

  SPU_mutex_lock(&XIH::sm_critical_mutex);
  if (!XIH::sm_is_running || XIH::sm_kill) {
    SPU_mutex_unlock(&XIH::sm_critical_mutex);
    return(false);
  }
  return(true);
}


bool
XIH::unlock(void)
{
  if (SPU_mutex_unlock(&XIH::sm_critical_mutex) == 0) {
    return(true);
  }
  return(false);
}

bool
XIH_set_finished_callback(SPU_PFv spu_pfv) 
{
  return(XIH::set_finished_callback(spu_pfv));
}

bool 
XIH::set_finished_callback(SPU_PFv spu_pfv) 
{

  SPU_mutex_lock(&XIH::sm_critical_mutex);

  if (XIH::sm_is_running) {
    cerr << "XIH: Cannot set finished callback while running." << endl;
    SPU_mutex_unlock(&XIH::sm_critical_mutex);
    return(false);
  } else {
    XIH::sm_finished_cb                = spu_pfv;
    XIH::sm_finished_cb_with_user_data = 0;
  }
  SPU_mutex_unlock(&XIH::sm_critical_mutex);
  return(true);
}

bool XIH_set_finished_callback(SPU_PFvp spu_pfvp) 
{
  return(XIH::set_finished_callback(spu_pfvp));
}

bool 
XIH::set_finished_callback(SPU_PFvp spu_pfvp) 
{
  SPU_mutex_lock(&XIH::sm_critical_mutex);
  if (XIH::sm_is_running) {
    cerr << "XIH: Cannot set finished callback while running." << endl;
    SPU_mutex_unlock(&XIH::sm_critical_mutex);
    return(false);
  } else {
    XIH::sm_finished_cb                = 0;
    XIH::sm_finished_cb_with_user_data = spu_pfvp;
  }
  SPU_mutex_unlock(&XIH::sm_critical_mutex);
  return(true);
}

bool 
XIH_set_finished_callback_user_data(void* data) 
{
  return(XIH::set_finished_callback_user_data(data));
}

bool 
XIH::set_finished_callback_user_data(void* data) 
{
  SPU_mutex_lock(&XIH::sm_critical_mutex);
  if (XIH::sm_finished_cb_with_user_data == 0) {
    cerr << "XIH: Cannot set user data without appropriate callback." << endl;
    SPU_mutex_unlock(&XIH::sm_critical_mutex);
    return(false);
  }
  XIH::sm_finished_cb_user_data = data;
  SPU_mutex_unlock(&XIH::sm_critical_mutex);
  return(true);
}

void 
XIH::handle_events() 
{

  while (XPending(m_display_p)) {
    XEvent xevent;
    XNextEvent(m_display_p, &xevent);

    switch (xevent.type) {

    case ClientMessage:
      ((*m_callbacks_p)[CB_CloseWindow]).execute(&xevent);
      close();
      break;

    case Expose:
      ((*m_callbacks_p)[CB_ExposeWindow]).execute(&xevent);
      // FIXME
      put();
      refresh();
      break;

    case ButtonPress:
      ((*m_callbacks_p)[CB_ButtonPress]).execute(&xevent);
      break;

    case ButtonRelease:
      ((*m_callbacks_p)[CB_ButtonRelease]).execute(&xevent);
      break;

    case KeyPress:
      ((*m_callbacks_p)[CB_KeyPress]).execute(&xevent);
      break;

    case KeyRelease:
      ((*m_callbacks_p)[CB_KeyRelease]).execute(&xevent);
      break;

    case CreateNotify:
      break;

    case DestroyNotify:
      break;

    case UnmapNotify:
      break;

    case MappingNotify:
      XRefreshKeyboardMapping((XMappingEvent*) &xevent);
      break;

    default:
      cerr << "XIH: Recieved unexpected " << XEvent_type_to_str(xevent.type) 
	   << " event (" << xevent.type << ")." << endl;
      break;
    } // end switch
  } // end while
  return;
}

bool XIH_init_statics(void) {

  if (XIH_static_initialization_hack == true) 
    return(true);

  pthread_mutexattr_t* mutex_type_p;
#if SPU_MUTEX_LOCK_DEBUGGING == 1
  pthread_mutexattr_t  mutex_type;
  pthread_mutexattr_init(&mutex_type);

#if SPU_USE_PTHREAD_MUTEXATTR_SETKIND_NP == 1
  pthread_mutexattr_setkind_np(&mutex_type, PTHREAD_MUTEX_ERRORCHECK_NP);
#endif
#if SPU_USE_PTHREAD_MUTEXATTR_SETTYPE == 1
  pthread_mutexattr_settype(&mutex_type, PTHREAD_MUTEX_ERRORCHECK_NP);
#endif
  mutex_type_p = &mutex_type;
#else
  mutex_type_p = 0;
#endif
  pthread_mutex_init(&XIH::sm_critical_mutex,         mutex_type_p);
  pthread_mutex_init(&XIH::sm_start_stop_count_mutex, mutex_type_p);
  pthread_mutex_init(&XIH::sm_is_running_mutex,       mutex_type_p);
  
  pthread_cond_init(&XIH::sm_is_running_cv, 0);
  
  XIH::sm_start_stop_count                  = 0;

  XIH::sm_finished_cb                       = 0;
  XIH::sm_finished_cb_with_user_data        = 0;
  XIH::sm_finished_cb_user_data             = 0;

  // Main control thread state;
  XIH::sm_is_running                        = false;
  XIH::sm_is_shutdown                       = false;
  XIH::sm_kill                              = false;
  XIH::sm_die_with_last                     = true;
  XIH::sm_instance_count                    = 0;
  XIH::sm_sighandler_installed              = false;
  XIH::sm_called_XInitThreads               = false;
  XIH::sm_currently_executing_callback_type = XIH::CB_None;
  
  XIH::sm_default_x_pos                     = 5;
  XIH::sm_default_y_pos                     = 5;
  XIH::sm_default_border_width              = 2;
  XIH::sm_default_background_value          = 0;
  XIH::sm_default_font_name = "-*-lucida-bold-r-*-*-18-*-*-*-*-*-*-*";

#if SPU_HAVE_XEXT_SHM == 1
  XIH::sm_disable_shm_for_all              = false;
#endif

#if SPU_HAVE_XEXT_DBE == 1
  XIH::sm_disable_dbe_for_all              = false;
#endif

  for (unsigned int i=0; i < XIH::sm_valid_list.size(); i++) {
    XIH::sm_valid_list[i] = 0;
  }

  return(true);
}

XIH::XIH() 
{
  /* Don't need signal handler here for me - jayg
  if (!sm_sighandler_installed) {
    sm_sigaction.sa_handler  = XIH_sigint_handler;
    sm_sigaction.sa_flags    = SA_RESTART;
    sm_sigaction.sa_restorer = 0;
    sigaction(SIGINT, &sm_sigaction, 0);
    sigaction(SIGTERM, &sm_sigaction, 0);
    sm_sighandler_installed = true;
  }
  */

  // Create the set of callbacks
  m_callbacks_p = new vector<Callback>
    (static_cast<vector<Callback>::size_type>(CB_COUNT));

  m_valid                            = false;
  m_malloced_ximage_data             = false;
  m_destroy_me                       = false;
  m_destroyed                        = false;
  m_display_name                     =     0;
  m_list_index                       =    -1;
  m_name                             =     0;
  m_x                                =     0;
  m_y                                =     0;
  m_width                            =     0;
  m_height                           =     0;
  m_display_p                        =     0;
  m_vis_info_list_p                  =     0;
  m_visual_p                         =     0;
  m_font_name                        =     0;
  m_font_loaded                      = false;
  m_ximage_p                         =     0;
  m_ximage_data_p                    =     0;

  memset(&m_screen,        0, sizeof(m_screen));
  memset(&m_root_win,      0, sizeof(m_root_win));
  memset(&m_main_win,      0, sizeof(m_main_win));
  memset(&m_gc,            0, sizeof(m_gc));
  memset(&m_gcvalues,      0, sizeof(m_gcvalues));
  memset(&m_drawable,      0, sizeof(m_drawable));
  memset(&m_sizehints,     0, sizeof(m_sizehints));
  memset(&m_winattrib,     0, sizeof(m_winattrib));
  memset(&m_vis_info_tmpl, 0, sizeof(m_vis_info_tmpl));
  memset(&m_drawable,      0, sizeof(m_drawable));
  memset(&m_sizehints,     0, sizeof(m_sizehints));
  memset(&m_winattrib,     0, sizeof(m_winattrib));
  memset(&m_colormap,      0, sizeof(m_colormap));
  memset(&m_font,          0, sizeof(m_font));;
  memset(m_ip_disp_head,   0, sizeof(m_ip_disp_head));

  m_shm_capable  = false;
#if SPU_HAVE_XEXT_SHM == 1
  memset(&m_shm_info, 0, sizeof(m_shm_info));
  m_try_shm      = true;
  m_shm_attached = false;
  m_shm_ximage_p = 0;
  m_shm_size     = 0;
#endif

  m_dbe_capable    = false;
#if SPU_HAVE_XEXT_DBE == 1
  m_try_dbe        = true;
  m_dbe_svi_p      = 0;
  m_dbe_vis_info_p = 0;
  m_dbe_visual_p   = 0;
  m_dbe_depth      = 0;
  memset(&m_dbe_vis_info,    0, sizeof(m_dbe_vis_info));
  memset(&m_dbe_win,         0, sizeof(m_dbe_win));
  memset(&m_dbe_back_buffer, 0, sizeof(m_dbe_back_buffer));
  memset(&m_dbe_swap_action, 0, sizeof(m_dbe_swap_action));
  memset(&m_dbe_swap_info,   0, sizeof(m_dbe_swap_info));
#endif

  // ICCCM 
  memset(&m_atom_proto,         0, sizeof(m_atom_proto));
  memset(&m_atom_delete,        0, sizeof(m_atom_delete));
  memset(&m_atom_wm_colmap_win, 0, sizeof(m_atom_wm_colmap_win));

  // Color formats
  m_output_color_format = SPU_Color_Format_Unknown;
  m_input_color_format  = SPU_Color_Format_Unknown;
  m_fill_func_fp = 0;
  return;
}

XIH::~XIH() 
{
  SPU_mutex_lock(&sm_critical_mutex);
  if (!m_destroyed && m_valid) destroy();
  SPU_mutex_unlock(&sm_critical_mutex);
  return;
}

void 
XIH::close() 
{
  m_destroy_me = true;
  return;
}

void
XIH::destroy() 
{
  if (m_destroyed) {
    cerr << "XIH: Window already destroyed. Probably a bad thing." << endl;
    return;
  }

  // State
  m_destroyed = true;
  m_valid     = false;
  sm_valid_list[m_list_index] = 0;
  --sm_instance_count;


  // Discard pending events.
  if (m_display_p)
    XSync(m_display_p, True);

  // Free fonts.
  if (m_font_loaded)
    cerr << "XIH: Hey, implement font freeing please." << endl;

  // Detach shared memory or delete malloc()ed memory.
  detach_memory();

  // Destroy XImage.
  if (!m_shm_capable && m_ximage_p)
    XDestroyImage(m_ximage_p);

  // Destroy window.
  if (m_display_p) 
    XDestroyWindow(m_display_p, m_main_win);

  // Make sure the destruction events arrive.
  if (m_display_p)
    XSync(m_display_p, False);

  // Free other memory.
  free(m_display_name);
  free(m_name);

  // Should we kill the control thread?
  if (sm_instance_count == 0 && sm_die_with_last) 
    sm_kill = true;

  return;
}

bool 
XIH::init(const int width, 
	  const int height, 
	  const char* const icf,
	  const char* const name, 
	  const char* const display_name,
	  const int bytes_per_line,
	  XColor* color_pallete) 
{

  // Flag to indicate we should output information about the X server.
  bool print_info = false;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Ensure proper state.                                       //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  if (m_valid) {
    cerr << "XIH: Window already initialized." << endl;
    return(false);
  }
  
  if (sm_kill) {
    if (name)
      cerr << "XIH: Cannot create window \"" << name 
	   << "\" after XIH has been killed." << endl;
    else
      cerr << "XIH: Cannot create window after XIH has been killed." 
	   << endl;
    return(false);
  }
  
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Copy args into instance; using defaults where appropriate  //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  // 
  // -- Width
  m_width = width;
  //
  // -- Height
  m_height = height;
  //
  // --- Input color format
  if (icf) m_input_color_format = str_to_SPU_Color_Format(icf);
  else     m_input_color_format = SPU_Color_Format_Unknown;
  //
  // --- Window name
  if (name) m_name = strdup(name);
  else      m_name = strdup("XiH");
  //
  // --- Display name

  bool disp_name_valid       = XDisplayName(display_name);
  bool disp_name_not_empty   = strcmp(XDisplayName(display_name), "");
  //  bool localhost_not_implied = strncmp(XDisplayName(display_name), ":", 1);

  if (disp_name_valid && disp_name_not_empty) // && localhost_not_implied)
    m_display_name = strdup(XDisplayName(display_name));
  else 
    m_display_name = strdup(":0.0");
  
  //
  // --- Bytes per line
  if (bytes_per_line) 
    m_input_bytes_per_line = bytes_per_line;
  else 
    m_input_bytes_per_line = 
      m_width * ::bytes_per_pixel(m_input_color_format);
  
  // Get the IP, display, and head for this X server
  extract_ip_display_head(m_display_name, m_ip_disp_head);
  m_ip_disp_head_as_u64 = ip_disp_head_to_u64(m_ip_disp_head);

  // Discard head information (i.e., we just care about the IP and
  // server number --- hopefully, this corresponds to a unique X
  // process.
  //
  SPU_u64 ip_disp = m_ip_disp_head_as_u64 >> 8;

  if (sm_per_disp_map[ip_disp].m_init == false) {
    // Initilize per_display information here.
    sm_per_disp_map[ip_disp].m_init = true;
    print_info = true;
  }
  
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Enter critical section.                                    //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  pthread_mutex_lock(&sm_critical_mutex);

#if SPU_HAVE_X_THREADS == 1
  if (!sm_called_XInitThreads && XInitThreads()) {
    cerr << "XIH (Xlib): Initialized concurrent thread support." << endl;
    sm_called_XInitThreads = true;
  }
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Connect to the X server.                                   //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  
  m_display_p = XOpenDisplay(m_display_name);
  if (m_display_p == 0) {
    cerr << "XIH: Could not open DISPLAY \"" << XDisplayName(m_display_name) 
	 << "\"." << endl;
    pthread_mutex_unlock(&sm_critical_mutex);
    return(false);    
  }
  
#if XIH_SYNCHRONIZED == 1
  XSynchronize(m_display_p, True);
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Get Atom IDs (ICCCM).                                      //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  
  m_atom_proto  = XInternAtom(m_display_p, "WM_PROTOCOLS", False);
  m_atom_delete = XInternAtom(m_display_p, "WM_DELETE_WINDOW", False);

  
#if SPU_HAVE_XEXT_SHM == 1
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Detect MIT-SHM (Shared memory extension).                  //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  
  m_shm_attached = false;

  bool shm_okay =
    !sm_disable_shm_for_all &&	// SHM is not disabled
    m_try_shm &&		// SHM should be tried for this win
    ip_disp_head_is_local(m_ip_disp_head) && // is localhost
    XShmQueryExtension(m_display_p); // SHM detected

  if (shm_okay) {
    m_shm_capable = true;
    int shm_major, shm_minor;
    Bool sm_pixmaps;
    XShmQueryVersion(m_display_p, &shm_major, &shm_minor, &sm_pixmaps);
    if (print_info)
      cout << "XIH (SHM): MIT-SHM v" << shm_major << "." << shm_minor
	   << " detected on display \"" << m_display_name << "\"." << endl;
  }
#endif
  

#if SPU_HAVE_XEXT_DBE == 1
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Detect DBE (Double Buffer Extension).                      //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  int dbe_maj;
  int dbe_min;
  if (!sm_disable_dbe_for_all && // Ensure DBE globally enabled.
      m_try_dbe &&               // Ensure DBE enabled for object.
      XdbeQueryExtension(m_display_p, &dbe_maj, &dbe_min) // Ensure DBE exists.
      ) {
    if (print_info)
      cerr << "XIH (DBE): DBE v" << dbe_maj << "." << dbe_min 
	   << " detected on display \"" << m_display_name << "\"." << endl;
    m_dbe_capable = true;
  }
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Extract default information.                               //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  m_screen   = DefaultScreen    (m_display_p);
  m_root_win = DefaultRootWindow(m_display_p);
  m_colormap = DefaultColormap  (m_display_p, m_screen);
  m_visual_p = DefaultVisual    (m_display_p, m_screen);
  m_depth    = DefaultDepth     (m_display_p, m_screen);
  m_bytes_per_pixel = m_depth >>3;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Create the new window.                                     //
  //                                                            //
  ////////////////////////////////////////////////////////////////
#if SPU_HAVE_XEXT_DBE == 1
  if (m_dbe_capable) {
    m_dbe_visual_p = m_visual_p;
    XSetWindowAttributes win_attrs;
    win_attrs.background_pixel = BlackPixel(m_display_p, 0);
    win_attrs.border_pixel     = WhitePixel(m_display_p, 0);
    m_dbe_depth = m_depth;
    // printf("Creating %d %x %x %d %d %d (%d) %d (%d) %d %d %d\n", m_dbe_capable, (unsigned) m_display_p, (unsigned) m_root_win, m_x, m_y, m_width, width, m_height, height, m_border_width, m_border_val, m_background_val);
    m_dbe_win = 
      XCreateSimpleWindow(m_display_p, m_root_win, m_x, m_y,
			  m_width, m_height, m_border_width,
			  m_border_val, m_background_val);
    m_dbe_back_buffer = 
      XdbeAllocateBackBufferName(m_display_p, m_dbe_win, XdbeUndefined);
    m_main_win = m_dbe_win;    
    m_depth    = m_dbe_depth;
    m_bytes_per_pixel = m_depth >>3;
    m_visual_p = m_dbe_visual_p;
    m_drawable = m_dbe_back_buffer;
  } 
#endif
  if (!m_dbe_capable) {
    m_main_win = XCreateSimpleWindow(m_display_p, m_root_win, m_x, m_y,
				     m_width, m_height, m_border_width,
				     m_border_val, m_background_val);
    m_drawable = m_main_win;
  }

  // Finish setting up ICCCM; all this just so that we can listen to
  // "close" requests. Oy.
  //
  XSetWMProtocols(m_display_p, m_main_win, &m_atom_delete, 1);

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Allocate XImage storage.                                   //
  //                                                            //
  ////////////////////////////////////////////////////////////////

#if SPU_HAVE_XEXT_SHM == 1
  attach_shared_memory();
#endif

  // No shared memory. We have to malloc space for the image ourselves. :(
  //
  if (!m_shm_capable) {
    m_ximage_p = 
      XCreateImage(m_display_p, m_visual_p, m_depth, ZPixmap, 0, 
		   static_cast<char*>(0), m_width, m_height, 16, 0);
    m_ximage_data_mem_size =  m_height * m_ximage_p->bytes_per_line;
    m_ximage_data_p = static_cast<char*>(malloc(m_ximage_data_mem_size * sizeof(char)));
    m_ximage_p->data = m_ximage_data_p;
    m_output_bytes_per_line = m_ximage_p->bytes_per_line;
  }


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Set the desired/appropriate event masks                    //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  XSelectInput(m_display_p, m_main_win, 
	       KeyPressMask      | KeyReleaseMask    | 
	       ButtonPressMask   | ButtonReleaseMask | 
	       ExposureMask);
  
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Set size hints                                             //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  m_sizehints.flags      = PMaxSize | PMinSize;
  m_sizehints.min_height = m_height;
  m_sizehints.max_height = m_height;
  m_sizehints.min_width  = m_width;
  m_sizehints.max_width  = m_width;

  XSetNormalHints(m_display_p, m_main_win, &m_sizehints);
  XSetStandardProperties(m_display_p, m_main_win, m_name, m_name,
			 None, 0, 0, &m_sizehints);
  

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Create the Graphics Context                                //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  m_gcvalues.graphics_exposures = False;
  m_gc = XCreateGC(m_display_p, m_main_win,
		   GCGraphicsExposures, &m_gcvalues);


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Detect output color format.                                //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  
  m_output_color_format = detect_output_color_format();

  // Was the input color mode specified? If not, take a best guess.
  //
  if (m_input_color_format == SPU_Color_Format_Unknown)
    m_input_color_format = m_output_color_format;
  if (print_info)
    cerr << "XIH: X server " << m_display_name << " using " 
	 << SPU_Color_Format_to_long_str(m_output_color_format) << "." << endl;
  
  ////////////////////////////////////////////////////////////////
  //
  // Set pallette for 8-bit Pseudocolor (if appropriate)
  //
  if (m_output_color_format == SPU_Color_Format_Grey8) {
    m_colormap = XCreateColormap(m_display_p, m_root_win,
				 m_visual_p, AllocAll);

    // Allocate the XColor array.
    //
    // a) Use a default palette
    if (!color_pallete) {
      int colormap_size = DisplayCells(m_display_p, m_screen);
      XColor *colors = (XColor*) calloc(colormap_size, sizeof(XColor));
      XIH_fill_P8_palette(colors, 8);
      XStoreColors(m_display_p, m_colormap, colors, 1<<8);
      free(colors);
    } 
    // b) Use a user specified palette
    else {
      XStoreColors(m_display_p, m_colormap, color_pallete, 1<<8);
    }

    // Register colormap with window.
    XSetWindowColormap(m_display_p, m_main_win, m_colormap);

    // Register colormap with window manager.
    m_atom_wm_colmap_win =
      XInternAtom(m_display_p, "WM_COLORMAP_WINDOWS", False);
    XChangeProperty(m_display_p, m_main_win, m_atom_wm_colmap_win,
		    XA_WINDOW, 32, PropModeReplace, 
		    (unsigned char *) &m_main_win, 1);
  }

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Set palette for TrueColor (if appropriate)                 //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  if (m_output_color_format >= SPU_Color_Format_RGB24 &&
      m_output_color_format <= SPU_Color_Format_LSBPadded_BGR32) {
    m_vis_info_tmpl.screen = m_screen;
    m_vis_info_tmpl.c_class = TrueColor;
    m_colormap = DefaultColormap(m_display_p, m_screen);
    XSetWindowColormap(m_display_p, m_main_win, m_colormap);
  }

  set_fill_function();

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Map the window and wait.                                   //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  XMapWindow(m_display_p, m_main_win);
  XSync(m_display_p, False);
  
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Update state.                                              //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  // If we are here, then we have been sucessfully created! Up the
  // count, and add ourselves to the set of current windows.
  
  ++sm_instance_count;
  m_valid       = true;
  //
  // Add ourselves to the list of valid windows
  for (unsigned int i=0; i < sm_valid_list.size(); i++) {
    if (sm_valid_list[i] == 0) {
      sm_valid_list[i] = this;
      m_list_index = i;
      break;
    }
  }

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Finished critical section.                                 //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  pthread_mutex_unlock(&sm_critical_mutex);

  return(m_valid);
}

bool 
XIH::init(const SPU_Image& image, const char* name, const char* display) 
{

  bool rv = init(image.get_width(), 
		 image.get_height(),
		 image.get_color_format_as_string(), 
		 name, 
		 display, 
		 image.get_bytes_per_line());
  if (rv) {
    if (XIH_lock() && this->can_draw()) {
      this->fill(image);
      this->refresh();
      XIH_unlock();
    }
  }

  return(rv);
}

bool XIH::refresh() 
{
  XIH_VERIFY_LOCK("refresh");
  if (!m_valid) return(false);
#if SPU_HAVE_XEXT_DBE == 1
  if (m_dbe_capable) {
    m_dbe_swap_info.swap_window = m_main_win;
    m_dbe_swap_info.swap_action = XdbeUndefined; 
    XdbeSwapBuffers(m_display_p, &m_dbe_swap_info, 1);
  }
#endif
  XSync(m_display_p, False);
  return(true);
}

bool 
XIH::fill(const SPU_u8* from_p) 
{
  XIH_VERIFY_LOCK("fill");
  bool rv = false;
  if (m_valid && m_fill_func_fp && from_p) {
    (this->m_fill_func_fp)((SPU_u8*) from_p, m_input_bytes_per_line,
			  (SPU_u8*) m_ximage_data_p, m_output_bytes_per_line,
			  m_width, m_height);
  }
  rv = put();
  return(rv);
}

// TODO: Check the return values of these X calls?

bool 
XIH::put() 
{
  XIH_VERIFY_LOCK("put");
  if (!m_valid) return(false);

#if SPU_HAVE_XEXT_DBE == 1
  if (m_dbe_capable) {
#if SPU_HAVE_XEXT_SHM == 1
                                // Begin DBE + SHM section
    if (m_shm_capable)		// 
      XShmPutImage(m_display_p, m_dbe_back_buffer, m_gc,
		   m_shm_ximage_p, 0, 0, 0, 0, m_width, m_height, 0);
                                //
                                // End DBE + SHM section
#endif
                                // Begin "DBE only" section
    if (!m_shm_capable)		// 
      XPutImage(m_display_p, m_dbe_back_buffer, m_gc, 
		m_ximage_p, 0, 0, 0, 0, m_width, m_height);
                                //
                                // End "DBE only" section
  }
#endif
  if (!m_dbe_capable) {
#if SPU_HAVE_XEXT_SHM == 1
    if (m_shm_capable)		// SHM only
      XShmPutImage(m_display_p, m_main_win, m_gc,
		   m_shm_ximage_p, 0, 0, 0, 0,
		   m_width, m_height, 0);
#endif
    if (!m_shm_capable)		// Niether SHM nor DBE
      XPutImage(m_display_p, m_main_win, m_gc,
		m_ximage_p, 0, 0, 0, 0, m_width, m_height);
    
  }
  return(true);
}

void 
XIH::detach_memory() 
{
#if SPU_HAVE_XEXT_SHM == 1
  // Detach shared memory.
  if (m_shm_capable && m_shm_attached) {
    XShmDetach(m_display_p, &m_shm_info);
    XDestroyImage(m_shm_ximage_p);
    XSync(m_display_p, False);
    shmdt(m_shm_info.shmaddr);
    m_shm_attached = false;
  }
#endif

  // Free malloc()ed memory.
  if (m_malloced_ximage_data) {
    free(m_ximage_data_p);
    m_malloced_ximage_data = false;
  }
  return;
}

void XIH_fill_color(XColor* palette, int depth, int i, int r, int g, int b) 
{
  palette[i].pixel = i;
  palette[i].flags = DoRed | DoGreen | DoBlue;
  palette[i].red   = r << (16 - depth);
  palette[i].green = g << (16 - depth);
  palette[i].blue  = b << (16 - depth);
  return;
} 

void XIH_fill_P8_palette(XColor* palette, int depth) 
{
  int x_num_colors = 1 << depth;
  for (int i=0; i < x_num_colors; i++) 
    XIH_fill_color(palette, depth, i, i, i, i);

  XIH_fill_color(palette, depth,  0,   0,   0,   0); // Black
  XIH_fill_color(palette, depth,  1, 110,   0, 160); // Purple
  XIH_fill_color(palette, depth,  2,   0,   0, 180); // Navy blue
  XIH_fill_color(palette, depth,  3,  75,  75, 255); // Blue
  XIH_fill_color(palette, depth,  4, 100, 140, 170); // Grey blue
  XIH_fill_color(palette, depth,  5,  70, 200, 220); // Greenish blue
  XIH_fill_color(palette, depth,  6, 110, 255, 180); // Seafoam
  XIH_fill_color(palette, depth,  7, 154, 255,  50); // Limish green
  XIH_fill_color(palette, depth,  8,   0, 255,   0); // Green
  XIH_fill_color(palette, depth,  9, 143, 255,  27);
  XIH_fill_color(palette, depth, 10, 255, 255,   0);
  XIH_fill_color(palette, depth, 11, 255, 150,   0);
  XIH_fill_color(palette, depth, 12, 255, 120,  50);
  XIH_fill_color(palette, depth, 13, 255, 146, 100);
  XIH_fill_color(palette, depth, 14, 255,  76,  90);
  XIH_fill_color(palette, depth, 15, 255,   0,   0);
  return;
}

#if SPU_HAVE_XEXT_SHM == 1
void XIH::disable_shm() 
{
  m_try_shm = false;
  return;
}
#endif

#if SPU_HAVE_XEXT_DBE == 1
void XIH::disable_dbe() 
{
  m_try_dbe = false;
  return;
}
#endif

void 
XIH::set_fill_function() 
{
  m_fill_func_fp = SPU_get_colconv_fp(m_input_color_format,
				      m_output_color_format);
  if (!m_fill_func_fp)
    cerr << "XIH: " << SPU_Color_Format_to_str(m_input_color_format) 
	 << " to "  << SPU_Color_Format_to_str(m_output_color_format) 
	 << " conversion unsupported." << endl;

  return;
}

/* Don't need a signal handler here for me - jayg
void 
XIH_sigint_handler(int) 
{
  // This needs to be short and sweet; we really don't want a signal
  // handler eating up a whole lot of time.
  //
  if (XIH::sm_is_running) 
    XIH::sm_kill = true;
  else 
    exit(1);
  return;
}
*/

bool XIH_wait() 
{
  return(XIH::wait());
}

bool 
XIH::wait() 
{
  void *pt_status_p;
  pthread_mutex_lock(&XIH::sm_is_running_mutex);
  if (XIH::sm_is_running) {
    int err = pthread_join(XIH::sm_xih_control_thread, &pt_status_p);
    if (err == ESRCH)
      cerr << "XIH: Join error. No thread found." << endl;
    else if (err == EINVAL)
      cerr << "XIH: Join error. Thread already detached or being waited upon." << endl;
    else if (err == EDEADLK)
      cerr << "XIH: Join error. Attempt to join to self." << endl;
  }
  pthread_mutex_unlock(&XIH::sm_is_running_mutex);
  return(true);
}

void
XIH::set_screensaver(const bool& flag) 
{
  pthread_mutex_lock(&sm_critical_mutex);
  if (flag)
    XSetScreenSaver(m_display_p, -1, 0, 0, 0);
  else
    XSetScreenSaver(m_display_p, 0, 0, 0, 0);
  pthread_mutex_unlock(&sm_critical_mutex);
  return;
}

void
XIH::set_autorepeat(const bool& flag) 
{
  pthread_mutex_lock(&sm_critical_mutex);
  if (flag)
    XAutoRepeatOn(m_display_p);
  else
    XAutoRepeatOff(m_display_p);
  pthread_mutex_unlock(&sm_critical_mutex);
  return;
}

#if SPU_HAVE_XEXT_SHM == 1
void 
XIH_disable_all_shm() 
{
  XIH::sm_disable_shm_for_all = true;
}
#endif

#if SPU_HAVE_XEXT_DBE == 1
void 
XIH_disable_all_dbe() 
{
  XIH::sm_disable_dbe_for_all = true;
}
#endif


#if SPU_HAVE_XEXT_SHM == 1
bool 
XIH::attach_shared_memory() 
{
  
  if (strncmp(m_display_name, ":0", 10) != 0)
    return(m_shm_capable = false);

  // Create the shared memory XImage structure.
  m_shm_ximage_p = m_ximage_p =
    XShmCreateImage(m_display_p, m_visual_p, m_depth, ZPixmap, 0,
		    &m_shm_info, m_width, m_height);

  if (!m_shm_ximage_p) {
    cerr << "XIH (MIT-SHM): Could not create shared XImage." << endl;
    return(false);
  } else {
    m_output_bytes_per_line = m_shm_ximage_p->bytes_per_line;
  }

  // Create the shared memory segment to store image data.
  m_ximage_data_mem_size = m_output_bytes_per_line * m_height;
  m_shm_size = m_ximage_data_mem_size;
  m_shm_info.shmid = shmget(IPC_PRIVATE, m_shm_size, IPC_CREAT | 0777);
  if (m_shm_info.shmid == -1) {
    cerr << "XIH (MIT-SHM): Shared memory allocation failed." << endl;
    return(false);
  }

  // Attach the shared segment to the current process
  m_shm_info.shmaddr = m_shm_ximage_p->data = 
    static_cast<char*>(shmat(m_shm_info.shmid, 0, 0));
  if (m_shm_info.shmaddr == reinterpret_cast<char*>(-1)) {
    cerr << "XIH (MIT-SHM): Could not attach shared memory." << endl;
    return(false);
  }

  // Mark the segment to be destroyed after detachment.
  if (shmctl(m_shm_info.shmid, IPC_RMID, 0) == -1) {
    cerr << "XIH (MIT-SHM): Shared memory control failure." << endl;
    return(false);
  }

  // Initialize the segment.
  memset(m_shm_ximage_p->data, m_shm_size, 0);

  // Attach and register the segment.
  m_shm_info.readOnly = 0;
  if (XShmAttach(m_display_p, &m_shm_info)) {
    m_shm_attached = true;
  } else {
    m_shm_attached= false;
    m_shm_capable = false;
    return(false);
  }
  XSync(m_display_p, False);
  m_ximage_data_p = m_shm_ximage_p->data;
  return(true);
}
#endif

char 
XIH::extract_key_as_char(const XEvent* const xev) const
{
  char key = 0;
  if (xev->type == KeyPress || xev->type == KeyRelease) {
    KeySym keysym = XKeycodeToKeysym(m_display_p, xev->xkey.keycode, 0);
    if (keysym) key = *(XKeysymToString(keysym));
  } else 
    cerr << "XIH: Bad event type in XIH_extract_key()." << endl;
  return(key);
}

unsigned int 
XIH::extract_keysym(const XEvent* const xev) const
{
  return(XKeycodeToKeysym(m_display_p, xev->xkey.keycode, 0));
}

bool 
XIH::pressed_helper(const XEvent* const xev, unsigned int mask) const
{
  bool pressed = false;
  if (xev->type == KeyPress    || xev->type == KeyRelease)
    pressed = xev->xkey.state & mask;
  else if (xev->type == ButtonPress || xev->type == ButtonRelease)
    pressed = xev->xbutton.state & mask;
  else
    cerr << "XIH: Bad event type detected." << endl;
  return(pressed);
}

bool 
XIH::pressed_Control(const XEvent* const xev) const
{
  return(pressed_helper(xev, ControlMask));
}

bool 
XIH::pressed_Shift(const XEvent* const xev) const
{
  return(pressed_helper(xev, ShiftMask));
}

bool 
XIH::pressed_Mod1(const XEvent* const xev) const
{
  return(pressed_helper(xev, Mod1Mask));
}

bool 
XIH::pressed_Mod2(const XEvent* const xev) const
{
  return(pressed_helper(xev, Mod2Mask));
}

bool 
XIH::pressed_Mod3(const XEvent* const xev) const
{
  return(pressed_helper(xev, Mod3Mask));
}

bool 
XIH::pressed_Mod4(const XEvent* const xev) const
{
  return(pressed_helper(xev, Mod4Mask));
}

bool 
XIH::pressed_Mod5(const XEvent* const xev) const
{
  return(pressed_helper(xev, Mod5Mask));
}

int 
XIH::extract_button(const XEvent* const xev) const
{
  int button = 0;
  if (xev->type == ButtonPress || xev->type == ButtonRelease) 
    switch (xev->xbutton.button) {
    case Button1: button = 1; break;
    case Button2: button = 2; break;
    case Button3: button = 3; break;
    case Button4: button = 4; break;
    case Button5: button = 5; break;
    }
  else
    cerr << "XIH: Bad event type in XIH::extract_button()." << endl;
  return(button);
}

pair<int,int> 
XIH::extract_position(const XEvent* const xev) const
{
  pair<int,int> location(-1,-1);
  if (xev->type == ButtonPress || xev->type == ButtonRelease) {
    location.first  = xev->xbutton.x;
    location.second = xev->xbutton.y;
  } else
    cerr << "XIH: Bad event type in XIH::extract_position()." << endl;
  return(location);
}


SPU_Color_Format
XIH::detect_output_color_format() 
{

#if 0
  cout << m_depth << endl;
  cout << m_ximage_p->bits_per_pixel << endl;
  cout << hex << m_visual_p->red_mask << endl;
  cout << hex << m_visual_p->green_mask << endl;
  cout << hex << m_visual_p->blue_mask << endl;
#endif  

//    int red_max   = m_visual_p->red_mask;
//    int green_max = m_visual_p->green_mask;
//    int blue_max  = m_visual_p->blue_mask;
  
//    for (int red_shift = 0;   !(red_max & 1);   ++red_shift)   red_max >>= 1;
//    for (int green_shift = 0; !(green_max & 1); ++green_shift) green_max >>= 1;
//    for (int blue_shift = 0;  !(blue_max & 1);  ++blue_shift)  blue_max >>= 1;

  if (m_depth == 8 && m_visual_p->c_class == PseudoColor)
    return(SPU_Color_Format_Grey8);

  // 16-bit RGB or BGR
  //
  if (m_depth == 16 && m_visual_p->c_class == TrueColor) {
    if (m_visual_p->red_mask   == 0xf800 &&
	m_visual_p->green_mask == 0x07e0 &&
	m_visual_p->blue_mask  == 0x001f)
      return(SPU_Color_Format_RGB565);
    if (m_visual_p->red_mask   == 0x001f &&
	m_visual_p->green_mask == 0x07e0 &&
	m_visual_p->blue_mask  == 0xf800)
      return(SPU_Color_Format_BGR565);
  }

  // 24-bit RGB or BGR (not padded)
  //
  if (m_depth == 24 && 
      m_ximage_p->bits_per_pixel == 24 &&
      m_visual_p->c_class == TrueColor) {

    if (m_visual_p->red_mask   == 0xff0000 &&
	m_visual_p->green_mask == 0x00ff00 &&
	m_visual_p->blue_mask  == 0x0000ff) {
      if (ImageByteOrder(m_display_p) == LSBFirst)
	return(SPU_Color_Format_BGR24);
      else 
	return(SPU_Color_Format_RGB24);
    }

    if (m_visual_p->red_mask   == 0x0000ff &&
	m_visual_p->green_mask == 0x00ff00 &&
	m_visual_p->blue_mask  == 0xff0000)
      if (ImageByteOrder(m_display_p) == LSBFirst)
	return(SPU_Color_Format_RGB24);
      else
	return(SPU_Color_Format_BGR24);
  }

  // MSB or LSB padded 24-bit RGB or BGR
  //
  if (m_depth == 24 && 
      m_ximage_p->bits_per_pixel == 32 &&
      m_visual_p->c_class == TrueColor) {
    
    if (m_visual_p->red_mask   == 0x00ff0000 &&
	m_visual_p->green_mask == 0x0000ff00 &&
	m_visual_p->blue_mask  == 0x000000ff) {
      if (ImageByteOrder(m_display_p) == LSBFirst) 
	return(SPU_Color_Format_LSBPadded_BGR32);
      else
	return(SPU_Color_Format_MSBPadded_RGB32);
    }

    if (m_visual_p->red_mask   == 0x000000ff &&
	m_visual_p->green_mask == 0x0000ff00 &&
	m_visual_p->blue_mask  == 0x00ff0000) {
      if (ImageByteOrder(m_display_p) == LSBFirst) 
	return(SPU_Color_Format_LSBPadded_RGB32);
      else
	return(SPU_Color_Format_MSBPadded_BGR32);
    }

    if (m_visual_p->red_mask   == 0xff000000 &&
	m_visual_p->green_mask == 0x00ff0000 &&
	m_visual_p->blue_mask  == 0x0000ff00) {
      if (ImageByteOrder(m_display_p) == LSBFirst)
	return(SPU_Color_Format_MSBPadded_BGR32);
      else
	return(SPU_Color_Format_LSBPadded_RGB32);
    }

    if (m_visual_p->red_mask   == 0x0000ff00 &&
	m_visual_p->green_mask == 0x00ff0000 &&
	m_visual_p->blue_mask  == 0xff000000) {
      if (ImageByteOrder(m_display_p) == LSBFirst)
	return(SPU_Color_Format_MSBPadded_RGB32);
      else
	return(SPU_Color_Format_LSBPadded_BGR32);
    }
  }
  return(SPU_Color_Format_Unknown);
}

XIH::per_disp_info::per_disp_info() {
  m_init = false;
}

XIH::Callback::Callback() :
  m_cbtype(CB_None),
  m_xih(0),
  m_pf(0),
  m_pfvp(0),
  m_data(0),
  m_enabled(false),
  m_mutex_p(0)
{
  return;
}

XIH::Callback::Callback(const Callback& cb) 
{
  m_cbtype  = cb.m_cbtype;
  m_xih     = cb.m_xih;
  m_pf      = cb.m_pf;
  m_pfvp    = cb.m_pfvp;
  m_mutex_p = cb.m_mutex_p;
  m_data    = cb.m_data;
  m_enabled = cb.m_enabled;
}

void 
XIH::Callback::set(const Callback_Type cbtype,
		   XIH* xih,
		   const PF_XIHevent& cb,
		   pthread_mutex_t* mutex_p,
		   const bool enable)
{
  m_cbtype  = cbtype;
  m_xih     = xih;
  m_pf      = cb;
  m_pfvp    = 0;
  m_mutex_p = mutex_p;
  m_enabled = enable;
  return;
}

void
XIH::Callback::set(const Callback_Type cbtype,
		   XIH* xih,
		   const PF_XIHevent_vp& cb,
		   void* data,  
		   pthread_mutex_t* mutex_p,
		   const bool enable)
{
  m_cbtype    = cbtype;
  m_xih       = xih;
  m_pf        = 0;
  m_pfvp      = cb;
  m_data      = data;
  m_mutex_p   = mutex_p;
  m_enabled   = enable;
  return;
}

inline void
XIH::Callback::enable() 
{
  if (m_xih && (m_pf || m_pfvp))
    m_enabled = true;
  else
    cerr << "XIH: Can not enable unregistered callback." << endl;
  return;
}

inline void
XIH::Callback::disable() 
{
  if (m_xih && (m_pf || m_pfvp))
    m_enabled = false;
  else
    cerr << "XIH: Can not disable unregistered callback." << endl;
  return;
}

inline bool
XIH::Callback::toggle()
{
  if (m_xih && (m_pf || m_pfvp))
    m_enabled = !m_enabled;
  else
    cerr << "XIH: Can not toggle unregistered callback." << endl;
  return(m_enabled);
}

inline void
XIH::Callback::execute(const XEvent* const& xev) 
{
  if (!m_enabled) {
    return;
  }

  if (m_mutex_p)
    pthread_mutex_lock(m_mutex_p);

  if (m_pf) { 
    XIH::sm_executing_callback = true;
    XIH::sm_currently_executing_callback_type = m_cbtype;
    m_pf(m_xih, xev);
    XIH::sm_currently_executing_callback_type = XIH::CB_None;
    XIH::sm_executing_callback = false;
  }

  else if (m_pfvp) {
    XIH::sm_executing_callback = true;
    XIH::sm_currently_executing_callback_type = m_cbtype;
    m_pfvp(m_xih, xev, m_data);
    XIH::sm_currently_executing_callback_type = XIH::CB_None;
    XIH::sm_executing_callback = false;
  }

  if (m_mutex_p)
    pthread_mutex_unlock(m_mutex_p);


  return;
}

void
XIH::register_callback(const Callback_Type cb_type,
		       const PF_XIHevent_vp cb,
		       void* data,
		       pthread_mutex_t* mutex_p,
		       bool enable)
{
  int idx = static_cast<int>(cb_type);
  ((*m_callbacks_p)[idx]).set(cb_type, this, cb, data, mutex_p, enable);
  return;
}

void
XIH::register_callback(const Callback_Type cb_type,
		       const PF_XIHevent cb,
		       pthread_mutex_t* mutex_p,
		       bool enable) 
{
  int idx = static_cast<int>(cb_type);
  ((*m_callbacks_p)[idx]).set(cb_type, this, cb, mutex_p, enable);
  return;
}

bool
XIH::verify_lock(char* name) const
{
  bool rv = false;
  if (pthread_mutex_lock(&sm_critical_mutex) != EDEADLK) {
    cerr << "XIH Warning: Call XIH_lock() before calling " << name << "() "
	 << "on window \"" << m_name << "\"." << endl;
    rv = false;
  } else {
    rv = true;
  }
  return(rv);
}

bool
XIH::can_draw(void)
{
  if (!XIH::sm_is_running || XIH::sm_kill || !m_valid)
    return(false);

  XIH_VERIFY_LOCK("can_draw");
  return(true);
}

bool
XIH::lock_and_draw(void)
{
  if (!XIH::lock()) return(false);
  if (!this->can_draw()) {
    XIH::unlock();
    return(false);
  }
  return(true);
}

void
XIH::clear_cursor(void)
{
  XIH_VERIFY_LOCK("clear_cursor");
  static char empty_bm[] = { 0x00, 0x00 };
  
  Pixmap pmap = XCreatePixmapFromBitmapData
    (m_display_p, m_main_win, empty_bm, 8, 8, 0, 0, 1);
  XColor fg, bg;
  unsigned int xhot=0, yhot=0;
  Cursor cursor = XCreatePixmapCursor(m_display_p, pmap, pmap,
                                      &fg, &bg, xhot, yhot);
  XDefineCursor(m_display_p, m_main_win, cursor);
  return;
}


#if 0
inline bool XIH::lock(void) {

  if (sm_executing_callback) {
    int i = static_cast<int>(sm_currently_executing_callback_type);
    cerr << "XIH: Ignoring request to lock; currently executing "
	 << scm_Callback_Type_strings[i] << " callback.\n";
    return(false);
  }

  SPU_mutex_lock(&XIH::sm_critical_mutex);

  if (!XIH::sm_is_running || XIH::sm_kill || !m_valid) {
    SPU_mutex_unlock(&XIH::sm_critical_mutex);
    return(false);
  }
  return(true);
}

inline void 
XIH::unlock(void) {
  SPU_mutex_unlock(&XIH::sm_critical_mutex);
  return;
}
#endif



