// ----------------------------------------------------------------
// SPU-Toolbox
//
//        File: XIH.h (include/SPU-Toolbox/XIH.h)
// Description: XIH class declarations.
// ----------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file may use Doxygen style comments to facilitate automatic
// documentation generation. More information on Doxygen can be found
// at "http://www.stack.nl/~dimitri/doxygen/index.html".
//
// Author: Ross J. Micheals
//         rjm2@cse.lehigh.edu
//
// (c) 1999-2002 Ross J. Micheals
//
// TODO:
//
// * Fix the PSUEDO8 hack in set_draw_color.
// * Make a fake DBE when not available?
// * Code-out more color conversion routines. Test them.
// * RELATED: Integrate HERMES graphics libarary
// * Add rubber-banding selections?
// * Find out why init() isn't enough for image display (i.e. you have to 
//   fill/refresh() yourself)
//
// BE SURE TO DOCUMENT: 
//
// * How do avoid getting scan lines in the image. V4L does a shallow
//   copy, so if a put and a fill are separated by enough time, despite
//   locking there will be scanlines. Separating the ximage fill from the 
//   display requires a fill_without_put().

#ifndef __SPU_Toolbox__included__XIH_h__
#define __SPU_Toolbox__included__XIH_h__

#include "config.h"
#include "SPU-Types.h"

extern "C" {
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <netdb.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#if SPU_HAVE_XEXT_DBE == 1
#include <X11/extensions/Xdbe.h>
#endif

#if SPU_HAVE_XEXT_SHM == 1
#include <X11/extensions/XShm.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#include <pthread.h>
}

#if SPU_XIH_LOCK_DEBUGGING == 1
#define XIH_VERIFY_LOCK(name) verify_lock(name)
#else
#define XIH_VERIFY_LOCK(name)
#endif

#include <vector>
#include <map>

#include <SPU-Toolbox/SPU-Types.h>
#include <SPU-Toolbox/utils.h>
#include <SPU-Toolbox/x-utils.h>
#include <SPU-Toolbox/SPU-Image.h>
#include <SPU-Toolbox/pnm_helper.h>
#include <SPU-Toolbox/color-utils.h>
#include <SPU-Toolbox/color-convert.h>

using namespace std;

bool XIH_init_statics();
static bool XIH_static_initialization_hack = XIH_init_statics();

/*!
  \file XIH.h
  \brief XIH class declaration
*/

/*!

  \class XIH XIH.h SPU-Toolbox/XIH.h
  \brief Image display class
  \author Ross J. Micheals (rjm2@eecs.lehigh.edu)

  XIH is an object-based C++ API that simplifies both the display of
  images and drawing of basic 2D primitives under X
  Windows. Certainly, there exists a large number of APIs currently
  availble for the display of various types of images (the XIE
  extension for instance). However, none have been written with the
  frame-rate display of video in mind.

  XIH is designed to be as efficient as possible; it is multi-threaded
  (via pthreads, which are required) and will take advantage of
  various X extensions (such as MIT-SHM or DBE whenever possible). For
  video applications, the MIT shared memory X extension (MIT-SHM) is
  absolutely vital. In order to draw flicker-free primitives over
  video, DBE support is required. However, most (Unix) X
  implementations support both of the aformentioned extensions. Being
  GPL software, the most well supported X server is XFree86
  (xfree.org), but there have been several reports of XIH working
  under both the default SGI and Solaris X servers.

  Model

  XIH is composed of two major parts; a (singleton) thread of conrol
  and the windows themselves (which will be referred to as XIH windows
  or objects). This singleton thread handles all of the various X
  events.  Each XIH object is corresponds to the single window for
  which it is responsible. However, for all XIH objects present in a
  process, the single thread of control handles the trapping and
  execution of user events. These events can be triggered in user
  programs through the setting of various callbacks.

  Starting and Stopping

  In order to activate XIH windows, the main control thread must be
  started via a call to (the static member function)
  XIH::start(). Individual XIH windows may be constructed either
  before or after XIH::start(), in other words, the order in which XIH
  objects are instantiated is irrelevant. They will only become
  active, however, after the call to XIH::start(). Note that
  regardless of the number of windows in an application, XIH::start()
  only needs to be called once.

  The XIH::stop() callback terminates the XIH control thread.
  XIH::stop() should be called from the same thread that calls
  XIH::start(), but this is not stricly necessary. However, it should
  be noted that XIH::stop() returns immediately --- you must use
  XIH::wait() to block until the XIH control thread finally
  terminates. If the application is terminated before the XIH control
  thread ends cleanly, then the applications behavior is "undefined."
  (This usually means a core dump).

  Handling Events

  If the XIH control thread detects an event (for which there is a
  callback registered), the control thread will execute the user
  callback, but in a locked state. 

  It is critical to remember that callbacks are executed within the
  XIH control thread, and *not* during the user thread. Therefore,
  you have to be careful about how much work you do in each callback,
  because this is time that you will not be able to handle additional
  events. You might want to just have the callback simply set a flag,
  and have a main control loop act accordingly.

  One very important callback is the the "finished" callback. This
  callback is executed immediately after the XIH control thread
  recieves a stop signal, which could be triggered by a call to
  XIH::stop(), or from an external signal such as SIGINT (or
  Ctrl-C). After the "finished" callback executes, XIH will complete
  its termination. 



 
  SIGINT

  By default, XIH installs its own SIGINT (Ctrl-C) handler. 

  Locking

  At the lowest level, X can only handle one X event at a
  time. Because of this, most of the (useful) XIH member functions
  will require that the XIH control thread is *locked* before they are
  executed.
  

  

  Drawables

  Events

  Due to the nature of X, once started, the XIH control thread
  continually checks for new events. (It does this by looking at the X
  event queue "every once in a while", where "every once in a while"
  is currently set to about 30 times per second).

  System events, such as keyboard or mouse activity, are handled via a
  standard callback model. 

  To set a callback

  However, there are often times where the user wants to be able to
  pass their own information into the callback. 

*/

class XIH 
{
public:
  static bool start(void);
  static bool stop_all(void);
  static bool wait(void);

  static bool set_finished_callback(SPU_PFv);
  static bool set_finished_callback(SPU_PFvp);
  static bool set_finished_callback_user_data(void*);

  // Old interface to class statics
  friend bool XIH_start(void);
  friend bool XIH_stop_all(void);
  friend bool XIH_set_finished_callback(SPU_PFv);
  friend bool XIH_set_finished_callback(SPU_PFvp);
  friend bool XIH_set_finished_callback_user_data(void*);
  friend bool XIH_wait(void);
  friend bool XIH_init_statics(void);

  friend void* XIH_true_start(void*);

  friend void XIH_sigint_handler(int);

  friend bool XIH_lock(void);
  friend bool XIH_unlock(void);

#if SPU_HAVE_XEXT_SHM == 1
  friend void XIH_disable_all_shm(void);
#endif

#if SPU_HAVE_XEXT_DBE == 1
  friend void XIH_disable_all_dbe(void);
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Public types                                               //
  //                                                            //
  ////////////////////////////////////////////////////////////////
  
  typedef void (*PF_XIHevent)   (XIH*, const XEvent* );
  typedef void (*PF_XIHevent_vp)(XIH*, const XEvent*, void*);
  
  enum Callback_Type { CB_CloseWindow,
		       CB_ExposeWindow,
		       CB_ButtonPress,
		       CB_ButtonRelease,
		       CB_KeyPress,
		       CB_KeyRelease,
		       CB_COUNT,
                       CB_None };

  static const char* scm_Callback_Type_strings[];

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Construction / destruction                                 //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  //! Public-access default constructor
  /*!

    Creates an \e empty XIH instance. An instance is considered \e
    empty if it has been constructed, but is in no useful state. All
    \e empty instnaces must be explicitly initialized via init()
    before they can be used to display images

    \sa init()
    
   */
  XIH(void);

  //! Public-access default destructor
  /*!

    Public-access default destructor. Frees X resources and shared
    memory as appropriate.

   */
  ~XIH(void);

  //! Initialize XIH window
  /*!

    Create a new window for image display. Once initialized, XIH
    windows still must be filled with image data via fill() or
    read_pnm(). XIH windows will not appear on the display until the
    main XIH control thread is started (via XIH::start()). If the XIH
    control thread has already been start()ed, a newly initialized XIH
    windows will be displayed immediately after construction. If the
    XIH control thread has not already been start()ed, then the window
    will be displayed after the call XIH::start().

    The values for any unspecified parameters will be taken from their
    respective (user-defineable) defaults.

    Returns \e true if the object is initialized sucessfully. Returns
    \e false and prints an appropriate error message otherwise. If
    init() is called on an XIH instance that has already been
    sucessfully initialized, then init() returns \e true.
    
    \param width width of window (in pixels) to be created
    \param height height of window (in pixels) to be created
    \param icf color format of the input image
    \param name title of window
    \param display display to which to export window (localhost by
    default)
    \param custom color palette to use if X Windows is using 8-bit
    indexed color

    \sa XIH(), XIH::start(), fill()

  */
  bool init(const int width, 
	    const int height,
	    const char* const icf       = 0,  
	    const char* const name      = 0, 
	    const char* const display   = 0,
	    int const bytes_per_line    = 0,
	    XColor* color_pallete       = 0);

  //! Initialize XIH window according to a SPU_Image object
  /*!

    Create a new window for image display, appropriate to the image
    parameters in SPU_Image.


   */
  bool init(const SPU_Image& image,
	    const char* const name    = 0,
	    const char* const display = 0);

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Query State                                                //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  bool executing_callback();

  //! Query XIH window drawing state
  /*!

    Determines if the XIH object is in a \e drawable state.

   */
  bool can_draw();

  static bool lock();
  static bool unlock();

  //! Lock and check object drawing state simultaneously
  /*!

    Lock the main XIH thread and check to see if the XIH object is in
    a \e drawable state. Returns true if the object is
    drawable. Returns false otherwise.

   */

  bool lock_and_draw();

  ////////////////////////////////////////////////////////////////
  //
  // Attributes
  //
  ////////////////////////////////////////////////////////////////
  
  SPU_Color_Format output_color_format() const;
  char*            raw_data() const;


  void window_title(char* title);

  int get_bytes_per_line() const;
  int depth() const;
  int bytes_per_pixel() const;

  //! Query XIH window height
  /*!

    Returns the height of the XIH window.

   */
  int height() const;

  //! Query XIH window width
  /*!
    
    Returns the width of the XIH window.

    \sa height()

   */
  int width() const;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Drawing routines                                           //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  inline bool set_draw_color       (const char* const);
  inline bool set_draw_color_long  (const unsigned long int color);
  inline bool set_draw_font        (const char* const);
  inline void set_line_width       (const int& line_width);
  inline void draw_filled_rectangle(const int& x, const int& y,
				    const int& width, const int& height);
  inline void draw_rectangle       (const int& x, const int& y,
			            const int& width, const int& height);
  inline void draw_text            (const char* const text, 
				    const int& x, const int& y);
  inline void draw_circle          (const int& x, const int& y,
				    const int& radius);
  inline void draw_filled_circle   (const int& x, const int& y, 
				    const int& radius);
  inline void draw_point           (const int& x, const int& y);

  inline void draw_line            (const int& x1, const int& y1,
		                   const int& x2, const int& y2);

  inline void draw_outline_text    (const char* const text, int x, int y,
				    const char* const bg_color, 
				    const char* const fg_color,
				    bool full_border = false,
				    int thickness = 1);

  inline void bell(int percent = 100);

  void clear(void);

    
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Callbacks                                                  //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  void register_callback(const Callback_Type cb_type,
			 const PF_XIHevent cb,
			 pthread_mutex_t* = 0,
			 bool enable = true);

  void register_callback(const Callback_Type cb_type,
			 const PF_XIHevent_vp cb,
			 void* data,
			 pthread_mutex_t* = 0,
			 bool enable = true);


  ///////////////////////////////////////////////////////////////
  //                                                           //
  // Callback utilities                                        //
  //                                                           //
  ///////////////////////////////////////////////////////////////

  char          extract_key_as_char(const XEvent* const xev) const;
  unsigned int  extract_keysym     (const XEvent* const xev) const;
  bool          pressed_Control    (const XEvent* const xev) const;
  bool          pressed_Shift      (const XEvent* const xev) const;
  bool          pressed_Mod1       (const XEvent* const xev) const;
  bool          pressed_Mod2       (const XEvent* const xev) const;
  bool          pressed_Mod3       (const XEvent* const xev) const;
  bool          pressed_Mod4       (const XEvent* const xev) const;
  bool          pressed_Mod5       (const XEvent* const xev) const;
  int           extract_button     (const XEvent* const xev) const;
  pair<int,int> extract_position(const XEvent* const xev) const;


  ///////////////////////////////////////////////////////////////
  //                                                           //
  // X                                                         //
  //                                                           //
  ///////////////////////////////////////////////////////////////

  void set_screensaver(const bool& flag);
  void set_autorepeat(const bool& flag);
  void clear_cursor(void);

#if SPU_HAVE_XEXT_SHM == 1  
  void disable_shm(void);
#endif

#if SPU_HAVE_XEXT_DBE == 1
  void disable_dbe(void);
#endif

  bool fill(const SPU_u8* from_p);
  inline bool fill(const SPU_Image& image);

  bool fill_without_put(SPU_u8* from_p, SPU_Color_Format* icf_p = 0);
  bool fill_without_put(SPU_Image& image, SPU_Color_Format* icf_p = 0);
  
  bool refresh(void);
  bool put(void);

  void close(void);

private:

  // Member function pointer.
  typedef void (XIH::*XIH_MFP_uc)(unsigned char*);

  struct per_disp_info {
    per_disp_info(void);
    
    bool m_init;
  };

  class Callback {

  public:

    Callback(void);
    Callback(const Callback& cb);

    void set(const Callback_Type cbtype,
	     XIH* xih, 
	     const PF_XIHevent& cb, 
	     pthread_mutex_t* mutex_p = 0,
	     const bool enable = true);

    void set(const Callback_Type cbtype,
	     XIH* xih, 
	     const PF_XIHevent_vp& cb, 
	     void* data,
	     pthread_mutex_t* mutex_p = 0,
	     const bool enable = true);
    
    void execute(const XEvent* const& xev);

    void enable();
    void disable();
    bool toggle();

  private:

    XIH::Callback_Type m_cbtype;
    XIH*               m_xih;
    PF_XIHevent        m_pf;
    PF_XIHevent_vp     m_pfvp;

    void*              m_data;
    
    bool               m_enabled;
    pthread_mutex_t*   m_mutex_p;
  };
  friend class XIH::Callback;
  

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Private helper function prototypes                         //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  // Event handling helpers
  bool pressed_helper(const XEvent* const xev, unsigned int mask) const;

  // Construction / destruction helpers
  void set_fill_function(void);
  void detach_memory(void);
  void initialize_colconv_table(void);
  void destroy(void);
  SPU_Color_Format detect_output_color_format(void);
  
  bool verify_lock(char* name) const;

#if SPU_HAVE_XEXT_SHM == 1
  bool attach_shared_memory(void);
#endif

  // X
  void handle_events(void);
  
  // Start/stop helpers
  static void XIH::true_stop(void*);


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // S t a t i c   m e m b e r s                                //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  static const int                 scm_interevent_usleep_time;

  // Main control thread 
  static const int                 sm_default_valid_list_size;
  static vector<XIH*>              sm_valid_list;

  static pthread_mutex_t           sm_critical_mutex;

  //  static struct sigaction          sm_sigaction; jayg - removed this
  static pthread_t                 sm_xih_control_thread;
  static pthread_mutex_t           sm_start_stop_count_mutex;
  static pthread_mutex_t           sm_is_running_mutex;
  static pthread_cond_t            sm_is_running_cv;
  static int                       sm_start_stop_count;
  static int                       sm_instance_count;
  static bool                      sm_static_mutexes_initialized;
  static bool                      sm_sighandler_installed;
  static bool                      sm_called_XInitThreads;
  static bool                      sm_colconv_table_initialized;
  static bool                      sm_executing_callback;
  static Callback_Type             sm_currently_executing_callback_type;

  static map<SPU_u64, per_disp_info, less<SPU_u64> > sm_per_disp_map;

#if SPU_HAVE_XEXT_SHM == 1
  static bool                      sm_disable_shm_for_all;
#endif

#if SPU_HAVE_XEXT_DBE == 1
  static bool                      sm_disable_dbe_for_all;
#endif

  // Main control thread callbacks
  static SPU_PFv                   sm_finished_cb;
  static SPU_PFvp                  sm_finished_cb_with_user_data;
  static void*                     sm_finished_cb_user_data;

  // Main control thread state
  static bool                      sm_is_running;
  static bool                      sm_shutting_down;
  static bool                      sm_is_shutdown;

  static bool                      sm_kill;

  // Should the main control thread die when the last window does?
  static bool                      sm_die_with_last;

  // Default values
  static int           sm_default_x_pos;
  static int           sm_default_y_pos;
  static unsigned int  sm_default_border_width;
  static unsigned long sm_default_border_value;
  static unsigned long sm_default_background_value;
  static char*         sm_default_font_name;


  ////////////////////////////////////////////////////////////////
  //                                                            //  
  // N o n - s t a t i c   m e m b e r s                        // 
  //                                                            //  
  ////////////////////////////////////////////////////////////////


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Per-instance callbacks                                     //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  vector<Callback>* m_callbacks_p;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // State                                                      //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  bool m_valid;			// Am I a valid window?
  int  m_list_index;		// What is my index into the valid XIH list?
  bool m_destroy_me;		// If true, please destroy me.
  bool m_destroyed;		// I have been destoyed.


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // General window parameters                                  //
  //                                                            //
  ////////////////////////////////////////////////////////////////
 
  char* m_display_name;		// String describing display
  char* m_name;			// String describing window
  int   m_x;			// Preferred window origin x coord
  int   m_y;			// Preferred window origin y coord
  int   m_width;		// Window width
  int   m_height;		// Window height
  int   m_depth;		// output *bits* (not bytes!) per pixel
  int   m_bytes_per_pixel;	// output bytes per pixel
  int   m_border_width;
  int   m_border_val;
  int   m_background_val;
  int   m_input_bytes_per_line;	// bytes per line of incoming data
  int   m_output_bytes_per_line;// bytes per line of outgoing/X data


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // X Window specific structures                               //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  SPU_u8            m_ip_disp_head[6];
  SPU_u64           m_ip_disp_head_as_u64;

  Display*          m_display_p;
  Window            m_screen;
  Window            m_root_win;
  Window            m_main_win;
  GC                m_gc;
  XGCValues         m_gcvalues;
  Drawable          m_drawable;
  XSizeHints        m_sizehints;
  XWindowAttributes m_winattrib;
  XVisualInfo       m_vis_info_tmpl;
  XVisualInfo*      m_vis_info_list_p;
  Visual*           m_visual_p;
  Colormap          m_colormap;
  Font              m_font;
  char*             m_font_name;
  bool              m_font_loaded;

  XImage*           m_ximage_p;
  char*             m_ximage_data_p;
  bool              m_malloced_ximage_data;
  int               m_ximage_data_mem_size;	
  unsigned long     m_pixel_val;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // X extention: MIT-SHM                                       //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  bool              m_shm_capable;
#if SPU_HAVE_XEXT_SHM == 1
  bool              m_try_shm;
  bool              m_shm_attached;
  XShmSegmentInfo   m_shm_info;
  XImage*           m_shm_ximage_p;
  int               m_shm_size;
#endif


  ////////////////////////////////////////////////////////////////
  //                                                            //
  // X extension: DOUBLE-BUFFER                                 //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  bool                  m_dbe_capable;
#if SPU_HAVE_XEXT_DBE == 1
  bool                  m_try_dbe;
  XdbeScreenVisualInfo* m_dbe_svi_p;
  XVisualInfo           m_dbe_vis_info;
  XVisualInfo*          m_dbe_vis_info_p;
  Visual*               m_dbe_visual_p;
  Window                m_dbe_win;
  XdbeBackBuffer        m_dbe_back_buffer;
  XdbeSwapAction        m_dbe_swap_action;
  XdbeSwapInfo          m_dbe_swap_info;
  int                   m_dbe_depth;
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // ICCCM                                                      //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  Atom m_atom_proto;
  Atom m_atom_delete;
  Atom m_atom_wm_colmap_win;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Color formats and fill info                                //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  SPU_Color_Format     m_output_color_format;
  SPU_Color_Format     m_input_color_format;
  SPU_color_convert_FP m_fill_func_fp;
  
};

////////////////////////////////////////////////////////////////
//                                                            //
// XIH utility function prototypes                            //
//                                                            //
////////////////////////////////////////////////////////////////

SPU_Color_Format guess_ICF_from_OCF(const SPU_Color_Format&);
void XIH_fill_P8_palette(XColor*, int);

////////////////////////////////////////////////////////////////
//                                                            //
// Inline function definitions                                //
//                                                            //
////////////////////////////////////////////////////////////////

inline bool 
XIH::fill(const SPU_Image& image) 
{
  return fill(image.get_data());
}


inline void
XIH::draw_rectangle(const int& x, const int& y,
		    const int& width, const int& height) {
  XIH_VERIFY_LOCK("draw_rectangle");
  XDrawRectangle(m_display_p, m_drawable, m_gc, x, y, width, height);
  return;
}

inline void
XIH::draw_filled_rectangle(const int& x, const int& y,
			   const int& width, const int& height) {
  XFillRectangle(m_display_p, m_drawable, m_gc, x, y, width, height);
  return;
}

inline bool
XIH::set_draw_color(const char* const color) {
  XIH_VERIFY_LOCK("set_draw_color");
  bool rv = false;
  XColor xcolor;

  //! Fix this hack. Bad programmer, bad.
  if (m_output_color_format == SPU_Color_Format_Grey8) {
    if (strcasecmp(color, "white") == 0) {
      m_pixel_val = 255;
      rv = true;
    } else if (strcasecmp(color, "black") == 0) {
      m_pixel_val = 0;
      rv = true;
    }
  } else {
    if (XParseColor(m_display_p, m_colormap, color, &xcolor) &&
	XAllocColor(m_display_p, m_colormap, &xcolor)) {
      m_pixel_val = xcolor.pixel;
      rv = true;
    }
  }
  XSetForeground(m_display_p, m_gc, m_pixel_val);
  return rv;
}

inline bool
XIH::set_draw_color_long(const unsigned long int color) 
{
  XIH_VERIFY_LOCK("set_draw_color_long");
  XSetForeground(m_display_p, m_gc, color);
  return true;
}

inline bool
XIH::set_draw_font(const char* const font_name) {
  XIH_VERIFY_LOCK("set_draw_font");
  if (!font_name) return false;
  m_font = XLoadFont(m_display_p, font_name);
  XSync(m_display_p, False); //?
  if (!m_font) {
    cerr << "XIH: Could not load font \"" << font_name << "\"." << endl;
    return false;
  }
  XSetFont(m_display_p, m_gc, m_font);
  return true;
}

inline void
XIH::draw_text(const char* const text, 
	       const int& x, const int& y) {
  XIH_VERIFY_LOCK("draw_text");
  if (m_font) 
    XDrawString(m_display_p, m_drawable, m_gc, x, y, text, strlen(text));
  else {
    if (!m_font_name) m_font_name = sm_default_font_name;
    set_draw_font(m_font_name);
    XDrawString(m_display_p, m_drawable, m_gc, x, y, text, strlen(text));
  }
  return;
}

inline void
XIH::draw_circle(const int& x, const int& y, 
		 const int& radius) {
  XIH_VERIFY_LOCK("draw_circle");
  XDrawArc(m_display_p, m_drawable, m_gc, x-radius, y-radius, 
	   radius*2, radius*2, 0, 360*64);
}

inline void
XIH::draw_filled_circle(const int& x, const int& y, 
			const int& radius) {
  XIH_VERIFY_LOCK("draw_filled_circle");
  XFillArc(m_display_p, m_drawable, m_gc, x-radius, y-radius, 
	   radius*2, radius*2, 0, 360*64);
}

inline void
XIH::draw_point(const int& x, const int& y) {
  XIH_VERIFY_LOCK("draw_point");
  XDrawPoint(m_display_p, m_drawable, m_gc, x, y);
  return;
}

inline void 
XIH::draw_line(const int& x1, const int& y1,
	       const int& x2, const int& y2) {
  XIH_VERIFY_LOCK("draw_line");
  XDrawLine(m_display_p, m_drawable, m_gc, x1, y1, x2, y2);
  return;
}

inline void
XIH::set_line_width(const int& line_width) {
  XIH_VERIFY_LOCK("set_line_width");
  XSetLineAttributes(m_display_p, m_gc, line_width, 
		     LineSolid, CapProjecting, JoinMiter);
  return;
}


inline void 
XIH::draw_outline_text(const char* const text, int x, int y,
		       const char* const bg_color, 
		       const char* const fg_color,
		       bool full_border, int thickness) {
  XIH_VERIFY_LOCK("draw_outline_text");

  int t = thickness;
  set_draw_color(bg_color);

  //? Automagically do 8 points if it's small enough?
  draw_text(text, x+t, y+t);
  draw_text(text, x-t, y+t);
  draw_text(text, x+t, y-t);
  draw_text(text, x-t, y-t);
  
  if (full_border) {
    draw_text(text, x, y+t);
    draw_text(text, x, y-t);
    draw_text(text, x+t, y);
    draw_text(text, x-t, y);
  }
  
  set_draw_color(fg_color);
  draw_text(text, x, y);
  return;
}

inline void
XIH::bell(int percent) {
  XIH_VERIFY_LOCK("bell");
  XBell(m_display_p, percent);
}

inline SPU_Color_Format
XIH::output_color_format(void) const 
{
  XIH_VERIFY_LOCK("output_color_format");
  return m_output_color_format;
}

inline char*
XIH::raw_data(void) const 
{
  return m_ximage_data_p;
}

inline void
XIH::window_title(char* title) 
{
  free(m_name);
  m_name = strdup(title);
  XSetStandardProperties(m_display_p, m_main_win, m_name, m_name,
			 None, 0, 0, &m_sizehints);
}

inline bool
XIH::executing_callback()
{
  return sm_executing_callback;
}

inline void
XIH::clear()
{
  XIH_VERIFY_LOCK("clear");
  XSetForeground(m_display_p, m_gc, 0);
  draw_filled_rectangle(0, 0, m_width, m_height);
  XSetForeground(m_display_p, m_gc, m_pixel_val);
  return;
}

inline int 
XIH::get_bytes_per_line() const 
{
  return m_output_bytes_per_line;
}

inline int
XIH::depth() const 
{
  XIH_VERIFY_LOCK("depth");
  return m_depth;
}

inline int
XIH::width() const 
{
  XIH_VERIFY_LOCK("width");
  return m_width;
}

inline int
XIH::height() const 
{
  XIH_VERIFY_LOCK("height");
  return m_height;
}


inline int
XIH::bytes_per_pixel() const 
{
  XIH_VERIFY_LOCK("bytes_per_pixel");
  return m_bytes_per_pixel;
}

inline bool
XIH_lock(void) {
  return XIH::lock();
}

inline bool
XIH_unlock(void) {
  return XIH::unlock();
}

#endif // __SPU_Toolbox__included__XIH_h__


