// ----------------------------------------------------------------
// SPU-Toolbox
//
//        File: SPU-Image.cc (src/SPU-Image.cc)
// Description: Image class definitions.
// ----------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file uses PERCEPS style comments to facilitate automatic
// documentation generation. More information on PERCEPS can be
// found at "http://starship.python.net/crew/tbryan/PERCEPS".
//
// Author: Ross J. Micheals
//         rmicheals@lehigh.edu
//
// (c) 1999-2001 Ross J. Micheals

#include <SPU-Toolbox/SPU-Image.h>
#include <SPU-Toolbox/clamp.h>

#include <assert.h>

/*!

  \file SPU-Image.cc
  \brief Image class definitions

*/

int SPU_Image::sm_byte_alignment;

bool
SPU_Image_init_statics(void) 
{
  if (SPU_Image_static_initialization_hack == true) return(true);

  SPU_Image::sm_byte_alignment = 8; // Default to 64-bit alignment

  return(true);
}

#if SPU_HAVE_LIBJPEG == 1
bool
SPU_Image::save_jpeg(const char* fname, int quality) 
{
  if (!((m_color_format == SPU_Color_Format_RGB24) ||
       (m_color_format == SPU_Color_Format_Grey8))) {
    return(false);
  }
  bool rv = false;

  clamp(quality, 0, 100);
  rv = jpeg_write_helper(fname, m_aligned_mem, m_width, m_height,
			 m_bytes_per_line, m_color_format, 
			 quality);
  return(rv);
}


bool
SPU_Image::read_jpeg(const char* fname)
{
  if (!fname) return(false);

  if (m_filename) {
    free(m_filename);
    m_filename = 0;
  }

  m_filename = strdup(fname);
  if (!m_filename) return(false);

  bool rv = jpeg_read_helper(m_filename,
			     m_width, m_height,
			     m_bytes_per_line, 
			     m_bpp, // bytes per pixel
			     m_raw_mem, m_raw_memsize,
			     m_aligned_mem, m_aligned_memsize,
			     sm_byte_alignment);
  if (!rv) return(false);

  if (m_bpp == 1) m_color_format = SPU_Color_Format_Grey8;
  if (m_bpp == 3) m_color_format = SPU_Color_Format_RGB24;

  m_aligned_memsize = get_size();
  m_allocated_mem   = true;
  m_empty           = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image = 0;
  create_IPL_hook();
#endif

  m_init_called = true;

  return(rv);
}

#endif

bool
SPU_Image::save_pnm(const char* fname, bool binary) 
{

  if (m_color_format == SPU_Color_Format_Grey8) {

    FILE* fd;
    if ((fd = fopen(fname, "w")) == NULL) {
      fprintf(stderr, "Could not open file \"%s\".\n", fname);
      return(false);
    }
    
    fprintf(fd, "P5\n%d %d\n255\n", m_width, m_height);
    if (m_aligned_mem) {
      if (m_width == m_bytes_per_line) {
	fwrite((void*) m_aligned_mem, 
	       sizeof(unsigned char), m_width * m_height, fd);
	fclose(fd);
	return(true);
      } else {
	unsigned char* data;
	for (int i=0; i < m_height; i++) {
	  data = m_aligned_mem + i * m_bytes_per_line;
	  fwrite((void*) data, sizeof(unsigned char), m_width, fd);
	}
      }
      fclose(fd);
    }
  }


  if (m_color_format == SPU_Color_Format_RGB24) {

    FILE* fd;
    if ((fd = fopen(fname, "w")) == NULL) {
      fprintf(stderr, "Could not open file \"%s\".\n", fname);
      return(false);
    }
    
    char header[64];
    sprintf(header, "P6\n%d %d\n255\n", m_width, m_height);
    fwrite((void*) header, sizeof(unsigned char), strlen(header), fd);
    if (m_aligned_mem) {
      unsigned char* data;
      if (m_bytes_per_line == m_width *3)
	fwrite(m_aligned_mem, sizeof(unsigned char),
	       m_width*m_height*3, fd);
      else for (int i=0; i < m_height; i++) {
	data = m_aligned_mem + i * m_bytes_per_line;
	fwrite(data, sizeof(unsigned char), m_width*3, fd);
      }
      fclose(fd);
      return(true);
    }
  }

  if (m_color_format == SPU_Color_Format_BGR24) {

    FILE* fd;
    if ((fd = fopen(fname, "w")) == NULL) {
      fprintf(stderr, "Could not open file \"%s\".\n", fname);
      return(false);
    }
    
    char header[64];
    sprintf(header, "P6\n%d %d\n255\n", m_width, m_height);
    fwrite((void*) header, sizeof(unsigned char), strlen(header), fd);

    unsigned char* src_p;
    for (int j=0; j < m_height; j++) {
      src_p = m_aligned_mem + j * m_bytes_per_line;
      for (int i=0; i < m_width * 3; i+=3) {
	fwrite((void*) (src_p+2), sizeof(unsigned char), 1, fd);
	fwrite((void*) (src_p+1), sizeof(unsigned char), 1, fd);
	fwrite((void*) (src_p),   sizeof(unsigned char), 1, fd);
	src_p += 3;
      }
    }
    fclose(fd);
    return(true);
  }

  return(false);
}

bool
SPU_Image::init(const char*   color_format,
		const int     width, 
		const int     height,
		const bool    allocate_storage,
		const bool    with_copy,
		SPU_u8* const data,
		const int     bytes_per_line,
		const bool    do_not_pad) {
  
  if (!m_empty) {
    cerr << "SPU-Image: Only is_empty() Images may be init()ialized." << endl;
    return(false);
  }

  m_width        = width;
  m_height       = height;
  m_color_format = str_to_SPU_Color_Format(color_format);
  m_bpp          = bytes_per_pixel(m_color_format);
  m_aligned_mem  = data;

  // If bytes_per_line was not given, pad the image.
  if (bytes_per_line) {
    m_bytes_per_line = bytes_per_line;
  } else {
    int bpl = m_width * m_bpp;
    if (!do_not_pad && bpl % sm_byte_alignment)
      bpl = bpl - (bpl% sm_byte_alignment) + sm_byte_alignment;
    m_bytes_per_line = bpl;
  }
  m_aligned_memsize = this->get_size();
  m_filename        = 0;

  // Storage.
  m_allocated_mem = false;

  if (allocate_storage) {
    
    m_raw_memsize = m_aligned_memsize + sm_byte_alignment;
    m_raw_mem     = (SPU_u8*) malloc(static_cast<size_t>(m_raw_memsize));

    if (!m_raw_mem) {
      cerr << "SPU-Image: Memory allocation failure" << endl;
      return(false);
    } 

    // See if the memory is appropriately byte aligned
    if ((unsigned int) m_raw_mem % sm_byte_alignment) {
      m_aligned_mem = 
	(m_raw_mem + sm_byte_alignment) - 
	((unsigned int) m_raw_mem % sm_byte_alignment);
    } else {
      m_aligned_mem     = m_raw_mem;
      m_aligned_memsize = this->get_size();
    }
#if 0
    assert(m_aligned_mem >= m_raw_mem);
    cerr << "raw    = " << (unsigned int) m_raw_mem << endl;
    cerr << "raw sz = " << (unsigned int) m_raw_memsize << endl;
    cerr << "alg    = " << (unsigned int) m_aligned_mem << endl;
    cerr << "alg sz = " << (unsigned int) m_aligned_memsize << endl;
    cerr << "bpl    = " << (unsigned int) m_bytes_per_line << endl;
    cerr << "width  = " << (unsigned int) m_width << endl;
#endif
    memset(m_raw_mem, 0, m_raw_memsize);
    m_allocated_mem = true;
  }

  // Copy the image data if specified.
  if (with_copy) {
    
    // Ensure the source data is valid
    if (!data) {
      cerr << "SPU-Image: Attempt to copy bad image data (null)." << endl;
      return(false);
    }
    memcpy(m_aligned_mem, data, m_aligned_memsize);
  }

  // Update object state.
  m_empty       = false;
  m_open_status = true;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image = 0;
  create_IPL_hook();
#endif
  // If everything went okay, then consider init() officially "called."
  m_init_called = true;

  return(m_init_called);
}

bool
SPU_Image::assign(const SPU_Image* image, 
		  bool allocate_storage, bool with_copy) 
{

  m_width            = image->m_width;
  m_height           = image->m_height;
  m_color_format     = image->m_color_format;
  m_bpp              = image->m_bpp;
  m_bytes_per_line   = image->m_bytes_per_line;
  m_aligned_memsize  = image->m_aligned_memsize;
  m_constant_pixel_p = image->m_constant_pixel_p;

  // Filename.
  if (m_filename) {
    free(m_filename); 
    m_filename = 0;
  }
  if (image->m_filename)
    m_filename = strdup(image->m_filename);

  // Storage.
  if (allocate_storage) {
    
    // Free already allocated storage.
    if (m_allocated_mem) free(m_raw_mem);

    m_raw_memsize = m_aligned_memsize + sm_byte_alignment;
    m_raw_mem     = (SPU_u8*) malloc(static_cast<size_t>(m_raw_memsize));
    if (!m_raw_mem) {
      cerr << "SPU-Image: Failed to allocate memory." << endl;
      return(false);
    }

    // Verify alignment
    if ((unsigned int) m_raw_mem % sm_byte_alignment) {
      m_aligned_mem = (m_raw_mem + sm_byte_alignment) - 
	((unsigned int) m_raw_mem % sm_byte_alignment);
    } else {
      m_aligned_mem     = m_raw_mem;
      m_aligned_memsize = this->get_size();
    }
#if 0
    assert(m_aligned_mem >= m_raw_mem);
    cerr << "raw    = " << (unsigned int) m_raw_mem << endl;
    cerr << "raw sz = " << (unsigned int) m_raw_memsize << endl;
    cerr << "alg    = " << (unsigned int) m_aligned_mem << endl;
    cerr << "alg sz = " << (unsigned int) m_aligned_memsize << endl;
    cerr << "bpl    = " << (unsigned int) m_bytes_per_line << endl;
    cerr << "width  = " << (unsigned int) m_width << endl;
#endif
    memset(m_raw_mem, 0, m_raw_memsize);
    m_allocated_mem = true;
  }

  // Copy the image.
  if (with_copy) {

    // Ensure the source image is valid
    if (!image->m_aligned_memsize || image->m_empty) {
      cerr << "SPU-Image: Attempt to duplicate empty/null image." << endl;
      return(false);
    }      

    // No memory allocated, but a copy was desired. Warn the user.
    if (!m_allocated_mem) {
      cerr << "SPU-Image: Requested copy without allocation.\n";
      return(false);
    }

    // Only assign images of identical size.
    if (image->m_width  != m_width ||
	image->m_height != m_height) {
      cerr << "SPU-Image: Cannot assign images of different size." << endl;
      return(false);
    } 

    // Nicely copy the image
    int     from_bpl = image->m_bytes_per_line;
    SPU_u8* from_p   = image->m_aligned_mem;
    int     to_bpl   = m_bytes_per_line;
    SPU_u8* to_p     = m_aligned_mem;

    if (from_p && from_bpl == to_bpl) {
      memcpy(to_p, from_p, m_height * from_bpl);
    } else {
      register SPU_u8* dest_p;
      register SPU_u8* src_p;
      int memcpy_bpl = from_bpl < to_bpl? from_bpl:to_bpl;
      for (int i=0; i < m_height; i++) {
	src_p  = from_p + i * from_bpl;
	dest_p = to_p   + i * to_bpl;
	memcpy(dest_p, src_p, memcpy_bpl);
      }
    } 
    
  } else {
    // No copy specified. If no storage allocation requested, shallow copy.
    if (!allocate_storage)
      m_aligned_mem = image->m_aligned_mem;
  }

#if SPU_HAVE_LIBIPL == 1
  create_IPL_hook();
#endif

  m_empty = false;

  return(true);
}

bool 
SPU_Image::insert_subimage(const SPU_Image& inim, 
			   int tx, int ty,  // start tx,ty
			   int iwidth, int iheight, 
			   int fx, int fy) // start from fx,fy
{


  if(inim.get_color_format() 
     != get_color_format()) {
    cerr << "Can not insert subimage with different color format" ;
      return false ;
  }
  // if negative we use whole sub image 
  if(iwidth <0) iwidth = inim.get_width()-fx;
  if(iheight <0) iheight = inim.get_height()-fy;

  if(fy<0) {
    iheight += fy;
    fy = 0;
  }
  if(fx<0) {
    iwidth += fx;
    fx = 0;
  }
  
  // make sure we will fit left to right
  // do it once since it never changes
  int length;
  length = iwidth;
  if(tx<0)  tx=0; 
  else if(tx+length > get_width()) {
    length = get_width() - tx;
  }
      

  if(ty < 0) ty=0;
  iheight += ty; // comput end point
  if(iheight > get_height()){
    iheight = get_height();
  } 

  const SPU_u8* indata = inim.get_data();
  SPU_u8* mydata = get_data();

  for(; ty< iheight; ty++,fy++){

    int myoffset = ty*get_bytes_per_line()
      +tx*get_bytes_per_pixel();
    int inoffset = fy*inim.get_bytes_per_line()
      +fx*inim.get_bytes_per_pixel();
    memcpy(mydata+myoffset,indata+inoffset,
	   length*get_bytes_per_pixel());
  }
  return true;
}

SPU_u8*
SPU_Image::get_pixel(const int x, const int y) 
{
  SPU_u8* data = 0;

  // If x and y are out of bounds, do the right thing depending on the
  // indexing style.
  //
  if (x > m_width || y > m_height || x < 0 || y < 0) {
    switch (m_index_type) {
    case Index_Torroidal: data = get_Torroidal_pixel(x,y); break;
    case Index_Mirrored:  data = get_Mirrored_pixel(x,y);  break;
    case Index_Extended:  data = get_Extended_pixel(x,y);  break;
    case Index_Constant:  data = get_Constant_pixel(x,y);  break;
    default: break;		// FIXME
    } // switch (index type)
  } else {
    data = m_aligned_mem + y*m_bytes_per_line + x*m_bpp;
  } // if (out of bounds)
  return(data);
}

const SPU_u8*
SPU_Image::get_pixel(const int x, const int y) const
{
  const SPU_u8* data = 0;

  // If x and y are out of bounds, do the right thing depending on the
  // indexing style.
  //
  if (x > m_width || y > m_height || x < 0 || y < 0) {
    switch (m_index_type) {
    case Index_Torroidal: data = get_Torroidal_pixel(x,y); break;
    case Index_Mirrored:  data = get_Mirrored_pixel(x,y);  break;
    case Index_Extended:  data = get_Extended_pixel(x,y);  break;
    case Index_Constant:  data = get_Constant_pixel(x,y);  break;
    default: break;		// FIXME
    } // switch (index type)
  } else {
    data = m_aligned_mem + y*m_width + m_bytes_per_line + x*m_bpp;
  } // if (out of bounds)
  return(data);
}

bool
SPU_Image::read_pnm(const char* fname) {

  if (!fname) return(false);

  if (m_filename) {
    free(m_filename);
    m_filename = 0;
  }

  m_filename = strdup(fname);
  if (!m_filename) return(false);
  
  int mode;

  // This will free m_data if image size changes
  bool rv =
    pnm_read_helper(m_filename, 
		    m_width, m_height, mode, 
		    m_bytes_per_line, 
		    m_raw_mem, m_raw_memsize,
		    m_aligned_mem, m_aligned_memsize,
		    sm_byte_alignment);
  if (!rv) return(false);

  
  // Image was a P2 or P5
  if (mode == 2 || mode == 5) {
    m_color_format = SPU_Color_Format_Grey8;
    m_bpp          = 1;
  }
  
  // Image was a P3 or P6
  else if (mode == 3 || mode == 6) {
    m_color_format = SPU_Color_Format_RGB24;
    m_bpp          = 3;
  }
  
  // Image was an unknown format
  else
    m_color_format = SPU_Color_Format_Unknown;
  
  m_aligned_memsize = get_size();
  m_allocated_mem   = true;
  m_empty           = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image = 0;
  create_IPL_hook();
#endif

  m_init_called = true;

  return(true);
}


bool
are_compatable(const SPU_Image& img0, const SPU_Image& img1) 
{
  return(img0.get_color_format() == img1.get_color_format() &&
	 img0.get_width()        == img1.get_width()        &&
	 img0.get_height()       == img1.get_height());
}


void
SPU_Image::set_byte_aligment(int bytes)
{
  SPU_Image::sm_byte_alignment = bytes;
  return;
}

bool
SPU_Image::verify_byte_alignment(int bytes)
{
  // If the bpl is not aligned, the image cannot be
  if (m_bytes_per_line % bytes)
    return(false);

  // Check for excessive padding
  if (m_bytes_per_line - m_width * m_bpp >= bytes)
    return(false);
  
  return(true);
}

bool
SPU_Image::release(void) 
{
  if (!m_allocated_mem) {
    return(false);
  } else {
    free(m_raw_mem);
    m_raw_mem         = 0;
    m_raw_memsize     = 0;
    m_aligned_mem     = 0;
    m_aligned_memsize = 0;
    m_allocated_mem   = false;
  }
  // FIXME IPL stuff here
  return(true);
}

void
SPU_Image::fill(const SPU_u8 val)
{
  memset(static_cast<void*>(m_raw_mem), static_cast<int>(val), m_raw_memsize);
  return;
}

bool SPU_Image::extract_subimage(SPU_Image& output_image,
				 int ox, int oy) 
{
  if (ox < 0 || oy < 0 || ox > m_width || oy > m_height) {
    cerr << "SPU-Image: Subimage origin out of bounds." << endl;
  }

  if (output_image.get_width() > m_width || 
      output_image.get_height() > m_height) {
    cerr << "SPU-Image: Subimage to extract too large." << endl;
  }

  if (m_color_format != output_image.get_color_format()) {
    cerr << "SPU-Image: Subimage to extract different color format." << endl;
  }

  if (output_image.is_empty()) {
    cerr << "SPU-Image: Subimage to extract not initialized." << endl;
  }

  //  SPU_u8* base_src_p  = m_aligned_mem + oy*m_bytes_per_line + m_bpp*ox;
  //  SPU_u8* base_dest_p = output_image.get_data();

  int real_height = output_image.get_height();
  int real_width  = output_image.get_width();
  
  // Check this
  if (ox + real_width  > m_width)  real_width  = m_width  - ox;
  if (oy + real_height > m_height) real_height = m_height - oy;

  SPU_u8 *src_p, *dest_p;
  for (int y = oy, src_y = 0; y < oy+real_height; y++, src_y++) {
    src_p  = m_aligned_mem + y*m_bytes_per_line + m_bpp*ox;
    dest_p = output_image.get_data() + src_y*output_image.get_bytes_per_line();
    memcpy(dest_p, src_p, real_width*m_bpp);
  }
  return(true);
}


#if SPU_HAVE_LIBIPL == 1
bool
SPU_Image::create_IPL_hook(void) {

  // Unallocate the previous header if necessary
  if (m_ipl_image) {
    iplDeallocate(m_ipl_image, IPL_IMAGE_HEADER);
  }

  int   ipl_channels      = 0;
  int   ipl_depth         = 0;
  int   ipl_data_order    = 0;
  char* ipl_color_model   = 0;
  char* ipl_channel_seq   = 0;

  SPU_Color_Format_to_ipl_Color_Format(m_color_format,
				       ipl_channels,
				       ipl_depth,
				       ipl_color_model,
				       ipl_channel_seq,
				       ipl_data_order);

  if (!verify_byte_alignment(8)) {
    cerr << "SPU_Image: IPL hooks require quad-word aligned images." << endl;
    return(false);    
  }

  m_ipl_image = iplCreateImageHeader
    (ipl_channels, 0, ipl_depth, ipl_color_model, 
     ipl_channel_seq, ipl_data_order, IPL_ORIGIN_TL,
     IPL_ALIGN_QWORD, m_width, m_height, 0, 0, 0, 0);
  
  if (!m_ipl_image || iplGetErrStatus()) {
    cerr << "IPL: Error (" << iplGetErrStatus() << ")."<< endl;
    return(false);
  }

  m_ipl_image->imageData       = (char*) m_aligned_mem;
  m_ipl_image->imageDataOrigin = (char*) m_aligned_mem;
  return(true);
}


#endif

