#include <stdio.h>
extern "C" {
#include <jpeglib.h>
}

#include <utils/Output.h>
#include <utils/ConfigFile.h>
#include <utils/CannedDataAccess.h>

#include <TimeSource/TimeSource.h>
#include <ImgDisplay/FileDisplay.h>

class JPEGWindow;
class JPEGDisplay : public FileDisplay
{
 public:
  JPEGDisplay(int quality, const char** names, const char* dest_fmt) :
    FileDisplay(names, dest_fmt) {}

  virtual FileWindow* newFileWindow(int width, int height,
                                    ImgDisplay::ColorType src,
                                    ImgDisplay::ColorType dest);

private:
  int _quality;
};

class JPEGWindow : public FileWindow   
{
 public:
  JPEGWindow(int quality,
             const char* base_name, ImgDisplay* display, int width, int height,
             ImgDisplay::ColorType src_type, 
             ImgDisplay::ColorType dest_type);
  virtual ~JPEGWindow();

  virtual bool begin();
  virtual bool writeFile();
  virtual void setTimeTag(utils::Time t) { _tag = t; }

private:
  utils::CannedDataWriter* _writer;
  unsigned char* _jpeg_tmp;
  int _total_size, _row_stride;
  struct jpeg_compress_struct _cinfo;
  struct jpeg_destination_mgr _dest;
  struct jpeg_error_mgr _jerr;
  JSAMPROW* _row_pointer;
  utils::Time _tag;
};

FileWindow* JPEGDisplay::newFileWindow(int width, int height,
                                       ImgDisplay::ColorType src_type,
                                       ImgDisplay::ColorType dest_type)
{
  return new JPEGWindow(_quality, getCurrentName(),
                        this, width, height, src_type, dest_type);
}
 
static void my_init_destination(j_compress_ptr cinfo)
{
}

static boolean my_empty_output_buffer(j_compress_ptr cinfo)
{
  fprintf(stderr, "Emptying buffer: this should never happen\n");
  return FALSE;
}

static void my_term_destination(j_compress_ptr cinfo)
{
}

JPEGWindow::JPEGWindow(int quality, const char* base_name,
                       ImgDisplay* display, int width, int height, 
                       ImgDisplay::ColorType src_type,
                       ImgDisplay::ColorType dest_type)
  : FileWindow(base_name, display, width, height, src_type, dest_type)
{
  if (dest_type != ImgDisplay::Grey8 && dest_type != ImgDisplay::RGB24) {
    _writer = NULL;
    _jpeg_tmp = NULL;
    _row_pointer = NULL;
    fprintf(stderr, "Cannot output jpeg with type %s\n",
            ImgDisplay::colorTypeToString(dest_type));
    return;
  }

  _writer = new utils::CannedDataWriter();
  
  try {
    printf("Opening %s for output\n", fileName());
    _writer->open(fileName());
    _writer->writeType(utils::CDT_VIDEO);
    _writer->writeVersion(0,0);
    _writer->writeDescription("Compressed image data");
  
    utils::ConfigFile header;
    int i;
    header.setInt("int num_rows", getHeight());
    header.setInt("int num_cols", getWidth());
    if (isColor()) 
      i = 3;
    else
      i = 1;
    header.setInt("int num_bands", i);
    header.setBool("bool interlaced", true);
    header.setBool("bool row_major", true);
    header.setInt("int pixel_size", i);

    char* buffer = new char[200];
    utils::Output output;
    output.setBuffer(buffer, 200, utils::Output::standardResize);
    header.write(output);
    output.write('\0');
    void* out_buf;
    int size;
    output.getBuffer(out_buf, size);
    _writer->writeHeader(out_buf, size);
    delete [] (char*) out_buf;
  } catch (utils::CannedDataError err) {
    utils::String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    delete _writer;
    _writer = NULL;
    return;
  }

  _row_stride = width*getBytesPerPixel(dest_type);
  _total_size = _row_stride*height;
  _jpeg_tmp = new unsigned char[_total_size];
  
  _cinfo.err = jpeg_std_error(&_jerr);
  jpeg_create_compress(&_cinfo);

  _cinfo.image_width  = width;
  _cinfo.image_height = height;

  if (!isColor()) {
    _cinfo.input_components = 1;
    _cinfo.in_color_space   = JCS_GRAYSCALE;
  } else {
    _cinfo.input_components = 3;
    _cinfo.in_color_space   = JCS_RGB;
  }
  _row_pointer = new JSAMPROW[_cinfo.image_height];
  for (int i=0; i < (int) _cinfo.image_height; i++) {
    _row_pointer[i] = data() + i * _row_stride;
  }

  _cinfo.client_data = this;
  _cinfo.dest = &_dest;
  _dest.init_destination = my_init_destination;
  _dest.empty_output_buffer = my_empty_output_buffer;
  _dest.term_destination = my_term_destination;

  jpeg_set_defaults(&_cinfo);
  jpeg_set_quality(&_cinfo, quality, TRUE);
}

JPEGWindow::~JPEGWindow()
{
  delete _writer; 
  delete [] _jpeg_tmp;
  delete [] _row_pointer;
};

bool JPEGWindow::begin()
{
  return _writer && FileWindow::begin();
}

bool JPEGWindow::writeFile()
{
  if (!_writer)
    return false;

  utils::Time t;
  if (_tag.isZero()) {
    t = TimeSource::now();
  } else
    t = _tag;

  _dest.next_output_byte = _jpeg_tmp;
  _dest.free_in_buffer = _total_size;

  jpeg_start_compress(&_cinfo, TRUE);

  while (_cinfo.next_scanline < _cinfo.image_height) {
    jpeg_write_scanlines(&_cinfo, _row_pointer,
                         _cinfo.image_height);
  }

  jpeg_finish_compress(&_cinfo);

  int size = _dest.next_output_byte - _jpeg_tmp;
  long secs, usecs;
  t.getValue(secs, usecs);
  try {
    _writer->writeData(_jpeg_tmp, size, secs, usecs);
  } catch (utils::CannedDataError err) {
    utils::String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }

  return true;
}
    
ImgDisplay* create_ImgDisplay_jpeg(utils::Generator<ImgDisplay>* gen,
                                   utils::ConfigFile* params,
                                   utils::SymbolTable* globals)
{
  const char* names[11];
  memset(names, 0, 11*sizeof(const char*));
  names[0] = "image";
  params->getStrings("names", names, 10);
  return new JPEGDisplay(params->getInt("quality", 70),
                         names, params->getString("dest_fmt", "unknown"));
}
