#include <stdio.h>
#include <ImgDisplay/BufferDisplay.h>

static unsigned short from_Red8_to_BGR565[0x100];
static unsigned short from_Green8_to_BGR565[0x100];
static unsigned short from_Blue8_to_BGR565[0x100];
static unsigned short from_Red8_to_BGR1555[0x100];
static unsigned short from_Green8_to_BGR1555[0x100];
static unsigned short from_Blue8_to_BGR1555[0x100];
static unsigned short from_Grey8_to_BGR1555[0x100];
static unsigned short from_Red8_to_RGB565[0x100];
static unsigned short from_Green8_to_RGB565[0x100];
static unsigned short from_Blue8_to_RGB565[0x100];
static unsigned short from_Grey8_to_RGB565[0x100];
static unsigned short from_Red8_to_RGB1555[0x100];
static unsigned short from_Green8_to_RGB1555[0x100];
static unsigned short from_Blue8_to_RGB1555[0x100];
static unsigned short from_Grey8_to_RGB1555[0x100];

static void init_color_to_BGR1555()
{
  if (from_Red8_to_BGR1555[0xff]) 
    return;
  for (int i=0;i<0x100;i++) {
    from_Blue8_to_BGR1555[i]=(i<<7)&0x7c00;
    from_Green8_to_BGR1555[i]=(i<<2)&0x03e0;
    from_Red8_to_BGR1555[i]=(i>>3)&0x001f;
  }
}

static void init_grey_to_BGR1555()
{
  init_color_to_BGR1555();
  if (from_Grey8_to_BGR1555[0xff])
    return;
  for (int i=0;i<0x100;i++) {
    from_Grey8_to_BGR1555[i] = (from_Red8_to_BGR1555[i]) |
      (from_Green8_to_BGR1555[i]) |
      (from_Blue8_to_BGR1555[i]);
  }
}
  
static void init_color_to_BGR565()
{
  if (from_Red8_to_BGR565[0xff]) 
    return;
  for (int i=0;i<0x100;i++) {
    from_Blue8_to_BGR565[i]=(i<<8)&0xf800;
    from_Green8_to_BGR565[i]=(i<<3)&0x07e0;
    from_Red8_to_BGR565[i]=(i>>3)&0x001f;
  }
}

static void init_color_to_RGB1555()
{
  if (from_Red8_to_RGB1555[0xff]) 
    return;
  for (int i=0;i<0x100;i++) {
    from_Red8_to_RGB1555[i]=(i<<7)&0x7c00;
    from_Green8_to_RGB1555[i]=(i<<2)&0x03e0;
    from_Blue8_to_RGB1555[i]=(i>>3)&0x001f;
  }
}

static void init_grey_to_RGB1555()
{
  init_color_to_RGB1555();
  if (from_Grey8_to_RGB1555[0xff])
    return;
  for (int i=0;i<0x100;i++) {
    from_Grey8_to_RGB1555[i] = (from_Red8_to_RGB1555[i]) |
      (from_Green8_to_RGB1555[i]) |
      (from_Blue8_to_RGB1555[i]);
  }
}
  
static void init_color_to_RGB565()
{
  if (from_Red8_to_RGB565[0xff]) 
    return;
  for (int i=0;i<0x100;i++) {
    from_Red8_to_RGB565[i]=(i<<8)&0xf800;
    from_Green8_to_RGB565[i]=(i<<3)&0x07e0;
    from_Blue8_to_RGB565[i]=(i>>3)&0x001f;
  }
}

static void init_grey_to_RGB565()
{
  init_color_to_RGB565();
  if (from_Grey8_to_RGB565[0xff])
    return;
  for (int i=0;i<0x100;i++) {
    from_Grey8_to_RGB565[i] = (from_Red8_to_RGB565[i]) |
      (from_Green8_to_RGB565[i]) |
      (from_Blue8_to_RGB565[i]);
  }
}

class ColorConverter {
public:
  ColorConverter(void (*converter)(ColorConverter*,
                                   unsigned char*, unsigned char*),
                 void (*putter)(ColorConverter*,
                                unsigned char*, int, int, ImgDisplay::Color)) {
    _converter = converter;
    _putter = putter;
  }

  void init(int lines, int pixel_width, int in_bytes_per_pixel,
            int out_bytes_per_line, int out_bytes_per_pixel) {
    _lines = lines;
    _in_bytes_per_pixel = in_bytes_per_pixel;
    _pixel_width = pixel_width;
    _in_bytes_per_line = _in_bytes_per_pixel*_pixel_width;
    _in_size = _in_bytes_per_line*_lines;
    _out_bytes_per_pixel = out_bytes_per_pixel;
    _out_bytes_per_line = out_bytes_per_line;
    _out_size = _out_bytes_per_line*_lines;
  }

  // default convert which only works for identical formats
  void convert(unsigned char* in, unsigned char* out) {
    if (_converter)
      (*_converter)(this, in, out);
  }

  void putPixel(unsigned char* out, int x, int y, ImgDisplay::Color color) {
    if (_putter)
      (*_putter)(this, out, x, y, color);
  }

  int outBytesPerLine() const { return _out_bytes_per_line; }
  int inBytesPerLine() const { return _in_bytes_per_line; }
  int numLines() const { return _lines; }
  int pixelWidth() const { return _pixel_width; }
  int outBytesPerPixel() const { return _out_bytes_per_pixel; }
  int inBytesPerPixel() const { return _in_bytes_per_pixel; }
  int inSize() const { return _in_size; }
  int outSize() const { return _out_size; }

protected:
  void (*_converter)(ColorConverter*,
                     unsigned char*, unsigned char*);
  void (*_putter)(ColorConverter*,
                  unsigned char*, int, int, ImgDisplay::Color);
  
  int _lines, _pixel_width;
  int _in_bytes_per_pixel, _out_bytes_per_pixel;
  int _in_bytes_per_line, _out_bytes_per_line;
  int _in_size, _out_size;
};

static void copy_converter(ColorConverter* converter,
                           unsigned char* in, unsigned char* out)
{
  if (converter->outBytesPerLine() == converter->inBytesPerLine()) {
    memcpy(out, in, converter->outSize());
  } else {
    for (int j=0;j<converter->numLines();j++) {
      memcpy(out, in, converter->inBytesPerLine());
      out+=converter->outBytesPerLine();
      in+=converter->inBytesPerLine();
    }
  }
}

static void grey8_putter(ColorConverter* converter, unsigned char* dst,
                         int x, int y, ImgDisplay::Color color)
{
  unsigned char* ptr =
    dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine());

  switch (color) {
  case ImgDisplay::Black:
    *ptr = 0; break;
  case ImgDisplay::White:
    *ptr = 0xff; break;
  case ImgDisplay::Red:  
  case ImgDisplay::Orange:
  case ImgDisplay::Yellow: 
  case ImgDisplay::Green:  
  case ImgDisplay::Blue:   
  case ImgDisplay::Purple: 
    *ptr = 0x80; break;
  case ImgDisplay::Pink: 
  case ImgDisplay::LtBlue:
    *ptr = 0xa0; break;
  default: break;
  }
}

static void bgr24_putter(ColorConverter* converter, unsigned char* dst,
                         int x, int y, ImgDisplay::Color color)
{
  unsigned char* ptr =
    dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine());

  switch (color)
    {
    case ImgDisplay::Black:
      *(ptr+0) = 0; *(ptr+1) = 0; *(ptr+2) = 0; break;
    case ImgDisplay::White:
      *(ptr+0) = 0xff; *(ptr+1) = 0xff; *(ptr+2) = 0xff; break;
    case ImgDisplay::Red:
      *(ptr+0) = 0; *(ptr+1) = 0; *(ptr+2) = 0xff; break;
    case ImgDisplay::Orange:
      *(ptr+0) = 0; *(ptr+1) = 0x80; *(ptr+2) = 0xff; break;
    case ImgDisplay::Yellow:
      *(ptr+0) = 0; *(ptr+1) = 0xff; *(ptr+2) = 0xff; break;
    case ImgDisplay::Green:
      *(ptr+0) = 0; *(ptr+1) = 0xff; *(ptr+2) = 0; break;
    case ImgDisplay::Blue:
      *(ptr+0) = 0xff; *(ptr+1) = 0; *(ptr+2) = 0; break;
    case ImgDisplay::Purple:
      *(ptr+0) = 0xff; *(ptr+1) = 0; *(ptr+2) = 0xc0; break;
    case ImgDisplay::Pink:
      *(ptr+0) = 0xff; *(ptr+1) = 0; *(ptr+2) = 0xff; break;
    case ImgDisplay::LtBlue:
      *(ptr+0) = 0xff; *(ptr+1) = 0xff; *(ptr+2) = 0; break;
    default: break;
    }
}

static void bgr1555_putter(ColorConverter* converter, unsigned char* dst,
                           int x, int y, ImgDisplay::Color color)
{
  unsigned short* ptr = (unsigned short*)
    (dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine()));

  switch (color)
    {
    case ImgDisplay::Black: *ptr = 0; break;
    case ImgDisplay::White: *ptr = 0xffffffff; break;
    case ImgDisplay::Red: *ptr = from_Red8_to_BGR1555[0xff]; break;
    case ImgDisplay::Orange: *ptr = from_Red8_to_BGR1555[0xff] |
                               from_Green8_to_BGR1555[0x80]; break;
    case ImgDisplay::Yellow: *ptr = from_Red8_to_BGR1555[0xff] |
                               from_Green8_to_BGR1555[0xff]; break;
    case ImgDisplay::Green: *ptr = from_Green8_to_BGR1555[0xff]; break;
    case ImgDisplay::Blue: *ptr = from_Blue8_to_BGR1555[0xff]; break;
    case ImgDisplay::Purple: *ptr = from_Red8_to_BGR1555[0xc0] |
                               from_Blue8_to_BGR1555[0xff]; break;

    case ImgDisplay::Pink: *ptr = from_Red8_to_BGR1555[0xff] |
                             from_Blue8_to_BGR1555[0xff]; break;
    case ImgDisplay::LtBlue: *ptr = from_Green8_to_BGR1555[0xff] |
                               from_Blue8_to_BGR1555[0xff]; break;
    default: break;
    }
}

static void bgr565_putter(ColorConverter* converter, unsigned char* dst,
                          int x, int y, ImgDisplay::Color color) 
{
  unsigned short* ptr = (unsigned short*)
    (dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine()));

  switch (color)
    {
    case ImgDisplay::Black: *ptr = 0; break;
    case ImgDisplay::White: *ptr = 0xffffffff; break;
    case ImgDisplay::Red: *ptr = from_Red8_to_BGR565[0xff]; break;
    case ImgDisplay::Orange: *ptr = from_Red8_to_BGR565[0xff] |
                               from_Green8_to_BGR565[0x80]; break;
    case ImgDisplay::Yellow: *ptr = from_Red8_to_BGR565[0xff] |
                               from_Green8_to_BGR565[0xff]; break;
    case ImgDisplay::Green: *ptr = from_Green8_to_BGR565[0xff]; break;
    case ImgDisplay::Blue: *ptr = from_Blue8_to_BGR565[0xff]; break;
    case ImgDisplay::Purple: *ptr = from_Red8_to_BGR565[0xc0] |
                               from_Blue8_to_BGR565[0xff]; break;

    case ImgDisplay::Pink: *ptr = from_Red8_to_BGR565[0xff] |
                             from_Blue8_to_BGR565[0xff]; break;
    case ImgDisplay::LtBlue: *ptr = from_Green8_to_BGR565[0xff] |
                               from_Blue8_to_BGR565[0xff]; break;
    default: break;
    }
}

static void bgr24_to_bgr1555(ColorConverter* converter,
                             unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Blue8_to_BGR1555[*(src)]) |
            (from_Green8_to_BGR1555[*(src+1)]) |
            (from_Red8_to_BGR1555[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void color24_to_grey8(ColorConverter* converter, 
                             unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int bpol = converter->outBytesPerLine();

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(unsigned char)
            ((int(*(src)) + int(*(src+1)) + int(*(src+2)))/3*255);
          src+=3; 
        }
      dst+=bpol;
    }
}

static void bgr24_to_bgr565(ColorConverter* converter, 
                            unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Blue8_to_BGR565[*(src)]) |
            (from_Green8_to_BGR565[*(src+1)]) |
            (from_Red8_to_BGR565[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void swap_color24(ColorConverter* converter,
                         unsigned char* in, unsigned char* out)
{
  int obbl = converter->outBytesPerLine();
  unsigned char* row = out;

  for (int j=0;j<converter->numLines();j++) {
    for (int i=0;i<converter->pixelWidth();i++) {
      *(row+2) = *in++;
      *(row+1) = *in++;
      *(row)   = *in++;
      row+= 3;
    }
    out += obbl;
  }
}

static void bgr24_to_rgb1555(ColorConverter* converter, 
                            unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Red8_to_BGR1555[*(src)]) |
            (from_Green8_to_BGR1555[*(src+1)]) |
            (from_Blue8_to_BGR1555[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void bgr24_to_rgb565(ColorConverter* converter, 
                            unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Red8_to_BGR565[*(src)]) |
            (from_Green8_to_BGR565[*(src+1)]) |
            (from_Blue8_to_BGR565[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void grey8_to_bgr1555(ColorConverter* converter, 
                             unsigned char* in, unsigned char* out)
{
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;
  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i) = from_Grey8_to_BGR1555[ *(in++)]; 
        }
      dst+=wpol;
    }
}

static void grey8_to_color565(ColorConverter* converter,
                              unsigned char* in, unsigned char* out)
{
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;
  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i) = from_Grey8_to_RGB565[ *(in++)]; 
        }
      dst+=wpol;
    }
}

static void grey8_to_color24(ColorConverter* converter, 
                             unsigned char* in, unsigned char* out)
{
  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(out+(i*3)+0) = *in;
          *(out+(i*3)+1) = *in;
          *(out+(i*3)+2) = *in++;
        }
      out+=converter->outBytesPerLine();
    }
}

static void rgb24_putter(ColorConverter* converter, unsigned char* dst,
                         int x, int y, ImgDisplay::Color color)
{
  unsigned char* ptr =
    dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine());

  switch (color)
    {
    case ImgDisplay::Black:
      *(ptr+0) = 0; *(ptr+1) = 0; *(ptr+2) = 0; break;
    case ImgDisplay::White:
      *(ptr+0) = 0xff; *(ptr+1) = 0xff; *(ptr+2) = 0xff; break;
    case ImgDisplay::Red:
      *(ptr+0) = 0xff; *(ptr+1) = 0; *(ptr+2) = 0x00; break;
    case ImgDisplay::Orange:
      *(ptr+0) = 0xff; *(ptr+1) = 0x80; *(ptr+2) = 0x00; break;
    case ImgDisplay::Yellow:
      *(ptr+0) = 0xff; *(ptr+1) = 0xff; *(ptr+2) = 0x00; break;
    case ImgDisplay::Green:
      *(ptr+0) = 0; *(ptr+1) = 0xff; *(ptr+2) = 0; break;
    case ImgDisplay::Blue:
      *(ptr+0) = 0x0; *(ptr+1) = 0; *(ptr+2) = 0xff; break;
    case ImgDisplay::Purple:
      *(ptr+0) = 0xc0; *(ptr+1) = 0; *(ptr+2) = 0xff; break;
    case ImgDisplay::Pink:
      *(ptr+0) = 0xff; *(ptr+1) = 0; *(ptr+2) = 0xff; break;
    case ImgDisplay::LtBlue:
      *(ptr+0) = 0x0; *(ptr+1) = 0xff; *(ptr+2) = 0xff; break;
    default: break;
    }
}

static void rgb1555_putter(ColorConverter* converter, unsigned char* dst,
                           int x, int y, ImgDisplay::Color color) {
  unsigned short* ptr = (unsigned short*)
    (dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine()));

  switch (color)
    {
    case ImgDisplay::Black: *ptr = 0; break;
    case ImgDisplay::White: *ptr = 0xffffffff; break;
    case ImgDisplay::Red: *ptr = from_Red8_to_RGB1555[0xff]; break;
    case ImgDisplay::Orange: *ptr = from_Red8_to_RGB1555[0xff] |
                               from_Green8_to_RGB1555[0x80]; break;
    case ImgDisplay::Yellow: *ptr = from_Red8_to_RGB1555[0xff] |
                               from_Green8_to_RGB1555[0xff]; break;
    case ImgDisplay::Green: *ptr = from_Green8_to_RGB1555[0xff]; break;
    case ImgDisplay::Blue: *ptr = from_Blue8_to_RGB1555[0xff]; break;
    case ImgDisplay::Purple: *ptr = from_Red8_to_RGB1555[0xc0] |
                               from_Blue8_to_RGB1555[0xff]; break;

    case ImgDisplay::Pink: *ptr = from_Red8_to_RGB1555[0xff] |
                             from_Blue8_to_RGB1555[0xff]; break;
    case ImgDisplay::LtBlue: *ptr = from_Green8_to_RGB1555[0xff] |
                               from_Blue8_to_RGB1555[0xff]; break;
    default: break;
    }
}

static void rgb565_putter(ColorConverter* converter, unsigned char* dst,
                          int x, int y, ImgDisplay::Color color) {
  unsigned short* ptr = (unsigned short*)
    (dst + (x*converter->outBytesPerPixel())+(y*converter->outBytesPerLine()));

  switch (color)
    {
    case ImgDisplay::Black: *ptr = 0; break;
    case ImgDisplay::White: *ptr = 0xffffffff; break;
    case ImgDisplay::Red: *ptr = from_Red8_to_RGB565[0xff]; break;
    case ImgDisplay::Orange: *ptr = from_Red8_to_RGB565[0xff] |
                               from_Green8_to_RGB565[0x80]; break;
    case ImgDisplay::Yellow: *ptr = from_Red8_to_RGB565[0xff] |
                               from_Green8_to_RGB565[0xff]; break;
    case ImgDisplay::Green: *ptr = from_Green8_to_RGB565[0xff]; break;
    case ImgDisplay::Blue: *ptr = from_Blue8_to_RGB565[0xff]; break;
    case ImgDisplay::Purple: *ptr = from_Red8_to_RGB565[0xc0] |
                               from_Blue8_to_RGB565[0xff]; break;

    case ImgDisplay::Pink: *ptr = from_Red8_to_RGB565[0xff] |
                             from_Blue8_to_RGB565[0xff]; break;
    case ImgDisplay::LtBlue: *ptr = from_Green8_to_RGB565[0xff] |
                               from_Blue8_to_RGB565[0xff]; break;
    default: break;
    }
}

static void rgb24_to_rgb1555(ColorConverter* converter,
                             unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Red8_to_RGB1555[*(src)]) |
            (from_Green8_to_RGB1555[*(src+1)]) |
            (from_Blue8_to_RGB1555[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void rgb24_to_rgb565(ColorConverter* converter, 
                            unsigned char* in, unsigned char* out)
{
  unsigned char* src = in;
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;

  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i)=(from_Red8_to_RGB565[*(src)]) |
            (from_Green8_to_RGB565[*(src+1)]) |
            (from_Blue8_to_RGB565[*(src+2)]);
          src+=3; 
        }
      dst+=wpol;
    }
}

static void grey8_to_rgb1555(ColorConverter* converter,
                             unsigned char* in, unsigned char* out)
{
  unsigned short* dst = (unsigned short*)out;
  int wpol = converter->outBytesPerLine()/2;
  for (int j=0;j<converter->numLines();j++)
    {
      for (int i=0;i<converter->pixelWidth();i++)
        {
          *(dst+i) = from_Grey8_to_RGB1555[ *(in++)]; 
        }
      dst+=wpol;
    }
}


ColorConverter* create_converter(ImgDisplay::ColorType src,
                                 ImgDisplay::ColorType dest)
{
  //  printf("From %s to %s\n", ImgDisplay::colorTypeToString(src), 
  //     ImgDisplay::colorTypeToString(dest));
  if (dest == ImgDisplay::Grey8) {
    if (src == ImgDisplay::Grey8) {
      return new ColorConverter(copy_converter, grey8_putter);
    } else if (src == ImgDisplay::RGB24 || src == ImgDisplay::BGR24) {
      return new ColorConverter(color24_to_grey8, grey8_putter);
    }

  } else if (dest == ImgDisplay::BGR24) {
    if (src == ImgDisplay::BGR24) {
      return new ColorConverter(copy_converter, bgr24_putter);
    } else if (src == ImgDisplay::Grey8) {
      return new ColorConverter(grey8_to_color24, bgr24_putter);
    } else if (src == ImgDisplay::RGB24) {
      return new ColorConverter(swap_color24, bgr24_putter);
    }


  } else if (dest == ImgDisplay::BGR1555) {
    if (src == ImgDisplay::BGR1555) {
      return new ColorConverter(copy_converter, bgr1555_putter);
    } else if (src == ImgDisplay::BGR24) {
      init_color_to_BGR1555();
      return new ColorConverter(bgr24_to_bgr1555, bgr1555_putter);
    } else if (src == ImgDisplay::Grey8) {
      init_grey_to_BGR1555();
      return new ColorConverter(grey8_to_bgr1555, bgr1555_putter);
    }

  } else if (dest == ImgDisplay::BGR565) {
    if (src == ImgDisplay::BGR565) {
      return new ColorConverter(copy_converter, bgr565_putter);
    } else if (src == ImgDisplay::BGR24) {
      init_color_to_BGR565();
      return new ColorConverter(bgr24_to_bgr565, bgr565_putter);
    } else if (src == ImgDisplay::Grey8) {
      init_grey_to_RGB565();
      return new ColorConverter(grey8_to_color565, bgr565_putter);
    }

  } else if (dest == ImgDisplay::RGB24) {
    if (src == ImgDisplay::RGB24) {
      return new ColorConverter(copy_converter, rgb24_putter);
    } else if (src == ImgDisplay::Grey8) {
      return new ColorConverter(grey8_to_color24, rgb24_putter);
    } else if (src == ImgDisplay::BGR24) {
      return new ColorConverter(swap_color24, rgb24_putter);
    }

  } else if (dest == ImgDisplay::RGB1555) {
    if (src == ImgDisplay::RGB1555) {
      return new ColorConverter(copy_converter, rgb1555_putter);
    } else if (src == ImgDisplay::RGB24) {
      init_color_to_RGB1555();
      return new ColorConverter(rgb24_to_rgb1555, rgb1555_putter);
    } else if (src == ImgDisplay::Grey8){
      init_grey_to_RGB1555();
      return new ColorConverter(grey8_to_rgb1555, rgb1555_putter);
    } else if (src == ImgDisplay::BGR24) {
      init_color_to_RGB1555();
      init_color_to_BGR1555();
      return new ColorConverter(bgr24_to_rgb1555, rgb1555_putter);
    }


  } else if (dest == ImgDisplay::RGB565) {
    if (src == ImgDisplay::RGB565) {
      return new ColorConverter(copy_converter, rgb565_putter);
    } else if (src == ImgDisplay::RGB24) {
      init_color_to_RGB565();
      return new ColorConverter(rgb24_to_rgb565, rgb565_putter);
    } else if (src == ImgDisplay::Grey8) {
      init_grey_to_RGB565();
      return new ColorConverter(grey8_to_color565, rgb565_putter);
    } else if (src == ImgDisplay::BGR24) {
      init_color_to_RGB565();
      init_color_to_BGR565();
      return new ColorConverter(bgr24_to_rgb565, rgb565_putter);
    }
  }

  printf("BufferDisplay: cannot convert between %s and %s\n",
         ImgDisplay::colorTypeToString(src),
         ImgDisplay::colorTypeToString(dest));
  return NULL;
}

static int bytes_per_pixel[] = {
  1, /*    Grey8,          !< 8-bit greyscale */
  2, /*    Grey16,         !< 16-bit greyscale */
  4, /*    Grey32,         !< 32-bit greyscale */

  2, /*    RGB1555,                !< 15-bit RGB (bit 15 unused) */
  2, /*    RGB5155,                !< 15-bit RGB (bit 9 unused) */
  2, /*    RGB5515,                !< 15-bit RGB (bit 6 unused) */
  2, /*    RGB5551,                !< 15-bit RGB (bit 0 unused) */

  2, /*    BGR1555,                !< 15-bit BGR (bit 15 unused) */
  2, /*    BGR5155,                !< 15-bit BGR (bit 9 unused) */
  2, /*    BGR5515,                !< 15-bit BGR (bit 6 unused) */
  2, /*    BGR5551,                !< 15-bit BGR (bit 0 unused) */

  2, /*    RGB565,         !< 16-bit RGB */
  2, /*    BGR565,         !< 16-bit BGR */

  3, /*    RGB24,          !< 24-bit RGB */
  3, /*    BGR24,          !< 24-bit BGR */

  4, /*    MSBPadded_RGB32,        !< 32-bit RGB (MSB unused) */
  4, /*    LSBPadded_RGB32, !< 32-bit RGB (LSB unused) */

  4, /*    MSBPadded_BGR32,        !< 32-bit BGR (MSB unused) */
  4, /*    LSBPadded_BGR32, !< 32-bit BGR (MSB unused) */
  
  8, /*    YUV422_Planar,  !< Planar YUV422 */
  8, /*    YUV422_Packed,  !< Packed YUV422 */
  0, /*    LastColorType, */
  0  /*    UnknownColorType */
};

BufferDisplay::~BufferDisplay()
{
  destroyAllWindows();
}

void BufferDisplay::addWindow(Window* win)
{
  _windows.append(win);
}

void BufferDisplay::destroyWindow(Window* win)
{
  _windows.remove(win);
  delete win;
}

void BufferDisplay::destroyAllWindows()
{
  while (Window* w = _windows.pop()) 
    delete w;
}

int BufferWindow::getBytesPerPixel(ImgDisplay::ColorType type)
{
  if (type < 0 || type >= ImgDisplay::LastColorType)
    return 0;
  return bytes_per_pixel[(int) type];
} 

BufferWindow::BufferWindow(ImgDisplay* display, int width, int height,
                           int dest_bytes_per_line,
                           ImgDisplay::ColorType src_type,
                           ImgDisplay::ColorType dest_type)
  : ImgDisplay::Window(display, width, height)
{
  printf("Buffer: Converting from %s to %s\n", 
         ImgDisplay::colorTypeToString(src_type),
         ImgDisplay::colorTypeToString(dest_type));
  _src_type = src_type;
  _dest = NULL;
  _dest_type = dest_type;
  _converter = create_converter(src_type, dest_type);
  _src_bytes_per_pixel = getBytesPerPixel(src_type);

  if (_converter) {
    _converter->init(height, width, getBytesPerPixel(src_type),
                     dest_bytes_per_line, getBytesPerPixel(dest_type));
  }

  _dest = NULL;
}

void BufferWindow::pixel(int x, int y, ImgDisplay::Color colorID)
{
  if (x < 0 || y < 0 || x >= getWidth() || y >= getHeight())
    return;
  if (_converter && _dest)
    _converter->putPixel(_dest, x, y, colorID);
}

void BufferWindow::clear()
{
  if (_converter && _dest) {
    memset(_dest, 0, getHeight()*_converter->outBytesPerLine());
  }
}

void BufferWindow::fill(unsigned char* src)
{
  if (_converter && _dest) {
    _converter->convert(src, _dest);
  }
}
  
