#include <utils/ConfigElem.h>
#include <utils/UnknownElem.h>
#include <utils/StructElem.h>
#include <utils/Output.h>
#include <utils/Input.h>

#include <ConfigSource/Active.h>

#include <CorbaUtils.h>

#include "RepositoryS.h"
#include "RepositoryCommon.h"

class Server_i;
class ServerConfigSource : public ActiveConfigSource
{
public:
  ServerConfigSource(utils::SymbolTable* table) { _table = table; }
  virtual ~ServerConfigSource();
    
  virtual bool processEvents(double timeout=-1);
  virtual bool startServer(int port);

private:
  utils::SymbolTable* _table;
  Repository::Server_var _repository;
  Server_i* _repository_i;
};

class Server_i : public POA_Repository::Server {
 public:
  Server_i(ActiveConfigSource*);

  virtual void get (const char * name,
                    CORBA::String_out type, CORBA::String_out value)
    throw (Repository::AccessError);
  virtual void parseSet (const char *, const char*)
    throw (Repository::AccessError);
  void Server_i::set(const char* type, const char *name, const char* value)
    throw (Repository::AccessError);
  virtual Repository::CBRegistration_ptr
    watch(const char*, Repository::Callback_ptr,
          CORBA::String_out, CORBA::String_out)
    throw (Repository::AccessError);
  virtual int numValues (const char * name)
    throw (Repository::AccessError);

 private:
  ActiveConfigSource* _source;
  utils::ConfigFile _file;
};

class CBRegistration_i : public POA_Repository::CBRegistration {
public:
  CBRegistration_i(ActiveConfigSource* source, const char* name,
                   Repository::Callback_ptr callback) {
    _source = source; _name = name; _watch = NULL;
    _callback = Repository::Callback::_duplicate(callback);
  }

  void setWatch(ActiveElem* watch) { _watch = watch; }

  virtual void unregister() throw (CORBA::SystemException) {
    CorbaUtils::deactivateServant(this);
    utils::ConfigElem* elem = _source->getFile().lookup(_name.getString());
    _source->unwatch(elem, _watch);
    delete this;
  }

  void callback(const char* name) {
    utils::ConfigElem* elem = _source->getFile().lookup(_name.getString());
    if (!elem) 
      return;

    utils::Output output;
    char* res = new char[50];
    output.setBuffer(res, 50, utils::Output::standardResize);
    elem->writeData(output);
    output.write('\0');
    void* out_buf;
    int size;
    output.getBuffer(out_buf, size);

    try {
      _callback->notify(name, elem->getTypeId().getName().getString(),
                        (char*) out_buf);
    } catch (CORBA::Exception &e) {
      char buffer[200];
      printf("Callback for '%s' failed: '%s'\n", name,
             CorbaUtils::error(e, buffer, 200));
      unregister();
    }
    delete [] (char*) out_buf;
  }

private:
  utils::String _name;
  ActiveConfigSource* _source;
  ActiveElem* _watch;
  Repository::Callback_var _callback;
};

ConfigSource* create_ConfigSource_server(ConfigSourceGenerator*,
                                         utils::ConfigFile* params,
                                         utils::SymbolTable* table)
{
  ServerConfigSource* intf = new ServerConfigSource(table);
  if (!intf->open(params->getString("name", "main.conf")) ||
      !intf->startServer(params->getInt("port", DEFAULT_REPOSITORY_PORT))) {
    delete intf;
    return NULL;
  }
  return intf;
}

ServerConfigSource::~ServerConfigSource()
{
  CorbaUtils::deactivateServant(_repository_i);
  delete _repository_i;
  CorbaUtils::releaseOrb(_table);
}

bool ServerConfigSource::processEvents(double timeout=-1)
{
  return CorbaUtils::process(timeout, _table);
}

bool ServerConfigSource::startServer(int port)
{
  try {
    if (CORBA::is_nil(CorbaUtils::createServer(_table, port)))
      return false;
    CorbaUtils::refOrb(_table);

    _repository_i = new Server_i(this);
    _repository = _repository_i->_this();

    CorbaUtils::nameObject(_table, _repository.in(), REPOSITORY_NAME);
  
    return true;
  } catch (CORBA::Exception &e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
 } 
  return false;
}

Server_i::Server_i(ActiveConfigSource* src)
{
  _source = src;
  _file = _source->getFile();
}

void Server_i::get (const char * name,
                    CORBA::String_out type, CORBA::String_out value)
  throw (Repository::AccessError)
{
  utils::ConfigElem* elem = _file.lookup(name);
  if (!elem) {
    elem = _file.makeElem(name);
    if (!elem)
      throw Repository::AccessError();
  }

  utils::Output output;
  char* res = new char[50];
  output.setBuffer(res, 50, utils::Output::standardResize);
  elem->writeData(output);
  output.write('\0');

  void* out_buf;
  int size;
  output.getBuffer(out_buf, size);

  value = (char*) out_buf;
  type = utils::String::copy(elem->getTypeId().getName().getString());
}

int Server_i::numValues(const char* name)
  throw (Repository::AccessError)
{
  utils::ConfigElem* elem = _file.lookup(name);
  if (!elem)
    throw Repository::AccessError();

  return elem->numValues();
}

void Server_i::parseSet(const char *name, const char* value)
    throw (Repository::AccessError)
{
  if (!_file.set(name, value))
    throw Repository::AccessError();
}

void Server_i::set(const char* type, const char *name, const char* value)
    throw (Repository::AccessError)
{
  utils::ConfigElem* elem =
    ((utils::StructElem*) _file.getRoot())->makeElem(name, true);
  if (!elem)
    throw Repository::AccessError();

  utils::Type t = utils::Type::fromName(type);
  utils::ConfigElem* setter = (utils::ConfigElem*) t.createInstance();
  if (!setter ||
      !setter->getTypeId().isDerivedFrom(utils::ConfigElem::getClassTypeId()))
    throw Repository::AccessError();

  utils::Input input;
  input.setBuffer((void*) value, strlen(value));
  if (!setter->readData(input))
    throw Repository::AccessError();

  if (!_file.set(t, name, setter->getData(), setter->numValues()))
    throw Repository::AccessError();
}

static void watch_cb(const char* name, utils::Type, void*, int, void* cbdata)
{
  ((CBRegistration_i*) cbdata)->callback(name);
}

Repository::CBRegistration_ptr
Server_i::watch(const char* name, Repository::Callback_ptr callback,
                CORBA::String_out type, CORBA::String_out value)
  throw (Repository::AccessError)
{
  get(name, type, value);
  CBRegistration_i* reg = new CBRegistration_i(_source, name, callback);
  ActiveElem* watch = _source->watch(utils::Type::badType(), name,
                                     NULL, 0, watch_cb, reg, NULL, 0);
  if (!watch) {
    delete reg;
    throw Repository::AccessError();
  }
  reg->setWatch(watch);
  return reg->_this();
}
