#include <stdio.h>
#include <string.h>

#include <utils/ConfigFile.h>
#include <utils/Generator.h>

#include <ConfigSource/Embedded.h>

ConfigSource* create_ConfigSource_embedded(ConfigSourceGenerator* gen,
                                           utils::ConfigFile* params,
                                           utils::SymbolTable* table)
{
  const char* name = params->getString("name");
  if (!*name) {
    printf("EmbeddedConfigSource: Need name\n");
    return NULL;
  }

  const char* spec = params->getString("container", "default");
  ConfigSource* container = gen->interface(spec, table);
  if (!container) {
    printf("EmbeddedConfigSource: Invalid container '%s'\n", spec);
    return NULL;
  }
    
  EmbeddedConfigSource* cs = new EmbeddedConfigSource(container, name);
  if (!cs->initialize()) {
    printf("EmbeddedConfigSource: Could not initialize\n");
    delete cs;
    return NULL;
  }
  return cs;
}

EmbeddedConfigSource::EmbeddedConfigSource(ConfigSource* container,
                                           const char* name)
{
  _container = container;
  _container->ref();
  _name = name;
}

EmbeddedConfigSource::~EmbeddedConfigSource()
{
  _container->unref();
}

bool EmbeddedConfigSource::initialize()
{
  utils::ConfigFile& file = getFile();
  const char* name = _name.getString();
  utils::ConfigFile src;
  if (!_container->getStruct(name, src))
    return false;
  
  utils::ConfigFile::copy(src, file);

  return true;
}

const char* EmbeddedConfigSource::make_full_name(utils::String& full_name,
                                                 const char* name)
{
  full_name = _name;
  full_name += ".";
  full_name += name;
  return full_name.getString();
}

struct EmbeddedCallbackData {
  EmbeddedConfigSource* source;
  ConfigCallback callback;
  void* cbdata;
};

void embedded_callback(const char* name, utils::Type type,
                       void* data, int num_elems, void* cbdata)
{
  EmbeddedCallbackData* ecd = (EmbeddedCallbackData*) cbdata;
  ecd->source->doCallback(ecd, name, type, data, num_elems);
}

bool EmbeddedConfigSource::attach(utils::Type type, const char* name,
                                  void* data, int max_num,
                                  ConfigCallback cb, void* cb_data,
                                  void* default_data, int num_defaults)
{
  utils::String full_name;
  // note this is never deleted - could be considered a memory leak
  EmbeddedCallbackData* ecd = new EmbeddedCallbackData;
  ecd->source = this;
  ecd->callback = cb;
  ecd->cbdata = cb_data;
  return _container->attach(type, make_full_name(full_name, name),
                            data, max_num, embedded_callback, ecd,
                            default_data, num_defaults);
}
    
bool EmbeddedConfigSource::set(const char* name, const char* value)
{
  utils::String full_name;
  if (!_container->set(make_full_name(full_name, name), value))
    return false;
  return _file.set(name, value);
}

bool EmbeddedConfigSource::set(utils::Type type, const char* name, void* data,
                               int num_values)
{
  utils::String full_name;
  if (!_container->set(type, make_full_name(full_name, name),
                       data, num_values))
    return false;
  return _file.set(type, name, data, num_values);
}

bool EmbeddedConfigSource::processEvents(double timeout)
{
  return _container->processEvents(timeout);
}

void EmbeddedConfigSource::doCallback(EmbeddedCallbackData* ecd, 
                                      const char* name, utils::Type type,
                                      void* data, int num_elems)
{
  const char* short_name = strchr(name, '.');
  if (!short_name)
    short_name = name;
  else {
    short_name++;
  }
  _file.set(type, short_name, data, num_elems);
  if (ecd->callback)
    ecd->callback(short_name, type, data, num_elems, ecd->cbdata);
}

bool EmbeddedConfigSource::refresh()
{
  _container->refresh();
  return initialize();
}
    
void EmbeddedConfigSource::reportState(ModuleState state)
{
  _container->reportState(state);
}

void EmbeddedConfigSource::setStatus(float status)
{
  _container->setStatus(status);
}

void EmbeddedConfigSource::setConfidence(float confidence)
{
  _container->setConfidence(confidence);
}

void EmbeddedConfigSource::setStatusMessage(const char* msg)
{
  _container->setStatusMessage(msg);
}

