#include <utils/ConfigFile.h>
#include <utils/ConfigElem.h>
#include <utils/UnknownElem.h>
#include <utils/Dict.h>
#include <utils/List.h>
#include <utils/Input.h>
#include <utils/Output.h>

#include <ConfigSource/ConfigSource.h>

#include <CorbaUtils.h>

#include "RepositoryS.h"
#include "RepositoryCommon.h"

class Callback_i;
class ClientConfigSource : public ConfigSource {
public:
  ClientConfigSource(utils::SymbolTable* table) {
    _table = table;
  }
  virtual ~ClientConfigSource();

  bool init(const char* host, int port, const char* name);
  
  virtual int get(utils::Type, const char* name, void* data, int max_num=1,
                  void* default_data=NULL, int num_defaults=1);

  virtual bool attach(utils::Type, const char* name, void* data, int max_num=1,
                      ConfigCallback notify=NULL, void* cbdata = NULL,
                      void* default_data=NULL, int num_defaults=1);

  virtual int numValues(const char* name);

  virtual bool set(const char* name, const char* value);
  virtual bool set(utils::Type, const char* name, void* data,
                   int num_values);

  virtual bool processEvents(double timeout=-1) ;
  virtual bool refresh() { return true; }
  utils::ConfigElem* putData(utils::Type t, 
                             const char* in_type_name, const char* in_value);

private:
  utils::ConfigElem* getElem(utils::Type);

private:
  utils::SymbolTable* _table;
  Repository::Server_var _repository;
  utils::Dict<utils::ConfigElem*> _unpack_elems;
  utils::ManagedList<Callback_i> _callbacks;
};

class Callback_i : public POA_Repository::Callback {
public:
  Callback_i(ClientConfigSource* src,
             utils::Type out_type, void* data, int max_num,
             ConfigCallback notify, void* cbdata) {
    _src = src;
    _out_type = out_type; _out_data = data; _out_max_num = max_num;
    _notify_cb = notify; _notify_cb_data = cbdata;
    _callback = this->_this();
  }
  virtual ~Callback_i() {
    try {
      if (!CORBA::is_nil(_registration.in()))
        _registration->unregister();
    } catch (CORBA::Exception &e) {
    }
    CorbaUtils::deactivateServant(this);
  }
  
  virtual void notify(const char* name, const char* type, const char* value)
    throw (CORBA::SystemException) {
    utils::ConfigElem* elem = _src->putData(_out_type, type, value);
    if (elem) {
      if (_out_data && _out_max_num)
        elem->getValue(_out_type, _out_data, _out_max_num);
      if (_notify_cb) 
        (*_notify_cb)(name, elem->getTypeId(),
                      elem->getData(), elem->numValues(), _notify_cb_data);
    }
  }

  void setRegistration(Repository::CBRegistration_ptr reg) {
    _registration = Repository::CBRegistration::_duplicate(reg);
  }

  Repository::Callback* callback() { return _callback.ptr(); }

public:
  ClientConfigSource* _src;
  utils::Type _out_type;
  void* _out_data;
  int _out_max_num;
  ConfigCallback _notify_cb;
  void* _notify_cb_data;
  Repository::CBRegistration_var _registration;
  Repository::Callback_var _callback;
};

ConfigSource* create_ConfigSource_client(ConfigSourceGenerator*,
                                         utils::ConfigFile* params,
                                         utils::SymbolTable* table)
{
  ClientConfigSource* src = new ClientConfigSource(table);
  if (!src->init(params->getString("host", CorbaUtils::hostname()),
                 params->getInt("port", DEFAULT_REPOSITORY_PORT),
                 params->getString("name", REPOSITORY_NAME))) {
    delete src;
    return NULL;
  }
  return src;
}

ClientConfigSource::~ClientConfigSource()
{
  utils::DictIterator<utils::ConfigElem*> iter(_unpack_elems);
  for (utils::ConfigElem* elem=iter.first(); elem; elem=iter.next()) 
    elem->unref();
  CorbaUtils::releaseOrb(_table);
}

bool ClientConfigSource::init(const char* host, int port, const char* name)
{
  try {
    if (CORBA::is_nil(CorbaUtils::createServer(_table)))
      return false;
    CorbaUtils::refOrb(_table);
    CORBA::Object_var repository_object =
      CorbaUtils::getNamedObject(_table, host, port, name);
    if (CORBA::is_nil(repository_object.in())) {
      return false;
    }
    _repository = Repository::Server::_narrow (repository_object.in ());

    return true;
  } catch (CORBA::Exception &e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
  }
  return false;
}

utils::ConfigElem* ClientConfigSource::getElem(utils::Type t)
{
  utils::ConfigElem* elem;

  if (_unpack_elems.find(t.getKey(), elem))
    return elem;

  elem = (utils::ConfigElem*) t.createInstance();
  if (!elem)
    return NULL;

  if (!elem->getTypeId().isDerivedFrom(utils::ConfigElem::getClassTypeId())) {
    delete elem;
    return NULL;
  }

  elem->ref();
  _unpack_elems.enter(t.getKey(), elem);

  return elem;
}

utils::ConfigElem*
ClientConfigSource::putData(utils::Type t, 
                            const char* in_type_name, const char* in_value)
{
  utils::Type in_type = utils::Type::fromName((const char*) in_type_name);
  utils::ConfigElem* elem = getElem(in_type);
  if (!elem) 
    elem = getElem(t);
  if (!elem) {
    printf("Invalid type '%s'\n", (const char*) in_type_name);
    return NULL;
  }

  utils::Input input;
  input.setBuffer((void*) (char*) in_value, strlen(in_value));
  if (!elem->readData(input)) {
    printf("Problem parsing '%s' as '%s'\n", (char*) in_value,
           t.getName().getString());
    return NULL;
  }
  return elem;
}

int ClientConfigSource::get(utils::Type t, const char* name, void* data,
                            int max_num, void* default_data, int num_defaults)
{
  utils::ConfigElem* elem=NULL;

  try {
    CORBA::String_var value, type_name;
    _repository->get(name, type_name, value);
    elem = putData(t, type_name, value);
  } catch (Repository::AccessError &) {
    if (!elem)
      elem = getElem(t);
    if (elem) {
      if (!elem->setValue(t, default_data, num_defaults))
        return 0;
    } 
  } catch (CORBA::Exception& e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
    return 0;
  }

  if (elem)
    return elem->getValue(t, data, max_num);
  else 
    return 0;
}

int ClientConfigSource::numValues(const char* name)
{
  try {
    return _repository->numValues(name);
  } catch (Repository::AccessError &) {
    return 0;
  } catch (CORBA::Exception& e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
    return 0;
  }
}

bool ClientConfigSource::processEvents(double timeout=-1)
{
  return CorbaUtils::process(timeout, _table);
}

bool ClientConfigSource::set(const char* name, const char* value)
{
  try {
    _repository->parseSet(name, value);
    return true;
  } catch (Repository::AccessError &) {
    return false;
  } catch (CORBA::Exception& e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
    return false;
  }
}  

bool ClientConfigSource::set(utils::Type out_type, const char* name,
                             void* data, int num_values)
{
  utils::ConfigElem* elem = getElem(out_type);
  if (!elem)
    return false;
  if (!elem->setValue(out_type, data, num_values))
    return false;
  
  utils::Output output;
  char* value = new char[50];
  output.setBuffer(value, 50, utils::Output::standardResize);
  elem->writeData(output);
  output.write('\0');
  int nbytes;
  void* out_buf;
  output.getBuffer(out_buf, nbytes);
  value = (char*) out_buf;

  bool res;
  try {
    _repository->set(out_type.getName().getString(), name, value);
    res = true;
  } catch (Repository::AccessError &) {
    res = false;
  } catch (CORBA::Exception& e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
    res = false;
  }

  delete [] value;
  return res;
}

bool ClientConfigSource::attach(utils::Type type, const char* name,
                                void* data, int max_num,
                                ConfigCallback notify, void* cbdata,
                                void* default_data, int num_defaults)
{
  Callback_i* callback = new Callback_i(this, type, data, max_num,
                                        notify, cbdata);
  utils::ConfigElem* elem = NULL;
  try {
    CORBA::String_var value, type_name;
    Repository::CBRegistration_var reg =
      _repository->watch(name, callback->callback(), type_name, value);
    callback->setRegistration(reg.ptr());
    elem = putData(type, type_name, value);
  } catch (Repository::AccessError &) {
    if (!elem && default_data)
      elem = getElem(type);
    if (elem && !elem->setValue(type, default_data, num_defaults))
      elem = NULL;
  } catch (CORBA::Exception& e) {
    char buffer[200];
    printf("CORBA Exception raised: '%s'\n",
           CorbaUtils::error(e, buffer, 200));
  }

  if (elem && data)
    elem->getValue(type, data, max_num);

  if (elem != NULL) {
    _callbacks.append(callback);
    return true;
  } else {
    delete callback;
    return false;
  }
}


