#include <stdio.h>
#include <ConfigSource/Active.h>
#include <utils/List.h>
#include <utils/ConfigElem.h>
#include <utils/StructElem.h>
#include <utils/ConfigWatch.h>

typedef utils::ListIterator<ActiveElem> ActiveListIter;

class ActiveList : public utils::List<ActiveElem> {
public:
  ~ActiveList() {
    utils::ListIterator<ActiveElem> iter(*this);
    for (ActiveElem* cur=iter.first(); cur; cur=iter.next())
      delete cur;
    clear();
  }
};

class SourceWatch : public utils::ConfigWatch
{
  UTILS_BASE_HEADER(SourceWatch);
public:
  SourceWatch(ActiveConfigSource* src = NULL);

  void init(utils::ConfigElem* elem, utils::StructElem* parent = NULL) {
    _elem = elem; _parent = parent;
  }

  virtual void set(utils::ConfigElem* elem) {
    ActiveListIter oiter(_callbacks);
    utils::List<ActiveElem> list;
    ActiveElem* active;
    for (active=oiter.first(); active; active=oiter.next()) {
      list.append(active);
    }
    ActiveListIter iter(list);
    int num;
    for (active=iter.first(); active; active=iter.next()) {
      if (active->data) {
        num = elem->getValue(active->type, active->data, active->max_num);
        if (num > 0 && active->callback)
          (*active->callback)(active->name, active->type, active->data, num,
                              active->cb_data);
      } else if (active->callback) 
        (*active->callback)(active->name, elem->getTypeId(), elem, 1,
                            active->cb_data);
    }
    if (_parent && _parent->watcher()) 
      _parent->watcher()->set(_parent);
  }

  virtual void add(utils::StructElem* parent, const char* name,
                   utils::ConfigElem* child) {
    utils::ListIterator<SourceWatch> iter(_children);
    for (SourceWatch* cur = iter.first(); cur; cur = iter.next()) {
      if (cur->getElem() == child)
        return;
    }
    utils::ConfigWatch* cw = child->watcher();
    SourceWatch* w;
    if (!cw || !cw->getTypeId().isDerivedFrom(SourceWatch::getClassTypeId())) {
      w = new SourceWatch(_config_source);
      child->setWatcher(w);
    } else {
      w = (SourceWatch*) cw;
    }
    w->init(child, parent);
    w->addTree();
  }

  utils::ConfigElem* getElem() const { return _elem; }
  void addTree() {
    if (_elem->getTypeId().isDerivedFrom(utils::StructElem::getClassTypeId())){
      utils::StructElem* s = (utils::StructElem*) _elem;
      utils::StringDictIterator<utils::ConfigElem*> iter(s->entries());
      for (utils::ConfigElem* cur = iter.first(); cur; cur = iter.next()) 
        add(s, iter.key(), cur);
    }
  }

  ActiveElem* addCallback(utils::Type type, const char* name,
                   void* data, int max_num,
                   ConfigCallback notify, void* cb_data) {
    ActiveElem* elem = new ActiveElem();
    elem->type = type;
    elem->name = utils::String::copy(name);
    // printf("Create %x %s\n", (unsigned) elem, elem->name);
    elem->data = data;
    elem->max_num = max_num;
    elem->callback = notify;
    elem->cb_data = cb_data;
    _callbacks.append(elem);
    return elem;
  }

  void appendCallback(ActiveElem* elem) {
    _callbacks.append(elem);
  }

  void removeCallback(ActiveElem* elem) {
    _callbacks.remove(elem);
    _config_source->removeCallback(elem);
  }

  static void initClass();

private:
  ActiveConfigSource* _config_source;
  utils::StructElem* _parent;
  utils::ConfigElem* _elem;
  utils::List<SourceWatch> _children;
  utils::List<ActiveElem> _callbacks;
};

UTILS_BASE_SOURCE(SourceWatch);

SourceWatch::SourceWatch(ActiveConfigSource* src)
{
  if (src == NULL) {
    fprintf(stderr,
            "SourceWatch::SourceWatch: DANGER! Null active config source\n");
  }
  UTILS_BASE_CONSTRUCTOR(SourceWatch);
  _elem = NULL;
  _parent = NULL;
  _config_source = src; 
}

void SourceWatch::initClass()
{
  UTILS_BASE_INIT_CLASS(SourceWatch, "SourceWatch", "ConfigWatch");
}

ConfigSource* create_ConfigSource_active(ConfigSourceGenerator*,
                                         utils::ConfigFile* params,
                                         utils::SymbolTable*)
{
  ActiveConfigSource* intf = new ActiveConfigSource();
  if (!intf->open(params->getString("name", "main.conf"))) {
    delete intf;
    return NULL;
  }
  return intf;
}

bool ActiveConfigSource::open(const char* name)
{
  if (!FileConfigSource::open(name))
    return false;

  instrument();
  return true;
}

void ActiveConfigSource::instrument()
{
  SourceWatch* w = new SourceWatch(this);
  w->init(_file.getRoot());
  _file.getRoot()->setWatcher(w);
  w->addTree();
}

ActiveElem* ActiveConfigSource::watch(utils::Type type, const char* name,
                                      void* data, int max_num,
                                      ConfigCallback notify, void* cb_data,
                                      void* default_data, int num_defaults)
{
  SourceWatch* w = do_watch(type, name, data, max_num, notify, cb_data,
                            default_data, num_defaults);
  if (!w)
    return NULL;

  ActiveElem* active_elem =
    w->addCallback(type, name, data, max_num, notify, cb_data);
  _active_elems.append(active_elem);
  return active_elem;
}

SourceWatch* ActiveConfigSource::do_watch(utils::Type type, const char* name,
                                          void* data, int max_num,
                                          ConfigCallback notify, void* cb_data,
                                          void* default_data, int num_defaults)
{
  utils::ConfigElem* elem;
  if (data) {
    if (!FileConfigSource::attach(type, name, data, max_num, notify, cb_data,
                                  default_data, num_defaults))
      return NULL;
  }
  elem = _file.lookup(name);
  if (!elem) {
    elem = getFile().makeElem(name);
    if (!elem)
      return NULL;
  }
  if (!elem->watcher()) {
    printf("Warning: %s is an orphan!\n", name);
    return NULL;
  }
  if (!elem->watcher()->getTypeId().
      isDerivedFrom(SourceWatch::getClassTypeId())) {
    printf("Warning: Something else is watching %s\n", name);
    return NULL;
  }
  return (SourceWatch*) elem->watcher();
}


void ActiveConfigSource::unwatch(utils::ConfigElem* elem, ActiveElem* watcher)
{
  SourceWatch* w = (SourceWatch*) elem->watcher();
  w->removeCallback(watcher);
}

bool ActiveConfigSource::refresh()
{
  utils::ConfigElem* root = getFile().getRoot();
  root->ref();

  getFile().close();
  if (!getFile().open(getName())) {
    root->unref();
    return false;
  }
  instrument();

  // you have to reapply the watches or you get in some serious trouble
  reapply_watches();

  root->unref();

  return true;
}

void ActiveConfigSource::removeCallback(ActiveElem* elem)
{
  bool res = _active_elems.remove(elem);
  if (res)
    delete elem;
}

void ActiveConfigSource::reapply_watches()
{
  utils::ListIterator<ActiveElem> iter(_active_elems);
  for (ActiveElem* elem = iter.first(); elem; elem = iter.next()) {
    SourceWatch* w =
      do_watch(elem->type, elem->name, elem->data, elem->max_num,
               elem->callback, elem->cb_data, NULL, 0);
    if (!w) {
      iter.removeCurrent();
      delete elem;
    } else {
      w->appendCallback(elem);
    }
  }
}

