#include <stdio.h>

#include <utils/Time.h>
#include <utils/String.h>
#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>
#include <TimeSource/TimeSource.h>

#include <VehPoseSource/VehPoseSource.h>

class VehPoseTestModule : public Module
{
public:
  VehPoseTestModule(const char* spec) : Module("VehPoseTest", spec) {}
  virtual ~VehPoseTestModule() {}
  
  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table) {
    const char* spec;
    spec = config->getString("veh_pose_source_spec", "default");
    _veh_pose_source = create<VehPoseSource>(spec);
    if (!_veh_pose_source) {
      printf("Invalid veh pose in specification '%s'\n", spec);
      return false;
    }
  
    _exit_on_no_data = config->getBool("exit_on_no_data", false);
    _print_data = config->getBool("print_data", false);
    _sleep_time = config->getFloat("sleep_time", 0.0);
    _search_time = config->getFloat("search_time", 0.05);

    return true;
  }

  virtual bool run() {
    if (_sleep_time)
      getConfigSource()->sleep(_sleep_time);

    utils::Time t = TimeSource::now();
    printf("%f:\n", (double) t);
    VehPose pose;
    if (_veh_pose_source->getPose(t, pose)) {
      if (_print_data) {
        printf("\tNow, got %f %f\n", pose.pos.x, pose.pos.y);
      }
    } else {
      if (_exit_on_no_data)
        return false;
      printf("Error getting now veh_pose\n");
      return true;
    }

    if (_veh_pose_source->getPose(t+utils::Time(_search_time), pose)) {
      if (_print_data) {
        printf("\t%f from now, got %f %f\n",
               _search_time, pose.pos.x, pose.pos.y);
      }
    } else {
      if (_exit_on_no_data)
        return false;
      printf("Error getting veh_pose\n");
      return true;
    }

    if (_veh_pose_source->getPose(t-utils::Time(_search_time), pose)) {
      if (_print_data) {
        printf("\t%f ago, got %f %f\n", _search_time, pose.pos.x, pose.pos.y);
      }
    } else {
      if (_exit_on_no_data)
        return false;
      printf("Error getting veh_pose\n");
      return true;
    }
    printf("\n");

    return true;
  }

private:
  VehPoseSource* _veh_pose_source;   ///< the veh_pose input
  bool _exit_on_no_data;      ///< true if an error in input is fatal
  bool _print_data;           ///< true if we output some debugging info
  float _sleep_time;          ///< if non-zero, sleep this many seconds
  float _search_time;         ///< time ahead and behind to get pose
};

MODULE_MAIN(veh_pose_test, VehPoseTestModule);
