/** \file
    Defines the RoadPlayer class
    \ingroup RoadGroup
*/
#ifndef VEH_POSE_PLAYER_H
#define VEH_POSE_PLAYER_H

#include <vector>
#include <utils/Player.h>
#include <TimeSource/TimeSource.h>
#include <VehPoseDest/VehPose.h>
#include <VehPoseDest/VehPoseStructs.h>

namespace utils {
  class ConfigFile;
  class Vec3d;
  class SymbolTable;
};

/** Standalone class for playing road information.
    This class lets you open a structured canned data file and get data
    from it

    \ingroup VehPoseGroup
*/
class VehPosePlayer {
 public:
  /** Open a file for playing.
      \param[in] params Configuration file including the file name and other 
                        standard TimeSource::PlayerManager parameters
      \param[in] globals The global symbol table                        
      \return \p true on success, \p false on failure
  */
  bool open(utils::ConfigFile& params, utils::SymbolTable* globals);

  /** Advance to the next vehicle pose
      \return \p true for success, \p false for failure
  */
  bool advance();

  /** Get the current vehicle pose tagged with a time.
      \param[out] time Time the vehicle pose was generated
      \param[out] pose The vehicle pose
      \return \p true for success, \p false for failure
  */
  bool getCurPose(utils::Time& time, VehPose& pose);

  /** Get the next vehicle pose tagged with a time.
      This could have a side effect of advancing the file pointers
      \param[out] time Time the vehicle pose was generated
      \param[out] pose The vehicle pose
      \param[in] blocking If \p false we will poll the manager to see if we
                          need to advance
      \return \p true for success, \p false for failure
  */
  bool nextPose(utils::Time& time, VehPose& pose, bool blocking=true);

  /** Read the pose at a given time, interpolating and extrapolating if 
      necessary.
      \param[in] time The time for which to get the vehicle pose.
      \param[out] pose The resulting vehicle pose
      \return \p true if the routine was able to obtain a vehicle pose at
              the desired time, \p false if not.
  */
  bool getPose(utils::Time time, VehPose& pose);
  
  /** Provides access to the underlying canned data manager. */
  TimeSource::PlayerManager* getManager() { return &_mgr; }

 private:
  void set_pose(VehPose& pose);

 private:
  TimeSource::PlayerManager _mgr;  ///< the player manager 
  utils::Player* _player;          ///< the actual player

  VehPoseDataStruct _input_area; ///< Packaged road information
  utils::Time _play_time;     ///< Timestamp for last read pose information
};

#endif

