/** \file
    Implements the OutputVehPoseSource interface instance
    \ingroup VehPoseGroup
*/
#include <iostream>

#include <utils/ConfigFile.h>
#include <utils/Generator.h>
#include <TimeSource/TimeSource.h>
#include <VehPoseDest/VehPoseDest.h>

#include "VehPoseSource.h"

/** Container interface definition for VehPoseSource that transparently
    redirects to a VehPoseDest.
*/
class OutputVehPoseSource : public VehPoseSource {
 public:
  OutputVehPoseSource();
  virtual ~OutputVehPoseSource();

  /// \copydoc VehPoseSource::getCurPose
  virtual bool getCurPose(utils::Time& time, VehPose& pose,
                          bool blocking = true);

  /// \copydoc VehPoseSource::getPose
  virtual bool getPose(utils::Time time, VehPose& pose);

  /// Initialization routine
  bool init(utils::ConfigFile& params,
            VehPoseSourceGenerator* gen, utils::SymbolTable* globals);

 private:
  VehPoseSource* _contained;
  VehPoseDest* _output;
};

/// The required creation function for the "out" tag
VehPoseSource* create_VehPoseSource_out(VehPoseSourceGenerator* gen,
                                  utils::ConfigFile* params,
                                  utils::SymbolTable* globals)
{
  OutputVehPoseSource* intf = new OutputVehPoseSource();
  if (!intf->init(*params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

/// Additional creation function for the "logger" tag which will create
/// a logger output.
VehPoseSource* create_VehPoseSource_logger(VehPoseSourceGenerator* gen,
                                     utils::ConfigFile* params,
                                     utils::SymbolTable* globals)
{
  // pose output specification inherited from params for the most part
  utils::ConfigFile output_params;
  utils::ConfigFile::copy(*params, output_params);
  output_params.setString("tag", "logger");  // set up logger tag
  output_params.set("contained", "{}"); //we clear contained to avoid confusion
  
  // get the contained specification
  utils::ConfigFile contained_params;
  params->getStruct("contained", contained_params);

  // now create the output instance parameters
  utils::ConfigFile final_params;
  final_params.setString("tag", "output");
  final_params.setStruct("contained", contained_params);
  final_params.setStruct("output", output_params);

  OutputVehPoseSource* intf = new OutputVehPoseSource();
  if (!intf->init(final_params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

/// Additional creation function for the "shmemPublish" tag which will create
/// a shmem output.
VehPoseSource* create_VehPoseSource_shmemPublish(VehPoseSourceGenerator* gen,
                                           utils::ConfigFile* params,
                                           utils::SymbolTable* globals)
{
  // pose output specification inherited from params for the most part
  utils::ConfigFile output_params;
  utils::ConfigFile::copy(*params, output_params);
  output_params.setString("tag", "shmem");  // set up shmem tag
  output_params.set("contained", "{}"); //we clear contained to avoid confusion
  
  // get the contained specification
  utils::ConfigFile contained_params;
  params->getStruct("contained", contained_params);

  // now create the output instance parameters
  utils::ConfigFile final_params;
  final_params.setString("tag", "output");
  final_params.setStruct("contained", contained_params);
  final_params.setStruct("output", output_params);

  OutputVehPoseSource* intf = new OutputVehPoseSource();
  if (!intf->init(final_params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

OutputVehPoseSource::OutputVehPoseSource()
{
  _contained = NULL;
  _output = NULL;
}

OutputVehPoseSource::~OutputVehPoseSource()
{
  if (_contained)
    _contained->unref();
  if (_output)
    _output->unref();
}

bool OutputVehPoseSource::init(utils::ConfigFile& params,
                            VehPoseSourceGenerator* gen,
                            utils::SymbolTable* globals)
{
  // create the contained interface
  _contained = gen->interface(params.getString("contained"), globals);
  if (!_contained) {
    cerr << "OutputVehPoseSource::init: could not create contained\n";
    return false;
  }
  _contained->ref();

  _output = VehPoseDest::generate(params.getString("output", "default"), globals);
  if (!_output) {
    cerr << "OutputVehPoseSource::init: could not create output\n";
    return false;
  }
  _output->ref();
  
  return true;
}

bool OutputVehPoseSource::getCurPose(utils::Time& time, VehPose& pose, 
                                     bool blocking)
{
  if (_contained->getCurPose(time, pose, blocking)) {
    _output->outputPose(time, pose);
    return true;
  } 
  return false;
}

bool OutputVehPoseSource::getPose(utils::Time time, VehPose& pose)
{
  if (_contained->getPose(time, pose)) {
    _output->outputPose(time, pose);
    return true;
  } 
  return false;
}
