/** \file
    Implements the ShmemVehPoseDest interface instance
    \ingroup VehPoseGroup
*/
#include <stdio.h>
#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <utils/SymbolTable.h>
#include <utils/ConfigFile.h>

#include "VehPoseDest.h"
#include "VehPoseStructs.h"

/// Shared memory interface definition for VehPoseDest
class ShmemVehPoseDest : public VehPoseDest {
 public:
  /// \copydoc VehPoseDest::outputPose
  virtual bool outputPose(utils::Time time, const VehPose& veh_pose);

  /// Initialization routine
  bool init(utils::ConfigFile& params, utils::SymbolTable* globals);

 private:
  IPSharedMemory* _shm;  ///< the shared memory region
  VehPoseShmemStruct _output_area;  ///< the output cache
};

/// The required creation function for the "shmem" tag
VehPoseDest* create_VehPoseDest_shmem(VehPoseDestGenerator*,
                                 utils::ConfigFile* params,
                                 utils::SymbolTable* globals)
{
  ShmemVehPoseDest* intf = new ShmemVehPoseDest();
  if (!intf->init(*params, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

bool ShmemVehPoseDest::init(utils::ConfigFile& params,
                         utils::SymbolTable* globals)
{
  // get or create the IPT communicator
  // If it is created, it is cached in the global symbol table
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params.getString("ipt_spec",
                                                  "unix: int port=0;"));
  if (!com)
    return false;

  // setup the shared memory specification
  const char* mem_name = params.getString("name", VEH_POSE_SHMEM_NAME);
  char buffer[200];
  // first set up the default, which is based on the memory name
  sprintf(buffer, "managed: name=%s; owner=true;", mem_name);
  // and then get the spec given the default (i.e., it can be arbitrarily
  // overridden
  const char* mem_spec = params.getString("mem", buffer);
  // create the shared memory region
  _shm =
    com->OpenSharedMemory(mem_spec, VEH_POSE_SHMEM_FMT,
                          sizeof(VehPoseShmemStruct));
  if (!_shm) {
    printf("Problem opening shared memory %s\n", mem_spec);
    return false;
  }

  return true;
}

bool ShmemVehPoseDest::outputPose(utils::Time time, const VehPose& veh_pose)
{
  // setup the output area
  _output_area.data.x = veh_pose.pos.x;
  _output_area.data.y = veh_pose.pos.y;
  _output_area.data.z = veh_pose.pos.z;
  _output_area.data.ori[0] = veh_pose.ori[0];
  _output_area.data.ori[1] = veh_pose.ori[1];
  _output_area.data.ori[2] = veh_pose.ori[2];
  _output_area.data.ori[3] = veh_pose.ori[3];
  time.getValue(_output_area.secs, _output_area.usecs);

  // and output to shared memory
  _shm->PutFormattedData((void*) &_output_area);

  return true;
}



  
