/** \file
    Implements the ShmemRoadSource class
    \ingroup RoadGroup
 */
#include <stdio.h>
#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <utils/SymbolTable.h>
#include <utils/ConfigFile.h>

#include <RoadDest/RoadStructs.h>
#include "RoadSource.h"

/** Shared Memory interface definition for RoadSource.
*/
class ShmemRoadSource : public RoadSource {
 public:
  ShmemRoadSource();

  /// \copydoc RoadSource::getPoints
  virtual bool getPoints(utils::Time& time,
                         std::vector<utils::Vec3d>& points,
                         bool blocking = true);

  /// Initialization routine
  bool init(utils::ConfigFile& params, utils::SymbolTable* globals);

 private:
  IPSharedMemory* _shm;  ///< the shared memory region
  int _last_tag;  ///< Tag of last data read
};

/// The required creation function for the "shmem" tag
RoadSource* create_RoadSource_shmem(RoadSourceGenerator*,
                                    utils::ConfigFile* params,
                                    utils::SymbolTable* globals)
{
  ShmemRoadSource* intf = new ShmemRoadSource();
  if (!intf->init(*params, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

ShmemRoadSource::ShmemRoadSource()
{
  _last_tag = -1;
}

bool ShmemRoadSource::init(utils::ConfigFile& params,
                           utils::SymbolTable* globals)
{
  // get or create the IPT communicator
  // If it is created, it is cached in the global symbol table
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params.getString("ipt_spec",
                                                  "unix: int port=0;"));
  if (!com)
    return false;

  // setup the shared memory specification
  // first set up the default, which is based on the memory name
  // the host machine, and the port for the memory manager
  const char* mem_name = params.getString("name", ROAD_SHMEM_NAME);
  const char* machine = params.getString("machine");
  int port = params.getInt("port", 1389);
  char buffer[200];
  if (!*machine) {
    sprintf(buffer, "managed: name=%s;", mem_name);
  } else {
    sprintf(buffer, "managed: name='%s@%s|%d';", mem_name, machine, port);
  }
  const char* mem_spec = params.getString("mem", buffer);
  // get the maximum expected number of points
  int max_points = params.getInt("max_points", 20);
  // create the shared memory region
  _shm =
    com->OpenSharedMemory(mem_spec, ROAD_SHMEM_FMT,
                          sizeof(RoadShmemStruct) +
                          max_points*sizeof(RoadDataPoint));
  if (!_shm) {
    printf("Problem opening shared memory %s\n", mem_spec);
    return false;
  }

  return true;
}

bool ShmemRoadSource::getPoints(utils::Time& time,
                                std::vector<utils::Vec3d>& points,
                                bool blocking)
{
  if (blocking) {
    // wait for new data in the shared memory region
    if (!_shm->Wait()) {
      // if waiting failed, mark result with a bad time and return false
      time = utils::Time();
      return false;
    }
  }
  // Get the data
  RoadShmemStruct input_area;  
  if (!_shm->FormattedData(&input_area)) {
    // if there was a formatting failure, mark as bad and return
    time = utils::Time();
    return false;
  }

  // package the results
  time.setValue(input_area.secs, input_area.usecs);
  points.clear();
  for (int i=0;i<input_area.data.num_points;i++) {
    RoadDataPoint& pt = input_area.data.points[i];
    points.push_back(utils::Vec3d(pt.x, pt.y, pt.z));
  }

  // and release memory back to IPT
  _shm->DeleteContents(&input_area);

  // determine if this is "new" data or not by the shared memory tags
  if (_shm->Tag() != _last_tag) {
    _last_tag = _shm->Tag();
    return true;
  } else 
    return false;
}

