/** \file RoadSource.h
    API header file for for the RoadSource reconfigurable interface 
    \ref RoadGroup "example".
    \ingroup RoadGroup
*/

#ifndef ROAD_SOURCE_INTF_H
#define ROAD_SOURCE_INTF_H

// include the STL vector
#include <vector>

#include <utils/Time.h>
#include <utils/Linear.h>
#include <utils/Interface.h>

/** \class RoadSource

    The RoadSource interface example.

    An abstract interface to 3D road information given by an STL vector of 3D 
    points

    \ingroup RoadGroup
 */
UTILS_INTERFACE(RoadSource) {
 public:
  /** Empty virtual destructor for RoadSource.
      For C++ syntax reasons, it is a good idea to explicitly define a virtual
      destructor, even if empty like this one, to any abstract class like this.
   */
  virtual ~RoadSource() {}

  /** Get the next set of road points.
      Returns the latest set of road points along with the time at which they
      were collected.  Normally, the call is blocking, i.e., we block until the
      next unread set of points is acquired, but it can be polling, i.e., it
      returns immediately with the most recent set of road points.

      \param[out] time The time at which the road information was collected. (0
                        if the road is invalid)
      \param[out] points The vector of 3D road points
      \param[in] blocking  If \p true, then the call is blocking, if \p false,
                           it is polling.
      \return \p true if "new" data has been acquired successfully, \p false
              on an invalid read or if no new road data has been acquired
              since the last read.  Note an invalid read can be verified by
              checking to see if \p time.isZero() is true.
  */
  virtual bool getPoints(utils::Time& time,
                         std::vector<utils::Vec3d>& points,
                         bool blocking = true) = 0;

  // declare the standard interface static methods
  UTILS_INTF_DECL(RoadSource);
};
  
#endif
