/** \file
    Impements the FakeRoadSource interface instance
    \ingroup RoadGroup
*/
#include <iostream>

#include <utils/ConfigFile.h>
#include <TimeSource/TimeSource.h>

#include "RoadSource.h"

/** Fake interface definition for RoadSource.
    This interface reads in the triples of floating point number from the
    parameter \p points and creates a list of road points which it continually
    returns using the current time
*/
class FakeRoadSource : public RoadSource {
 public:
  /// \copydoc RoadSource::getPoints
  virtual bool getPoints(utils::Time& time,
                         std::vector<utils::Vec3d>& points,
                         bool blocking = true);

  /// Initialization routine
  bool init(utils::ConfigFile& params);

 private:
  std::vector<utils::Vec3d> _points;  // stored points to return
};

/// The required creation function for the "fake" tag
RoadSource* create_RoadSource_fake(RoadSourceGenerator*,
                                   utils::ConfigFile* params,
                                   utils::SymbolTable* globals)
{
  FakeRoadSource* intf = new FakeRoadSource();
  if (!intf->init(*params)) {
    delete intf;
    return NULL;
  }
  return intf;
}

bool FakeRoadSource::init(utils::ConfigFile& params)
{
  // read in the points:  They are required
  // Note we use "double points" to force parsing as floating point if the
  // points field had no type specifier in the parameter file
  int num_pts = params.numValues("double points");
  if (!num_pts) {
    cerr << "FakeRoadSource::init: No points\n";
    return false;
  }
  // and you need at least 2 (6 numbers)
  if (num_pts < 6) {
    cerr << "FakeRoadSource::init: Not enough points\n";
    return false;
  }
  double* points = new double[num_pts];
  if (params.getDoubles("points", points, num_pts) != num_pts) {
    cerr << "FakeRoadSource::init: Error getting points\n";
    return false;
  }
  
  // make sure num_pts divisible by 3
  num_pts -= (num_pts % 3);

  // and fill the _points vector
  for (int i=0;i<num_pts;i+=3) {
    _points.push_back(utils::Vec3d(points[i], points[i+1], points[i+2]));
  }

  // clean up scratch memory
  delete [] points;

  return true;
}

bool FakeRoadSource::getPoints(utils::Time& time,
                               std::vector<utils::Vec3d>& points,
                               bool)
{
  time = TimeSource::now();
  points = _points;

  return true;
}

