/** \file
    Implements the ShmemRoadDest interface instance
    \ingroup RoadGroup
*/
#include <stdio.h>
#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <utils/SymbolTable.h>
#include <utils/ConfigFile.h>

#include "RoadDest.h"
#include "RoadStructs.h"

/// Shared memory interface definition for RoadDest
class ShmemRoadDest : public RoadDest {
 public:
  ShmemRoadDest();
  virtual ~ShmemRoadDest();

  /// \copydoc RoadDest::outputPoints
  virtual bool outputPoints(utils::Time time,
                            const std::vector<utils::Vec3d>& points);

  /// Initialization routine
  bool init(utils::ConfigFile& params, utils::SymbolTable* globals);

 private:
  IPSharedMemory* _shm;  ///< the shared memory region
  RoadShmemStruct _output_area;  ///< the output cache
  unsigned int _max_points;      ///< maximum number of points in output cache
};

/// The required creation function for the "shmem" tag
RoadDest* create_RoadDest_shmem(RoadDestGenerator*,
                                 utils::ConfigFile* params,
                                 utils::SymbolTable* globals)
{
  ShmemRoadDest* intf = new ShmemRoadDest();
  if (!intf->init(*params, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

ShmemRoadDest::ShmemRoadDest()
{
  _max_points = 5;
  _output_area.data.points = new RoadDataPoint[_max_points];
}

ShmemRoadDest::~ShmemRoadDest()
{
  delete [] _output_area.data.points;
}

bool ShmemRoadDest::init(utils::ConfigFile& params,
                         utils::SymbolTable* globals)
{
  // get or create the IPT communicator
  // If it is created, it is cached in the global symbol table
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params.getString("ipt_spec",
                                                  "unix: int port=0;"));
  if (!com)
    return false;

  // setup the shared memory specification
  const char* mem_name = params.getString("name", ROAD_SHMEM_NAME);
  char buffer[200];
  // first set up the default, which is based on the memory name
  sprintf(buffer, "managed: name=%s; owner=true;", mem_name);
  // and then get the spec given the default (i.e., it can be arbitrarily
  // overridden
  const char* mem_spec = params.getString("mem", buffer);
  // get the maximum expected number of points
  int max_points = params.getInt("max_points", 20);
  // create the shared memory region
  _shm =
    com->OpenSharedMemory(mem_spec, ROAD_SHMEM_FMT,
                          sizeof(RoadShmemStruct) +
                          max_points*sizeof(RoadDataPoint));
  if (!_shm) {
    printf("Problem opening shared memory %s\n", mem_spec);
    return false;
  }

  return true;
}

bool ShmemRoadDest::outputPoints(utils::Time time,
                                 const std::vector<utils::Vec3d>& points)
{
  // make sure there is enough room in the cached points for the data
  if (points.size() > _max_points) {
    _max_points = points.size();
    delete [] _output_area.data.points;
    _output_area.data.points = new RoadDataPoint[_max_points];
  }

  // setup the output area
  _output_area.data.num_points = (int) points.size();
  for (int i=0;i<_output_area.data.num_points;i++) {
    const utils::Vec3d& src = points[i];
    RoadDataPoint& dest = _output_area.data.points[i];
    dest.x = src.x;
    dest.y = src.y;
    dest.z = src.z;
  }
  time.getValue(_output_area.secs, _output_area.usecs);

  // and output to shared memory
  _shm->PutFormattedData((void*) &_output_area);

  return true;
}



  
