///////////////////////////////////////////////////////////////////////////////
//
//                                 messagetype.cc
//
// This file implements the class that holds a message's type information, 
// i.e., whether (and how) it is handled, whether (and how) it is formatted,
// what its destination is, what its type name is, etc.
//
// Classes defined for export:
//    IPMessageType
//
//  "1995, Carnegie Mellon University. All Rights Reserved." This
//  software is made available for academic and research purposes only. No
//  commercial license is hereby granted.  Copying and other reproduction is
//  authorized only for research, education, and other non-commercial
//  purposes.  No warranties, either expressed or implied, are made
//  regarding the operation, use, or results of the software.
//
///////////////////////////////////////////////////////////////////////////////

#include <ipt/libc.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#include <utils/List.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/DeleteAction.h>

#include <ipt/messagetype.h>
#include <ipt/callbacks.h>
#include <ipt/connection.h>

/* Create a message type with name "name", format "format" and message ID
   "id" */
IPMessageType::IPMessageType(const char* name, IPFormat* format, int id)
{
    int len = strlen(name)+1;
    
    _name = new char[len];
    bcopy(name, _name, len);
    _id = id;
    _format = format;
    IPManaged::ref(_format);
    _handler_active = 1;
    _handler_invoked = 0;
    _postponed_messages = new IPList<IPMessage>;
    _destination = 0L;
    _callback = 0L;
    _handler_context = 0;
    _size_cache = _cache_in_use = 0;
    _cache = 0L;
}

/* delete a message type */
IPMessageType::~IPMessageType()
{
    delete [] _name;
    delete _postponed_messages;

    IPManaged::unref(_format);

    if (_callback) {
        IPManaged::unref(_callback);
    	_callback = 0L;
	}
    if (_cache)	{
        delete [] _cache;
    	_cache = 0L;
	}
}

/* take unformatted data "input", and create formatted data and return it.
   If there is no formatter for this type, return 0L */
void* IPMessageType::FormatData(unsigned char* input, int size_input,
                                IPConnection* conn)
{
    if (_format) {
        if (!input) {
            fprintf(stderr, "IPT Fatal Error: Message type %s expects some data\n",
                    _name);
            exit(-1);
        }
        if (conn) 
            return IPUnpackAction::unpack(_format, input, size_input,
                                          conn->Alignment(),
                                          conn->ByteOrder());
        else
            return IPUnpackAction::unpack(_format, input, size_input,
                                          ALIGN, BYTE_ORDER);
    } else return 0L;
}

/* take unformatted data "input", and create formatted data and return it.
   If there is no formatter for this type, return 0L */
void IPMessageType::FormatData(unsigned char* input, int size_input,
                               void* output, IPConnection* conn)
{
    if (_format) {
        if (!input) {
            fprintf(stderr, "IPT Fatal Error: Message type %s expects some data\n",
                    _name);
            exit(-1);
        }
        if (conn)
            IPUnpackAction::unpack(_format, input, size_input, output, 0,
                                   conn->Alignment(), conn->ByteOrder());
        else
            IPUnpackAction::unpack(_format, input, size_input, output, 0,
                                   ALIGN, BYTE_ORDER);
    } 
}

/* Take formatted data "input" and create unformatted output buffer "output", 
   copy the data into it, and put the number of bytes in "output_size".
   If formatter is 0L, do nothing */
int IPMessageType::UnformatData(void* input,
                                int& output_size, unsigned char*& output)
{
    if (!_format) {
        output_size = 0;
        output = 0L;
        return 0;
    }
    output_size = IPBufferSizeAction::size(_format, input);
    if (_format->isPackedType()) {
        output = (unsigned char*) input;
        return 0;
    } else {
        output = new unsigned char[output_size];
        return IPPackAction::pack(_format, output, output_size, input);
    }
}

/* Take formatted data "input" and create unformatted output buffer "output", 
   copy the data into it, and put the number of bytes in "output_size".
   If formatter is 0L, do nothing */
int IPMessageType::CachedUnformatData(void* input,
                                      int& output_size, unsigned char*& output)
{
    if (!_format) {
        output_size = 0;
        output = 0L;
        return 0;
    }
    output_size = IPBufferSizeAction::size(_format, input);
    if (_format->isPackedType()) {
        output = (unsigned char*) input;
        return 0;
    } else {
        int allocated = 0;
        output = Cache(output_size);
        if (!output) {
            output = new unsigned char[output_size];
            allocated = 1;
        }
        if (!IPPackAction::pack(_format, output, output_size, input))
            return 0;
        return allocated;
    }
}

/* Take formatted data "input", unformat it into bytes, and copy the result
   into "output", which had better have the right number of bytes.  If there
   is no formatter for this type, do nothing */
void IPMessageType::RawUnformatData(void* input, unsigned char* output)
{
    if (_format)
        IPPackAction::pack(_format, output, 0, input);
}

/* Delete the formatted data "data" according to the type's message formatter,
   if there is one */
void IPMessageType::DeleteFormattedData(void* data,
                                        unsigned char* buffer, int bsize)
{
    if (_format)
        IPDeleteAction::deleteData(_format, data, buffer, bsize);
}

void IPMessageType::DeleteContents(void* data,
                                   unsigned char* buffer, int bsize)
{
    if (_format)
        IPDeleteAction::deleteStruct(_format, data, buffer, bsize);
}

/* Set the message type's handler to "callback" and the context to "context" */
void IPMessageType::Handler(IPHandlerCallback* callback, int context)
{
    if (_callback)
        IPManaged::unref(_callback);
    _callback = callback;
    IPManaged::ref(_callback);
    _handler_context = context;
}

/* Postpone message "msg" */
void IPMessageType::Postpone(IPMessage* msg)
{
    _postponed_messages->append(msg);
}

/* Get the earliest postponed message, remove it from the postponed list */
IPMessage* IPMessageType::Postponed()
{
    return _postponed_messages->pop();
}

void IPMessageType::SetFormat(IPFormat* fmt)
{
    if (!fmt)
        return;

    if (_format && _format != fmt) {
        IPManaged::unref(_format);
        IPManaged::ref(fmt);
    }
    _format = fmt;
}

unsigned char* IPMessageType::Cache(int num_bytes)
{
    if (_cache_in_use)
        return 0L;

    _cache_in_use = 1;

    if (num_bytes <= _size_cache) 
        return _cache;
    if (_cache)
        delete [] _cache;
    _cache = new unsigned char[num_bytes];
    _size_cache = num_bytes;

    return _cache;
}

void IPMessageType::ReleaseCache(unsigned char* cache)
{
    if (_cache_in_use && _cache == cache)
        _cache_in_use = 0;
}

int IPMessageType::GetCache(int& num_bytes, unsigned char*& cache)
{
    num_bytes = _size_cache;
    cache = _cache;
    return 1;
}
