#include <stdio.h>
#include <string.h>

#include <ipt/ipt.h>

#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>
#include <utils/Generator.h>
#include <utils/SpecElem.h>

#define USE_UNIX (!defined(VXWORKS) && !defined(SOLARIS) && !defined(LYNX))
#define USE_VXWORKS (defined(VXWORKS) && !defined(NO_VXCONNECTION))

#include <ipt/tcpcommunicator.h>

#if USE_UNIX==1
#include <ipt/unixcommunicator.h>
#endif

#if USE_VXWORKS==1
#include <ipt/vxcommunicator.h>
#endif

struct IPCommunicatorInfo {
  char* tag;
  IPCommunicator* (*create_interface)(IPGenerator<IPCommunicator>*,
                                      IPConfigFile*, IPSymbolTable*);
};

IPCommunicator* create_communicator_served(IPGenerator<IPCommunicator>*,
                                           IPConfigFile*, IPSymbolTable*);
IPCommunicator* create_communicator_tcp(IPGenerator<IPCommunicator>*,
                                        IPConfigFile*, IPSymbolTable*);
#if USE_UNIX==1
IPCommunicator* create_communicator_unix(IPGenerator<IPCommunicator>*,
                                         IPConfigFile*, IPSymbolTable*);
#endif
#if USE_VXWORKS==1
IPCommunicator*
create_communicator_vxworks(IPGenerator<IPCommunicator>*,
                            IPConfigFile*,IPSymbolTable*);
#endif

#define DEFAULT_INTF "served"

static struct IPCommunicatorInfo IP_communicators[] = {
  {"served", create_communicator_served },
  {"tcp", create_communicator_tcp},
#if USE_UNIX==1
  {"unix", create_communicator_unix},
#endif
#if USE_VXWORKS==1
  {"vxworks", create_communicator_vxworks},
#endif
  {0, 0}
};

IPCommunicator* create_communicator_served(IPGenerator<IPCommunicator>*,
                                           IPConfigFile* params,
                                           IPSymbolTable* globals)
{
  const char* module_name =
    params->getString("ModuleName", 
                      (const char*) globals->get("ModuleName"));
  if (!module_name || !*module_name)
    return NULL;

  return IPCommunicator::Instance(module_name,
                                  params->getString("host_name"));
}

IPCommunicator* create_communicator_tcp(IPGenerator<IPCommunicator>*,
                                        IPConfigFile* params,
                                        IPSymbolTable* globals)
{
  const char* module_name =
    params->getString("ModuleName", 
                      (const char*) globals->get("ModuleName"));
  if (!module_name || !*module_name)
    return NULL;

  int port = params->getInt("port", -1);
  TCPCommunicator* res;
  if (port != -1) {
    res = new TCPCommunicator(port);
    res->ModuleName(module_name);
  } else {
    res = new TCPCommunicator(module_name,
                              params->getString("host_name", NULL));
    if (!res->Valid()) {
      delete res;
      return NULL;
    }
    res->send_routes();
  }

  params->setInt("int port", res->getPort());

  return res;
}

#if USE_UNIX==1
IPCommunicator* create_communicator_unix(IPGenerator<IPCommunicator>*,
                                         IPConfigFile* params,
                                         IPSymbolTable* globals)
{
  const char* module_name =
    params->getString("ModuleName", 
                      (const char*) globals->get("ModuleName"));
  if (!module_name || !*module_name)
    return NULL;

  int port = params->getInt("port", -1);
  UnixCommunicator* res;
  if (port != -1) {
    res = new UnixCommunicator(port);
    res->ModuleName(module_name);
  } else {
    res = new UnixCommunicator(module_name,
                               params->getString("host_name", NULL),
                               params->getInt("allow_fail", 0));
    if (!res->Valid()) {
      delete res;
      return NULL;
    }
    res->send_routes();
  }

  port = res->getPort();
  params->setInt("int port", port);

  return res;
}
#endif

#if USE_VXWORKS==1
IPCommunicator*
create_communicator_vxworks(IPGenerator<IPCommunicator>*,
                            IPConfigFile* params, IPSymbolTable* globals)
{
  const char* module_name =
    params->getString("ModuleName", 
                      (const char*) globals->get("ModuleName"));
  if (!module_name || !*module_name)
    return NULL;

  VxCommunicator* res =
    new VxCommunicator(module_name, params->getString("host_name", NULL));

  if (!res->Valid()) {
    delete res;
    return NULL;
  }
  res->send_routes();

  port = res->getPort();
  params->set("int port", &port, 1);

  return res;
}
#endif

IPGenerator<IPCommunicator>* IPCommunicator::generator()
{
  int i;
  IPGenerator<IPCommunicator>* gen =
    new IPGenerator<IPCommunicator>();
  for (i=0;IP_communicators[i].tag; i++) {
    gen->registerInterface(IP_communicators[i].tag,
                           IP_communicators[i].create_interface, 0L);
    if (!strcmp(IP_communicators[i].tag,DEFAULT_INTF))
      gen->registerInterface("default",
                             IP_communicators[i].create_interface, 0L);
  }

  return gen;
}    

// Create a interface given a spec string and a globals pool
IPCommunicator* IPCommunicator::Create(const char* mod_name,
                                       IPConfigFile* params)
{
  IPSymbolTable sym_table;
  if (mod_name && *mod_name)
    sym_table.set("ModuleName", mod_name);

  IPGenerator<IPCommunicator>* gen = generator();
  IPCommunicator* res = gen->interface(params, &sym_table);
  delete gen;

  return res;
}

// Create a interface given a spec string and a globals pool
IPCommunicator* IPCommunicator::Create(const char* mod_name,
                                       const char* spec)
{
  IPSymbolTable sym_table;
  if (mod_name && *mod_name)
    sym_table.set("ModuleName", mod_name);

  IPGenerator<IPCommunicator>* gen = generator();
  IPCommunicator* res = gen->interface(spec, &sym_table);
  delete gen;

  return res;
}

IPCommunicator* IPCommunicator::Communicator(IPSymbolTable* globals)
{
  return (IPCommunicator*) globals->get("Communicator");
}

class IPCommunicatorDeleter : public utils::SymbolDeleter<IPCommunicator>
{
public:
  virtual bool isLast() const { return true; }
};

IPCommunicator* IPCommunicator::Communicator(IPSymbolTable* globals,
                                             IPConfigFile* params)
{
  IPCommunicator* com = (IPCommunicator*) globals->get("Communicator");
  if (!com) {
    const char* mod_name =
      params->getString("module_name",
                        (const char*) globals->get("ModuleName"));
    if (!mod_name || !*mod_name) {
      fprintf(stderr, "Remote ConfigSource requires ModuleName to be set\n");
      return NULL;
    }
    com = IPCommunicator::Create(mod_name, params);
    if (!com)
      return NULL;
    globals->set("Communicator", com, new IPCommunicatorDeleter);
    const char* output = params->getString("output", "");
    if (output && *output) 
      com->SetOutput(output);
  }

  return com;
}

void IPCommunicator::SetCommunicator(IPSymbolTable* globals,
                                     IPCommunicator* com)
{
    globals->set("Communicator", com,  new IPCommunicatorDeleter);
}

IPCommunicator* IPCommunicator::Communicator(IPSymbolTable* globals,
                                             const char* spec)
{
  if (!spec)
    return NULL;
  IPConfigFile params(new IPSpecElem);
  if (!params.parse(spec))
    return NULL;

  return Communicator(globals, &params);
}



  
  
