/* Shared memory communications test

   Basic example is:

     Process 1:  ./bshmtest "owner: sem_key=100; shm_key=101;" owner
     Process 2:  ./bshmtest "client: sem_key=100; shm_key=101;"

   Managed example is:
     Process 1:  ./iptshmgr
     Process 2:  ./bshmtest "managed: name=test; owner=true;" owner
     Process 3:  ./bshmtest "managed: name=test;" 

   Managed example using alternate port is:
     Process 1:  ./iptshmgr other.cfg
     Process 2:  ./bshmtest "managed: name=test; owner=true; port=1400;" owner
     Process 3:  ./bshmtest "managed: name=test; port=1400;"

   Single processor, multi manager example: run Managed example process 1 and 2
       on machine A, then
     Process 3:  ./iptshmgr other.cfg
     Process 4:  ./bshmtest "managed: name='test@A|1389'; port=1400;"

   Inter-processor example, run Managed example on machine A, then on machine B
     Process 1:  ./iptshmgr
     Process 2:  ./bshmtest "managed: name='test@A|1389';"

*/

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>

#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

static bool finished = false;

void ctrl_c_handler(int) {
  finished = true;
}

#define NUM_FLOATS 50000

int main(int argc, char** argv)
{
  char buf[100];
  sprintf(buf, "bshmtest%d", getpid());
  IPCommunicator* ipt = IPCommunicator::Create(buf, "unix: port=0;");

  
  snprintf(buf, 100, "[float : %d]", NUM_FLOATS);
  IPSharedMemory* shm = ipt->OpenSharedMemory(argv[1], buf,
                                              sizeof(float)*NUM_FLOATS);
  if (!shm) {
    printf("Problem opening shared memory '%s'\n", argv[1]);
    delete ipt;
    exit(0);
  }

  signal(SIGINT, ctrl_c_handler);

  float* array = new float[NUM_FLOATS];
  memset(array, 0, sizeof(float)*NUM_FLOATS);

  int n = 0;
  if (argc > 2) {
    // an owner
    while (!finished) {
      shm->PutFormattedData(array);
      printf("Outputted %d: %f\n", shm->Tag(), array[0]);
      ipt->Sleep(1.0);
      n++;
      array[0] += 1.0;
    }
  } else {
    while (shm->Wait() && !finished) {
      if (!shm->FormattedData(array))
        break;
      printf("Received data %d: %f\n", shm->Tag(), array[0]);
      n++;
    }
  }

  printf("Finishing\n");

  delete ipt;

  return 0;
}
      
      
    
