/*------------------------------------------------------------------------------
$Id: interactive.c,v 1.1 2001/04/24 18:02:42 jayg Exp $ 

     This software was developed by Martin Marietta Corporation under the
     DARPA-sponsored UGV Demo II program, Contract DAAH01-92-C-R101.
     Distribution is restricted to the government and contractors under the
     UGV Demo II program.

        FILE:    interactive.c

	AUTHOR:  kimberly a. sherrard
 
	DESCRIPTION:  

	FUNCTIONS:
		PUBLIC:

		PRIVATE:
$Log: interactive.c,v $
Revision 1.1  2001/04/24 18:02:42  jayg
Initial checkin of some start-up apps

 * Revision 1.1.1.1  1995/05/05  05:14:34  jayg
 * Initial entry of the start script management system
 *

------------------------------------------------------------------------------*/


#include <stdio.h>
#include <string.h>
#include <ctype.h>    
#include <signal.h>
#include "send.h"
#include "start.h"

/*
 *----------------------- PRIVATE DECLARATIONS ---------------------------------
 */

/*------------------------------------------------------------------------------
	CONSTANTS:
------------------------------------------------------------------------------*/

#define EXIT_STRING "exit"
#define QUIT_STRING "quit"
#define MENU_STRING "?"
#define LOAD_STRING "load"

/*------------------------------------------------------------------------------
	STRUCTURES:
------------------------------------------------------------------------------*/

typedef struct window_struct {
    char    name[MAX_STRING+1];
    WINDOW  *id;
    pid_t   pid;
    char    options[MAX_STRING+1];
    struct window_struct *next;
} window_t;

typedef struct command_data_struct {
    struct window_struct *window;
    char command[MAX_STRING+1];
    struct command_data_struct *next;
} command_data_t;

typedef struct command_struct {
    char    tag[MAX_STRING+1];
    struct command_data_struct *data;
    struct command_struct *next;
} command_t;

typedef struct window_list_struct {
    window_t *first;
    window_t *last;
} window_list_t;

typedef struct command_list_struct {
    command_t *first;
    command_t *last;
} command_list_t;

/*------------------------------------------------------------------------------
	VARIABLES:
------------------------------------------------------------------------------*/

window_list_t  Windows;
command_list_t Commands;

/*------------------------------------------------------------------------------
	FUNCTION PROTOTYPES:
------------------------------------------------------------------------------*/

static void *sig_child_handler();


/*------------------------------------------------------------------------------

	FUNCTION:  startup_interactive_from_file

	CLASS:

	DESCRIPTION:

	   This function is the main looping mechanism for the interactive mode
	   of the start program. It initializes signal handlers, reads the con-
	   figuration file and loops on the user input commands.
	
	BUGS:  none

	PARAMETERS:
	   file_name - name of configuration file
	   tags - list of command line input tags (argv)
	   num_tags - number of command line input tags (argc)
	   do_startup - flag which is ignored; maintains consistency
	   verbose - flag for informational printfs

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Windows:  initialized to NULL
	  Commands: initialized to NULL

	RETURNS:  n/a

------------------------------------------------------------------------------*/

startup_interactive_from_file(file_name, tags, num_tags, do_startup, verbose)
    char *file_name;
    char **tags;
    int num_tags;
    int do_startup;
    int verbose;
{
    char command_line[MAX_STRING+1];
    char command[MAX_STRING+1];

    if (verbose)
        do_verbose(file_name,tags,num_tags,FALSE);

    /* initialize signal handler to trap disappearance of windows */
    if (signal(SIGCHLD, sig_child_handler) == -1)
      perror("signal error");

    /* initialize global variables */
    Commands.first = Commands.last = NULL;
    Windows.first = Windows.last = NULL;

    /* read config file and setup structures */
    if (file_name)
      init_interactive(file_name,verbose,tags,num_tags);
    else {
	fprintf(stdout,"\nNo configuration file specified.\n");
	fprintf(stdout,"Use the load command.\n\n");
    }

    /* loop until quit */
    sprintf(command,"");
    fputs("command: ",stdout);  
    gets(command_line);
    sscanf(command_line,"%s",command);
    while ((strcmp(command,EXIT_STRING) != 0) && (strcmp(command,QUIT_STRING) != 0)){
	if (!strcmp(command,MENU_STRING)) /* if user request option menu */
	  print_user_menu();
	else if (!strcmp(command,LOAD_STRING)) /* else if user request file load */
	  load_config_files(verbose,command_line);
	else			/* otherwise send command to all windows */
	  send_all_commands(command,verbose);
	fputs("command: ",stdout);
	gets(command_line);
	sscanf(command_line,"%s",command);
    }
    if (!strcmp(command,EXIT_STRING)) /* check if EXIT vs QUIT */
      quit_all_windows(verbose); /* destroy the windows */
}


/*------------------------------------------------------------------------------

	FUNCTION:  new_window

	CLASS:

	DESCRIPTION:
	  This function allocates a new window_t structure and inserts the new
	  window into the Windows list.

	  This function will exit with notice in no space is available to
	  allocate.

	BUGS: n/a

	PARAMETERS: n/a

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Windows:  updated with the new window

	RETURNS:  pointer to the new window_t structure

------------------------------------------------------------------------------*/

window_t *new_window(name)
     char *name;
{
    window_t *new;
    window_t *win;

    /* search for window name in current window list */
    for (win=Windows.first;win!=NULL;win=win->next)
      if (!strcmp(name,win->name))
	return(win);

    /* allocate the new structure */
    if ((new = (window_t *)calloc(1,sizeof(window_t))) == NULL) {
	printf("unable to allocate memory - new_window\n");
	exit(1);
    }

    strcpy(new->name,name);
    if (Windows.first == NULL)	/* if no windows present in Windows list */
      Windows.first = new;	/* place at beginning */
    else			
      Windows.last->next = new;	/* otherwise place at end */
    Windows.last = new;
    return(new);		/* return pointer to window structure */
}


/*------------------------------------------------------------------------------

	FUNCTION:  new_command_data

	CLASS:

	DESCRIPTION:
	  This function allocates a new command data type and updates the 
	  command with a pointer to the window which uses this command.

	  This function will exit with notice in no space is available to
	  allocate.

	BUGS: n/a

	PARAMETERS:
	  window:  pointer to the window which is associated with this command

	GLOBAL VARIABLES AND SIDE EFFECTS: n/a

	RETURNS: pointer to new command_data_t structure

------------------------------------------------------------------------------*/

command_data_t *new_command_data(window)
     window_t *window;
{
    command_data_t *new;

    /* allocate the new structure */
    if ((new = (command_data_t *)calloc(1,sizeof(command_data_t))) == NULL) {
	printf("unable to allocate memory - new_command_data\n");
	exit(1);
    }
    new->window = window;	/* update the window pointer */
    return(new);		/* return pointer to command data structure*/
}


/*------------------------------------------------------------------------------

	FUNCTION:  new_command

	CLASS:

	DESCRIPTION:
	  This function allocates a new command_t structure. If the command 
	  already exists, then the structure is not allocated, rather the 
	  existing command data list is updated with a new command_data_t
	  structure. If the command does not exists, the new command is added
	  to the Commands list and the command is updated with the command_data.

	  This function will exit with notice in no space is available to
	  allocate.

	BUGS: n/a

	PARAMETERS:
	  tag:  the keyword associated with a unix command
	  window:  pointer to the window associated with this command

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Commands:  updated with the new command

	RETURNS:  pointer to command_t structure

------------------------------------------------------------------------------*/

command_t *new_command(tag,window)
     char *tag;
     window_t *window;
{
    command_t *new;
    command_t *cmd;
    command_data_t *data;
    
    /* if tag is already in list then update with window pointer */

    for (cmd = Commands.first;cmd != NULL;cmd = cmd->next)
        if (!strcmp(cmd->tag,tag)) {
	    for(data=cmd->data;data!=NULL;data=data->next)
	      if (data->window == window)
		return(cmd);
	    data = new_command_data(window); /* get new command data */
	    data->next = cmd->data; /* update data list */
	    cmd->data = data;
            return(cmd);	/* return pointer to command structure*/
	}

    /* else add new command to list */

    /* allocate new structure */
    if ((new = (command_t *)calloc(1,sizeof(command_t))) == NULL) {
	printf("unable to allocate memory - new_command\n");
	exit(1);
    }
    strcpy(new->tag,tag);	/* update new command with tag */

    data = new_command_data(window); /* get new command data */
    if (new->data)		/* update data list */
      data->next = new->data;
    new->data = data;

    if (Commands.first == NULL)	/* if no commands exist in Commands list */
      Commands.first = new;	/* place at beginning */
    else
      Commands.last->next = new; /* otherwise place at end */
    Commands.last = new;

    return(new);		/* return pointer to command structure*/
}

/*------------------------------------------------------------------------------

	FUNCTION:  init_interactive

	CLASS:

	DESCRIPTION:
	  This function initializes the window and command lists (Windows, 
	  Commands) with the information from the configuration file and creates
	  the xterm windows. 

	  The code for this function was based on the function 'startup_from_file' 
	  in the source file start.c acquired from CMU. In-line comments were 
	  added.

	  This function exits with notice if the configuration file cannot be 
	  opened.

	BUGS: n/a

	PARAMETERS:
	  filename: name of configuration file
	  verbose:  flag to turn on/off inforamtional printouts
	  tags:     list of keywords on command line (argv)
	  num_tags: number of keywords in list (argc)

        GLOBAL VARIABLES AND SIDE EFFECTS: n/a

	RETURNS: 
	  TRUE if file successfully loaded, otherwise returns FALSE.

------------------------------------------------------------------------------*/

int init_interactive(filename,verbose,tags,num_tags)
char *filename;
int   verbose;
char **tags;
int num_tags;
{
    FILE *fp;
    STATE_TYPE state;
    int lineno;
    char c;
    char str[MAX_STRING+1];
    int quoted;
    window_t *window_ptr;
    command_t *command_ptr;
    int continue_flag;

    if (verbose)		/* if verbose printout information stuff */
      printf("Reading config files.\n");

    fp = fopen(filename,"r");	/* open file */
    if (!fp) {
	printf("%s\n",filename);
        perror("Unable to open startup file");
	return(FALSE);
    }
    state = INITIAL;		/* initialize state and lineno */
    lineno = 1;

    while ((c = getc(fp)) != EOF) { /* loop on get character */
	if (c == '\n') {	
	    lineno++;
	    continue;
	}
	else if (isspace(c)) 
	  continue;
	else if (c == COMMENT_CHAR) {
	    skip_line(fp);
	    lineno++;
	    continue;
	}
	else if (c == '(') {
	    switch (state) {
	      case NAMED:
		state = OPTION;
		break;
	      case COMMAND:
		state = TAG;
		break;
	      default:
		syntax_error(lineno);
	    }
	    continue;
	}
	else if (c == ')'){
	    switch (state) {
	      case OPTION_READ:
		state = NAMED;
		break;
	      case GOT_COMMAND:
		state = INSIDE;
		break;
	      default:
		syntax_error(lineno);
	    }
	    continue;
	}
	else if (c == '{') {
	    switch (state) {
	      case NAMED:
		state = INSIDE;
		break;
	      default:
		syntax_error(lineno);
	    }
	    continue;
	}
	else if (c == '}') {
	    switch (state) {
	      case INSIDE:
		state = INITIAL;
		break;
	      default:
		syntax_error(lineno);
	    }
	    continue;
	}
	else if (c == ',') {
	    switch (state) {
	      case TAG_COMMA:
		state = GET_COMMAND;
		break;
	      default:
		syntax_error(lineno);
	    }
	    continue;
	}

        lineno += read_string(fp,c,str,MAX_STRING,&quoted);
        switch (state) {
          case INITIAL:
	    if (!strcasecmp(INCLUDE_STRING,str)) { /* if include then set state appropriately */
		state = INCLUDE;
		break;
	    }
	    window_ptr = new_window(str); /* if initial state create new window structure */
	    state = NAMED;	/* update state to reflect anticipated input */
            break;
	  case INCLUDE:
	    init_interactive(str,verbose,tags,num_tags);
	    state = INITIAL;
	    break;
          case OPTION:
            strcpy(window_ptr->options,str); /* copy window options */
            state = OPTION_READ; /* update state to reflect anticipated input */
            break;
          case INSIDE:
            if (strcmp(COMMAND_STRING,str)) /* test command format */
                syntax_error(lineno);
            state = COMMAND;	/* update state to reflect anticipated input */
            break;
          case COMMAND:		/* if startup command execute without saving data */
            if (num_tags == 0) {
                if (window_ptr->id == NULL) {
		    window_ptr->id = setup_window(window_ptr->name,window_ptr->options,
						  FALSE,verbose);
		    window_ptr->pid = get_last_pid(); /* save process id */
		}
		send_command(window_ptr->id,str,quoted,verbose);
            }
            state = INSIDE;	/* update state to reflect anticipated input */
            break;
          case TAG:		/* if tag format, create new command structure */
	    command_ptr = new_command(str,window_ptr); 
            state = TAG_COMMA;	/* update state to reflect anticipated input */
            break;
          case GET_COMMAND:
	    strcpy(command_ptr->data->command,str); /* copy unix command */
	    if (window_ptr->id == NULL) {
		window_ptr->id = setup_window(window_ptr->name,window_ptr->options,
					      FALSE,verbose);
		window_ptr->pid = get_last_pid(); /* save process id */
		command_ptr->data->window = window_ptr;	/* associate command list with window */
	    }
            if (tag_matches(command_ptr->tag,tags,num_tags)) { /* if tag in command line */
		send_command(window_ptr->id,str,quoted,verbose); /* execute immediately */
            }
            state = GOT_COMMAND; /* update state to reflect anticipated input */
            break;
          default:
            syntax_error(lineno);
	}
	continue;
    }
    fclose(fp);
    return(TRUE);
}


/*------------------------------------------------------------------------------

	FUNCTION: print_user_menu

	CLASS:

	DESCRIPTION:
	  This function output to the standard output the list of options from
	  which the user can choose.

	BUGS: n/a

	PARAMETERS: n/a

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Commands: referenced and not modified

	RETURNS: n/a

------------------------------------------------------------------------------*/

print_user_menu()
{
    command_t *cmd;

    fprintf(stdout,"\nSelect from one of the following options:\n\n");

    for(cmd=Commands.first;cmd!=NULL;cmd=cmd->next) /* loop through all commands */
	fprintf(stdout,"%s\n",cmd->tag);
    fprintf(stdout,"%s \tquit start and leave the windows\n",QUIT_STRING);
    fprintf(stdout,"%s \texit start and kill the windows\n",EXIT_STRING);
    fprintf(stdout,"%s \tload configuration files\n\n",LOAD_STRING);
}



/*------------------------------------------------------------------------------

	FUNCTION:  send_all_commands

	CLASS:

	DESCRIPTION:
	  This function loops through the command list searching for the input
	  command. When and if the command is found, the associated unix command
	  is sent to each associated window.

	BUGS: n/a

	PARAMETERS:
	  commands:
	  verbose:

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Commands:  referenced and not modified

	RETURNS: n/a

------------------------------------------------------------------------------*/

send_all_commands(command,verbose)
    char *command;
    int verbose;
{
    command_data_t *data;
    int leave_alone;
    command_t *cmd;

    leave_alone = TRUE;		/* do not at new-line character to end of command */

    if (verbose)		/* if verbose print informative stuff */
      fprintf(stdout,"Sending command %s to windows.\n",command);

    /* loop through command list; if command matches tag send unix command to all
       windows associated with that command */

    for(cmd=Commands.first; cmd!=NULL; cmd=cmd->next) {	
	if (!strcmp(cmd->tag,command)) {
	    for (data=cmd->data;data!=NULL;data=data->next) 
	      send_command(data->window->id,data->command,leave_alone,verbose);
	    return;
	}
    }
    printf("\tUnknown command.\n");
}


/*------------------------------------------------------------------------------

	FUNCTION: quit_all_windows

	CLASS:

	DESCRIPTION:
	  This function loops through the list of windows and sends an exit to
	  each xterm window.

	BUGS: n/a

	PARAMETERS:
	  verbose: flag to print informative stuff to user

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Windows:  referenced and not modified

	RETURNS: n/a

------------------------------------------------------------------------------*/

quit_all_windows(verbose)
     int verbose;
{
    window_t *window;
    int leave_alone;

    leave_alone = TRUE;		/* do not put new-line at end of command */

    if (verbose)
      fprintf(stdout,"Terminating program and exiting all windows.\n");

    /* cycle through windows and exit each one */
    for (window=Windows.first;window!=NULL;window=window->next)
      send_command(window->id,"exit\n",leave_alone,verbose);
}



/*------------------------------------------------------------------------------

	FUNCTION:  load_config_file

	CLASS:

	DESCRIPTION:

	BUGS:

	PARAMETERS:
	  verbose: flag to print informative stuff to user

	GLOBAL VARIABLES AND SIDE EFFECTS:

	RETURNS:

------------------------------------------------------------------------------*/

load_config_files(verbose,commands)
     int verbose;
     char *commands;
{
    char filename[MAX_STRING+1];
    char *token;

    commands = &commands[5];	/* by pass the load string */

    if (verbose)
      fprintf(stdout,"Loading config files: %s\n",commands);

    token = strtok(commands," ");
    while (token) {
	init_interactive(token,verbose,NULL,0);
	token = strtok(NULL," ");
    }
}

/*------------------------------------------------------------------------------

	FUNCTION: sig_child_handler

	CLASS:

	DESCRIPTION:
	  Signal handler for the SIGCHLD signal. When this function is called
	  it determines the process id of the child process which caused the 
	  signal. Given the process id, the window list is search and the xterm
	  window is restarted. The user is notified of the events.

	BUGS: na/

	PARAMETERS: n/a

	GLOBAL VARIABLES AND SIDE EFFECTS:
	  Windows: referenced and updated with the new process id.

	RETURNS: void

------------------------------------------------------------------------------*/

static void *sig_child_handler()
{
    pid_t pid;
    int   status;
    window_t *window;

    if (signal((int)SIGCHLD,(void *)sig_child_handler) == -1) /* reestablish handler */
      perror("signal error");

    if ( (pid = wait(&status)) < 0) /* fetch child status */
      perror("wait error");

    /* search window list for process id */
    for (window=Windows.first; window->pid != pid; window=window->next);
      
    /* notify user of events */
    printf("\n\nWARNING! window %s exited.\n",window->name);
    printf("\tRestarting window with original options.\n\n");

    /* recreate the xterm window with the original options */
    window->id = setup_window(window->name,window->options,0,0);
    window->pid = get_last_pid();
    
    return;			/* interrupts pause() */
}
