%module _ConfigFile
%{
#include <utils/ConfigFile.h>
#include <utils/ConfigElem.h>
#include <utils/Output.h>
#include <utils/Input.h>
#include <utils/SpecElem.h>

typedef utils::ConfigFile ConfigFile;
typedef utils::ConfigElem ConfigElem;

static PyObject* string_val(ConfigElem* elem) {
    utils::Output output;
    output.setBuffer(NULL, 0, utils::Output::standardResize);
    elem->writeData(output);
    output.write('\0');
    void* bufptr;
    int size;
    output.getBuffer(bufptr, size);

    // get rid of trailing semicolon if there
    char* val = (char*) bufptr;
    if (val[size-2] == ';')
      val[size-2] ='\0';
    
    PyObject* res = Py_BuildValue("s", val);
    delete [] val;
    return res;
}

%}

// When we add a method that takes a python object
%typemap(python, in) PyObject* {
    $target = $source;
}

// When we add a method that returns a python object,
// it should be returned as a python object
%typemap(python, out) PyObject* {
    $target = $source;
}

%name(_ConfigFile) class ConfigFile {
public:
    ConfigFile(ConfigElem* elem=0);
    ~ConfigFile();

    bool open(const char* name);
    bool parse(const char* source);
    void close();

    %name(parseSet) ConfigElem* set(const char* name, const char* value,
                                    bool ignore_duplicates=1);
    ConfigElem* lookup(const char* name);

    %addmethods {
        PyObject* set(const char* name, PyObject* val,
                      bool ignore_duplicates=1) {
            PyObject* repr;
            bool res;
            if (val == Py_None) {
              Py_INCREF(Py_None);
              return Py_None;
            }
            if (PyString_Check(val)) {
                res=self->set(name, PyString_AsString(val), ignore_duplicates);
            } else if (PySequence_Check(val)) {
                utils::Output output;
                output.setBuffer(NULL, 0, utils::Output::standardResize);
                int len = PySequence_Length(val);
                PyObject* item;
                for (int i=0; i<len; i++) {
                    item = PySequence_GetItem(val, i);
                    repr = PyObject_Repr(item);
                    output.write(PyString_AsString(repr));
                    if (i!=len-1)
                        output.write(' ');
                    Py_DECREF(repr);
                    Py_DECREF(item);
                }
                output.write('\0');

                void* buf_ptr;
                int size;
                output.getBuffer(buf_ptr, size);
                res=self->set(name, (const char*) buf_ptr, ignore_duplicates);
                delete (char*) buf_ptr;
            } else {
                repr = PyObject_Repr(val);
                res=self->set(name, PyString_AsString(repr),
                              ignore_duplicates);
                Py_DECREF(repr);
            }

            if (!res) {
                PyErr_SetString(PyExc_RuntimeError, "Unable to set");
                return NULL;
            }
            Py_INCREF(Py_None);
            return Py_None;
        }

        PyObject* get(const char* name, PyObject* def=Py_None) {
            ConfigElem* elem = self->lookup(name);
            if (!elem || elem->numValues() == 0) {
              PyObject* set_res = ConfigFile_set(self, name, def, false);
              if (!set_res) {
                PyErr_SetString(PyExc_RuntimeError,
                                "Invalid default data");
                return NULL;
              }

              Py_DECREF(set_res);
              elem = self->lookup(name);
              if (!elem) {
                Py_INCREF(def);
                return def;
              }
            }

            if (elem->numValues() < 0) {
                PyErr_SetString(PyExc_RuntimeError, "No data");
                return NULL;
            }

            if (elem->numValues() == 1) {
                if (elem->getTypeId() == ConfigFile::intType())
                    return Py_BuildValue("i", self->getInt(name));
                if (elem->getTypeId() == ConfigFile::floatType())
                    return Py_BuildValue("d", self->getFloat(name));
                if (elem->getTypeId() == ConfigFile::doubleType())
                    return Py_BuildValue("d", self->getDouble(name));
                if (elem->getTypeId() == ConfigFile::charType())
                    return Py_BuildValue("i", self->getChar(name));
                if (elem->getTypeId() == ConfigFile::stringType()) 
                    return Py_BuildValue("s", self->getString(name));
                if (elem->getTypeId() == ConfigFile::boolType())
                    return Py_BuildValue("i", (int) self->getBool(name));

                return string_val(elem);
            }

            if (elem->numValues() == 0 &&
                (elem->getTypeId() == ConfigFile::stringType())) {
              return Py_BuildValue("s", "");
            }                  

            PyObject* res = PyTuple_New(elem->numValues());
            int i;
            if (elem->getTypeId() == ConfigFile::intType() ||
                elem->getTypeId() == ConfigFile::boolType() ||
                elem->getTypeId() == ConfigFile::charType()) {
                int* data = new int[elem->numValues()];
                elem->getValue(ConfigFile::intType(), data,elem->numValues());
                for (i=0;i<elem->numValues();i++) 
                    PyTuple_SET_ITEM(res, i, PyInt_FromLong(data[i]));
                delete [] data;
            } else if (elem->getTypeId() == ConfigFile::floatType() ||
                elem->getTypeId() == ConfigFile::doubleType()) {
                double* data = new double[elem->numValues()];
                elem->getValue(ConfigFile::doubleType(), data,
                                elem->numValues());
                for (i=0;i<elem->numValues();i++) 
                    PyTuple_SET_ITEM(res, i, PyFloat_FromDouble(data[i]));
                delete [] data;
            } else if (elem->getTypeId() == ConfigFile::stringType()) {
                const char** data = new const char*[elem->numValues()];
                elem->getValue(ConfigFile::stringType(), data,
                                elem->numValues());
                for (i=0;i<elem->numValues();i++) 
                    PyTuple_SET_ITEM(res, i, PyString_FromString(data[i]));
                delete [] data;
            } else 
                return string_val(elem);
            return res;
        }

        PyObject* output() {
            return string_val(self->getRoot());
        }
    }
};



