///////////////////////////////////////////////////////////////////////////////
//
//                               PyAction.cc
//
// Implements classes for packing formatted C data into python structures and
// unpacking python structures into formatted C data
//
// Classes implemented for export
//   PyPackAction - format class which packs data into a python object
//   PyUnpackAction - format class which unpacks data from a python object
//
// Routines implemented for export:
//   initPyFormatting - must be called for python packing and unpacking to work
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>

#include <utils/Vector.h>
#include <utils/formatting/FormatParser.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/VectorSpec.h>

#include "PyPackAction.h"
#include "PyUnpackAction.h"

#include <Python.h>

extern "C" void SWIG_MakePtr(char*, void*, char*);
extern "C" char* SWIG_GetPtr(char*, void**, char*);


///////////////////////////////////////////////////////////////////////////////
// PyPackAction - packs data into a python object
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PyPackAction);

// Convert data into a python object using spec and return that object
PyObject* PyPackAction::pack(utils::FormatSpec* spec, void* data)
{
    if (!spec || !data)
        return NULL;

    PyPackAction action(data);
    if (!spec->act(&action))
        return NULL;
    Py_INCREF(action.getObject());
    return action.getObject();
}

// create a packer action.  We will pack data into a python object
PyPackAction::PyPackAction(void* data) : utils::DataAction(data)
{
    UTILS_BASE_CONSTRUCTOR(PyPackAction);

    _object = Py_None;
    Py_INCREF(_object);
        
}

PyPackAction::~PyPackAction()
{
    Py_XDECREF(_object);
}

void PyPackAction::setObject(PyObject* obj)
{
    if (_object == obj)
        return;
    Py_XDECREF(_object);
    _object = obj;
    Py_XINCREF(obj);
}

void PyPackAction::initClass()
{
    UTILS_BASE_INIT_CLASS(PyPackAction,
                          "PyPackFormatAction", "DataFormatAction");
}

bool PyPackAction::actLength(int len)
{
    unsigned char* buffer = curData();
    PyObject* tuple = PyTuple_New(len);
    for (int i=0;i<len;i++) 
        PyTuple_SetItem(tuple, i, PyInt_FromLong((long) buffer[i]));
    setObject(tuple);
    Py_DECREF(tuple);
    return utils::DataAction::actLength(len);
}

void PyPackAction::packLong(long l)
{
    PyObject* l_obj = PyInt_FromLong(l);
    setObject(l_obj);
    Py_DECREF(l_obj);
}

void PyPackAction::packDouble(double d)
{
    PyObject* d_obj = PyFloat_FromDouble(d);
    setObject(d_obj);
    Py_DECREF(d_obj);
}

bool PyPackAction::actChar()
{
    packLong((long) *(char*)curData());
    return utils::DataAction::actChar();
}

bool PyPackAction::actShort()
{
    packLong((long) *(short*)curData());
    return utils::DataAction::actShort();
}

bool PyPackAction::actLong()
{
    packLong(*(long*)curData());
    return utils::DataAction::actLong();
}

bool PyPackAction::actDouble()
{
    packDouble(*(double*)curData());
    return utils::DataAction::actDouble();
}

bool PyPackAction::actFloat()
{
    packDouble((double) *(float*)curData());
    return utils::DataAction::actFloat();
}

bool PyPackAction::actInt()
{
    packLong((int) *(long*)curData());
    return utils::DataAction::actInt();
}

// strings are just packed as strings
bool PyPackAction::actString()
{
    char* str = (char*) getSubData();
    if (!str) 
        str = "";
    PyObject* str_obj = PyString_FromString(str);
    setObject(str_obj);
    Py_DECREF(str_obj);

    return utils::DataAction::actString();
}

bool PyPackAction::actPtr(utils::FormatSpec* fmt)
{
    unsigned char* data = getSubData();

    if (data) {
        PyPackAction subaction(data);
        subaction.inherit(*this);
        fmt->act(&subaction);
        setObject(subaction.getObject());
    } else 
        setObject(Py_None);

    return utils::DataAction::actPtr(fmt);
}

bool PyPackAction::actFixedArray(const utils::Vector<int>& sizes,
                                 utils::FormatSpec* fmt)
{
    int size = fixedArraySize(sizes);
    PyObject* tuple = PyTuple_New(size);
    int i;
    for (i=0;i<size;i++) {
        if (!fmt->act(this))
            return false;
        Py_INCREF(getObject());
        PyTuple_SetItem(tuple, i, getObject());
    }
    setObject(tuple);
    Py_DECREF(tuple);
    return true;
}

bool PyPackAction::actVarArray(utils::StructSpec* parent,
                               const utils::Vector<int>& indices,
                               utils::FormatSpec* fmt)
{
    int size = varArraySize(parent, indices);
    PyObject* tuple = PyTuple_New(size);
    PyPackAction subaction(getSubData());
    int i;
    for (i=0;i<size;i++) {
        if (!fmt->act(&subaction))
            return false;
        Py_INCREF(subaction.getObject());
        PyTuple_SetItem(tuple, i, subaction.getObject());
    }
    setObject(tuple);
    Py_DECREF(tuple);
    return utils::DataAction::actVarArray(parent, indices, fmt);
}

bool PyPackAction::actStruct(int size, utils::FormatSpec** elems)
{
    PyObject* tuple = PyTuple_New(size);
    if (!utils::DataAction::startStruct())
        return false;

    for (int i = 0; i<size; i++) {
        if (!utils::DataAction::actStructElem(i, size, elems))
            return false;
        Py_INCREF(getObject());
        PyTuple_SetItem(tuple, i, getObject());
    }
    if (!utils::DataAction::endStruct())
        return false;
    setObject(tuple);
    Py_DECREF(tuple);
    return true;
}

///////////////////////////////////////////////////////////////////////////////
// PyUnpackAction - unpacks data from a python object
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PyUnpackAction);

bool PyUnpackAction::unpack(utils::FormatSpec* spec, PyObject* obj, void* data)
{
    if (!spec || !data)
        return false;

    PyUnpackAction action(obj, data);
    bool res = spec->act(&action);
    action.setObject(NULL);
    return res;
}

void* PyUnpackAction::unpack(utils::FormatSpec* spec, PyObject* obj)
{
    if (!spec)
        return NULL;
    unsigned char* res = new unsigned char[spec->dataSize()];
    if (!unpack(spec, obj, res)) {
        delete res;
        return NULL;
    }
    return res;
}

// create a unpacker action.  We will unpack data into a python object
PyUnpackAction::PyUnpackAction(PyObject* src, void* data)
    : utils::DataAction(data)
{
    UTILS_BASE_CONSTRUCTOR(PyUnpackAction);

    _object = NULL;
    setObject(src);
}

PyUnpackAction::~PyUnpackAction()
{
    Py_XDECREF(_object);
}

void PyUnpackAction::setObject(PyObject* obj)
{
    if (_object == obj)
        return;
    Py_XDECREF(_object);
    _object = obj;
    Py_XINCREF(obj);
}

void PyUnpackAction::initClass()
{
    UTILS_BASE_INIT_CLASS(PyUnpackAction,
                       "PyUnpackFormatAction", "DataFormatAction");
}

bool PyUnpackAction::actLength(int len)
{
    PyObject* tuple = getObject();
    if (!PyTuple_Check(tuple))
        return false;

    unsigned char* dest = curData();
    for (int i=0; i<PyTuple_Size(tuple); i++) 
        *dest++ = (unsigned char) PyInt_AS_LONG(PyTuple_GET_ITEM(tuple, i));
    return utils::DataAction::actLength(len);
}

bool PyUnpackAction::unpackLong(long& l)
{
    if (PyInt_Check(getObject())) {
        l = PyInt_AsLong(getObject());
        return true;
    } else if (PyFloat_Check(getObject())) {
        l = (long) PyFloat_AsDouble(getObject());
        return true;
    } else
        return false;
}

bool PyUnpackAction::unpackDouble(double& d)
{
    if (PyFloat_Check(getObject())) {
        d = PyFloat_AsDouble(getObject());
        return true;
    } else if (PyInt_Check(getObject())) {
        d = (double) PyInt_AsLong(getObject());
        return true;
    } else
        return false;
}

bool PyUnpackAction::actChar()
{
    long l;
    if (!unpackLong(l))
        return false;
    unsigned char c = (unsigned char) l;
    fromData(&c, sizeof(unsigned char));
    return true;
}

bool PyUnpackAction::actShort()
{
    long l;
    if (!unpackLong(l))
        return false;
    short s = (short) l;
    fromData(&s, sizeof(short));
    return true;
}

bool PyUnpackAction::actLong()
{
    long l;
    if (!unpackLong(l))
        return false;
    fromData(&l, sizeof(long));
    return true;
}

bool PyUnpackAction::actDouble()
{
    double d;
    if (!unpackDouble(d))
        return false;
    fromData(&d, sizeof(double));
    return true;
}

bool PyUnpackAction::actFloat()
{
    double d;
    if (!unpackDouble(d))
        return false;
    float f = (float) d;
    fromData(&f, sizeof(float));
    return true;
}

bool PyUnpackAction::actInt()
{
    long l;
    if (!unpackLong(l))
        return false;
    int i = (int) l;
    fromData(&i, sizeof(int));
    return true;
}

// strings are just unpacked as strings
bool PyUnpackAction::actString()
{
    PyObject* pystr = getObject();
    if (!PyString_Check(pystr))
        return false;

    int len = PyString_Size(pystr);
    char* new_str = new char[len+1];
    bcopy(PyString_AsString(pystr), new_str, len);
    new_str[len] = '\0';
    
    fromData(&new_str, sizeof(char*));
    return true;
}

bool PyUnpackAction::actPtr(utils::FormatSpec* fmt)
{
    unsigned char* newStruct;
    PyObject* obj = getObject();
    if (obj == Py_None) {
        newStruct = NULL;
    } else {
        newStruct = new unsigned char[fmt->dataSize()];
        PyUnpackAction subaction(obj, newStruct);
        subaction.inherit(*this);
        if (!fmt->act(&subaction))
            return false;
    }

    fromData(&newStruct, sizeof(unsigned char*));
    return true;
}

bool PyUnpackAction::actFixedArray(const utils::Vector<int>& sizes,
                                       utils::FormatSpec* fmt)
{
    PyObject* seq = getObject();
    if (!seq || !PySequence_Check(seq))
        return false;
    Py_INCREF(seq);

    int size = PySequence_Length(seq);
    if (size < 0)
        return false;
    if (size > fixedArraySize(sizes)) 
        size = fixedArraySize(sizes);
    int i;
    for (i=0;i<size;i++) {
        PyObject* item = PySequence_GetItem(seq, i);
        setObject(item);
        Py_DECREF(item);
        if (!fmt->act(this)) {
            Py_DECREF(seq);
            return false;
        }
    }

    Py_DECREF(seq);
    return true;
}

bool PyUnpackAction::actVarArray(utils::StructSpec*, const utils::Vector<int>&,
                                 utils::FormatSpec* fmt)
{
    PyObject* seq = getObject();
    if (!seq || !PySequence_Check(seq))
        return false;

    int size = PySequence_Length(seq);
    unsigned char* newStruct = new unsigned char[size*fmt->dataSize()];
    fromData(&newStruct, sizeof(unsigned char*));

    PyUnpackAction subaction(NULL, newStruct);
    subaction.inherit(*this);
    for (int i=0; i<size; i++) {
        PyObject* item = PySequence_GetItem(seq, i);
        subaction.setObject(item);
        Py_DECREF(item);
        if (!fmt->act(&subaction))
            return false;
    }
    return true;
}

bool PyUnpackAction::actStruct(int size, utils::FormatSpec** elems)
{
    PyObject* seq = getObject();
    if (!seq || !PySequence_Check(seq))
        return false;

    if (!utils::DataAction::startStruct())
        return false;
    Py_INCREF(seq);

    bool res;
    if (size == PySequence_Length(seq)) {
        for (int i = 0; i<size; i++) {
            PyObject* item = PySequence_GetItem(seq, i);
            setObject(item);
            Py_DECREF(item);
            if (!utils::DataAction::actStructElem(i, size, elems))
                return false;
        }
        res = true;
    } else
        res = false;
    Py_DECREF(seq);
    if (!res)
        return false;
    return utils::DataAction::endStruct();
}



///////////////////////////////////////////////////////////////////////////////
// Format routines for the custom format specifiers
///////////////////////////////////////////////////////////////////////////////

static bool py_pack_list(utils::FormatAction* action)
{
    utils::VectorSpec* spec = (utils::VectorSpec*) action->get("listSpec");
    if (!spec)
        return false;

    PyPackAction* pa = (PyPackAction*) action;
    utils::Vector<char>* list = (utils::Vector<char>*) pa->getSubData();

    PyObject* res;
    if (!list) 
        res = PyList_New(0);
    else {
        res = PyList_New(list->numElems());
        int size_type = spec->getFormat()->dataSize();
        char* data = list->getData();     // get data
        for (int i=0;i<list->numElems();i++) {
            PyPackAction subaction(data + i*size_type);
            subaction.inherit(*action);
            if (!spec->getFormat()->act(&subaction))
                return false;
            Py_INCREF(subaction.getObject());
            PyList_SetItem(res, i, subaction.getObject());
        }
    }
    pa->setObject(res);
    pa->addSize(sizeof(utils::Vector<char>*));
    Py_DECREF(res);
    return true;
}

static bool py_unpack_list(utils::FormatAction* action)
{
    utils::VectorSpec* spec = (utils::VectorSpec*) action->get("listSpec");
    if (!spec)
        return false;

    PyUnpackAction* ua = (PyUnpackAction*) action;

    PyObject* pyseq = ua->getObject();
    if (!PySequence_Check(pyseq))
        return false;
    int size = PySequence_Length(pyseq);
    int size_type = spec->getFormat()->dataSize();
    utils::SizedList* list = new utils::SizedList(size*size_type);
    
    char* data = list->getData();     // get data
    PyUnpackAction subaction(NULL, data);
    subaction.inherit(*action);
    for (int i=0;i<size;i++) {
        PyObject* item = PySequence_GetItem(pyseq, i);
        if (item) {
            subaction.setObject(item);
            if (!spec->getFormat()->act(&subaction))
                return false;
            Py_DECREF(item);
        } else
            return false;
    }

    list->finishSized(size_type, size);
    ua->fromData(&list, sizeof(utils::SizedList*));
    ua->needsPacking();
    return true;
}

void initPyFormatting(utils::FormatParser& parser)
{
    static utils::FormatParser* prev_parser = NULL;

    if (!prev_parser) {
        PyPackAction::initClass();
        PyUnpackAction::initClass();
    }

    if (prev_parser == &parser)
        return;
    prev_parser = &parser;

    utils::VectorSpec* list_spec = NULL;
    bool prev_verb = parser.getErrorVerbosity();
    parser.setErrorVerbosity(false);
    list_spec = (utils::VectorSpec*) parser.parseString("list");
    parser.setErrorVerbosity(prev_verb);
    if (!list_spec) {
        list_spec = new utils::VectorSpec;
        parser.registerFormat("list", list_spec);
    }
    list_spec->
        addFormatRoutine(PyPackAction::getClassTypeId(), py_pack_list, true);
    list_spec->
        addFormatRoutine(PyUnpackAction::getClassTypeId(), py_unpack_list,
                         true);

}
