/* -*-c++-*-
=========================================================================
=                                                                       =
=                                GLUT_WINDOW.CC                                                                       =
=                                                                       =
=========================================================================

Target: VxWorks/UNIX/Generic

Designed by:    Chris Urmson
Written by:     Chris Urmson
Date:           June 2002

Copyright 2002 Carnegie Mellon University
$Author: jayg $
$Date: 2004/04/26 19:58:43 $
$Header: /IUS/jeep/cvsroot/ModUtils/GLUtils/glut_window.cc,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Id: glut_window.cc,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Log: glut_window.cc,v $
Revision 1.1  2004/04/26 19:58:43  jayg
First version of GLUtils ripped from Redteam

Revision 1.7  2003/11/17 05:18:11  curmson
Added some accessor functions to allow non descendant users to tweak mouse and
keyboard input.

Revision 1.6  2003/11/09 19:48:36  mkj
fixed Fl GL init incompatiblities

Revision 1.5  2003/11/05 22:42:41  mkj
warnings removed

Revision 1.4  2003/11/05 22:40:23  mkj
headerfile missing

Revision 1.3  2003/11/05 22:37:24  mkj
mightnot work yet

Revision 1.2  2003/10/01 22:47:58  kp
PGMImage is updated to take const char* rather than just char* in order to
be compatible with module code

glut_window.h is changed to make display() virtual
glut_window.cc is not changed

Revision 1.1  2003/09/06 22:25:04  curmson
First commit of gl utils in this working directory structure.

Revision 1.2  2003/06/05 17:22:40  curmson
changes to allow compile with gcc3.2

Revision 1.1  2003/02/25 14:18:39  curmson
First commit of the new gl_utils library.  This code is somewhat cleaner than the old glHelpers code and is more powerful.

It supports new features including:
        multiple windows
        lighting
        object registration
        ability to set z up or z down orientation

Revision 1.7  2002/11/20 16:21:08  curmson
Added code to draw text at arbitrary locations in the 3d scene.

Revision 1.6  2002/10/14 19:05:10  curmson
Minor changes:
 added an init check in glut_window and added start and stop list calls to gl_display_list_object

Revision 1.5  2002/09/17 13:45:16  curmson
Added the ability to register objects for drawing, thus elliminating the need to subclass glut window in some cases.

Revision 1.4  2002/07/24 18:40:19  curmson
Modified openGL camera positioning routines to render with the correct coordinate system orientations (i.e. X forward, Y to the right, Z down).

Revision 1.3  2002/07/01 21:59:25  curmson
Added more comments and restructured initialization to avoid some warning messages on certain systems.

Revision 1.2  2002/06/26 20:28:24  curmson
removed dependance on share_cmu

Revision 1.1  2002/06/26 01:01:21  curmson
This is a first commit of the util_open_gl module.  It provides a simplified interface to glut and the openGL, and simplifies the task of creating a quick debugging UI.



 ------------------------------------------------------------------------*/
#include "glut_window.h"
#include <math.h>
#include <iostream>

/*
  define the GlutWindow static member*/
Glut_Window * Glut_Window::_windows[GL_MAX_GLUT_WINDOWS];
bool Glut_Window::_initialized = false;
int Glut_Window::_num_windows = 0;
/**
       This creates a new window and sets up the handlers etc. 
       if useTimer is set, a timer is also setup.*/
Glut_Window::Glut_Window(char *title, int width, int height) {
  char * dummy_argv[1];
  int dummy_argc =1;
  dummy_argv[0] = "dummy";
  
  _z_up = false;
  view_elevation= 45;
  view_azimuth = 0;
  view_distance = 50;
  view_x=view_y=view_z=0;
  this->width = width;
  this->height= height;
  if (!_initialized) {
    glutInit(&dummy_argc,dummy_argv);
  }
  glutInitDisplayMode(GLUT_DOUBLE| GLUT_DEPTH | GLUT_RGB);
//    }
  // hook into the callback system
  glutInitWindowSize(width,height);
  window_ID = glutCreateWindow(title);
  glShadeModel(GL_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_DEPTH_TEST);  

  _num_windows++;
  _windows[window_ID]=this;


//    if (!_initialized) {
  glutDisplayFunc(_display);
  glutReshapeFunc(_reshape);
  glutMouseFunc(_mouse);
  glutKeyboardFunc(_key);
  glutSpecialFunc(_specialKey);
  glutMotionFunc(_mouseMotion);
//    }
  step_size=0.05;
  zoom_scale=1.0;
  _initialized=true;

  clip_near =0.5;
  clip_far = 500.0;
  fov_y = 30.0;

  init();
}


void Glut_Window::hide(void) {
  int temp_ID = glutGetWindow();
  glutSetWindow(window_ID);
  glutHideWindow();
  glutSetWindow(temp_ID);
}

void Glut_Window::show(void) {    
  int temp_ID = glutGetWindow();
  glutSetWindow(window_ID);
  glutShowWindow();
  glutSetWindow(temp_ID);
}


/**
   This moves the window around the screen*/
void Glut_Window::position_window(int x,int y) {
  int temp_ID = glutGetWindow();
  glutSetWindow(window_ID);
  glutPositionWindow(x,y);
  glutSetWindow(temp_ID);
}

/**
   This changes the title of the window */
void Glut_Window::set_title(char * title) {
  int temp_ID = glutGetWindow();
  glutSetWindow(window_ID);
  glutSetWindowTitle(title);
  glutSetWindow(temp_ID);
}

Glut_Window::~Glut_Window() {
  glutDestroyWindow(window_ID);
}

/**
   Overide this function to change the location from which the scene is 
   rendered.  It currently uses information from the mouse and the special 
   keyboard keys to set its direction
   @see mouse, position_camera
*/
void Glut_Window::position_camera(void) {
  if (!_z_up) {
    /* z down- default*/
    glTranslated(0.0,0.0,-view_distance);
    glRotated(view_elevation-90,1.0,0.0,0.0);
    glRotated(view_azimuth+90,0.0,0.0,1.0);
    glTranslated(-view_x,view_y,view_z);
    glRotated(180,1,0,0);
  } else {
    /* z up */
    glTranslated(0.0,0.0,-view_distance);
    glRotated(-view_elevation,1.0,0.0,0.0);
    glRotated(view_azimuth,0.0,0.0,1.0);
    glTranslated(-view_x,-view_y,-view_z);
  }
}

void Glut_Window::display(void) {
  glMatrixMode(GL_MODELVIEW);    
  glLoadIdentity();
  position_camera();
  glClear(GL_COLOR_BUFFER_BIT |GL_DEPTH_BUFFER_BIT);    
  glPushMatrix();
  _draw_registered();
  draw();
  glPopMatrix();
  glFlush();
  glutSwapBuffers();
}

/**
   This performs generic matrix setup operations, you shouldn't need
   to override this unless you want to render something without perspective
   @see fov_y, near_clip, far_clip
*/
void Glut_Window::reshape(int w,int h) {
  glViewport(0,0,(GLsizei)w,(GLsizei)h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov_y,(GLfloat)w/(GLfloat)h,clip_near,clip_far);
  this->width = w;
  this->height =h;
}

/**
   This is used in conjunction with mouse_motion to provide the camera 
   movement.  Moving the mouse with the left button down rotates the camera 
   around a fixed point (specified by view_x, view_y, view_z).  Moving the 
   mouse with the right button down zooms the camera in and out
   
   If you would like to perform other operations with the mouse, override
   this function, you may want to consider calling Glut_Window::mouse() 
   in that routine to maintain the mouse interface.
   @see special_key, position_camera
*/
void Glut_Window::mouse(int button, int state, int x, int y) {
  if (state==GLUT_UP) {
    _button_down =-1;
  } else {
    _button_down = button;
    _mouse_start_x =x;
    _mouse_start_y =y;
    //    printf("button down is :%d\n ", button);
    _mouse_start_view_D = (int)view_distance;
    _mouse_start_view_Az = (int)view_azimuth;
    _mouse_start_view_El = (int)view_elevation;
  }
}


/**
   Override this function to cause action something to occur when a key
   is pressed
*/
void Glut_Window::key(unsigned char key,int x, int y) {
}

/**
   This function is used to perform actions when one of the non-ascii keys
   are presed.  In the base class, the arrow keys are used to move the 
   center of the field of view around in the plane, while PAGE_UP & 
   PAGE_DOWN are used to change the Z value of the center of view
   @see mouse, position_camera
*/
void Glut_Window::special_key(int key,int x, int y) {
  if (_z_up) {
    switch (key) {
    case GL_UP_KEY:
      view_x+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_y+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_DOWN_KEY:
      view_x-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_y-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_LEFT_KEY:
      view_x-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_y+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_RIGHT_KEY:
      view_x+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_y-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_PUP_KEY:
      view_z+=step_size;
    break;
    case GL_PDN_KEY:
      view_z-=step_size;
      break;
    default:
      cout << "key is: " << (int) key << endl;
    }
  } else { // !_z_up
    switch(key) {
    case GL_UP_KEY:
      view_y+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_x+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_DOWN_KEY:
      view_y-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_x-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_LEFT_KEY:
      view_y-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_x+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_RIGHT_KEY:
      view_y+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_x-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case GL_PUP_KEY:
        view_z-=step_size;
        break;
    case GL_PDN_KEY:
      view_z+=step_size;
      break;
    default:
      cout << "key is: " << (int) key << endl;
    }
  }
  glutPostRedisplay();
}

/**
   This routine is called whenever a mouse movement event has occured.
   If you are overriding it, consider calling Glut_Window::mouse_motion()
   in your descendant classes, to maintain the mouse interface
   @see mouse
*/
void Glut_Window::mouse_motion(int x, int y) {
  switch (_button_down) {
  case -1:
    return;
  case 2:
    view_distance = _mouse_start_view_D + (y-_mouse_start_y)*zoom_scale/5.0;
    break;
  case 0:
    view_azimuth = _mouse_start_view_Az +(_mouse_start_x-x)/5.0;
    if (!_z_up)
      view_elevation = _mouse_start_view_El + (y-_mouse_start_y)/5.0;
    else 
      view_elevation = _mouse_start_view_El - (y-_mouse_start_y)/5.0;
    if (view_elevation > 90) {
      view_elevation = 90;
    } else if (view_elevation < 0) {
      view_elevation = 0;
    }
    if (view_azimuth>360) {
      view_azimuth-=360;
    } else if (view_azimuth<-360) {
      view_azimuth+=360;
    }
    break;
        
  }
  glutPostRedisplay();
}


void Glut_Window::draw_text_in_scene(char *txt, float r,float g, 
                                     float b,
                                     void *font) {
  glColor3f(r,g,b);
  glRasterPos3f(0,0,0);
  char *ch;
  for(ch= txt; *ch; ch++) {
    glutBitmapCharacter(font, (int)*ch);
  }
}

/**
   This will draw the given string to the screen at the location given by
   x,y- (0,0 is the bottom left of the window, 1,1 is the top right).
   the color defaults to a gray*/
void Glut_Window::draw_text(char *txt,float x,float y,
                            float r,float g, float b, 
                            void *font) {
  GLint matrixMode;
  GLboolean lightingOn; 
  char *ch;
  GLenum error;
  lightingOn= glIsEnabled(GL_LIGHTING);        /* lighting on? */
  if (lightingOn) glDisable(GL_LIGHTING);
    
  glGetIntegerv(GL_MATRIX_MODE, &matrixMode);  /* matrix mode? */
    
  glMatrixMode(GL_PROJECTION);

  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, 1.0, 0.0, 1.0);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glPushAttrib(GL_COLOR_BUFFER_BIT);       /* save current colour */
  glColor3f(r, g, b);
  glRasterPos3f(x, y, 0);
  for(ch= txt; *ch; ch++) {
//    glutBitmapCharacter(font, (int)*ch);
    glutStrokeCharacter(GLUT_STROKE_ROMAN,(int)*ch);
  }
  glPopAttrib();
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(matrixMode);
  if (lightingOn) glEnable(GL_LIGHTING);
  error = glGetError();
  if (error !=GL_NO_ERROR)
    cerr << gluErrorString(error) << endl;
}

/** 
    This will cause the timer_func function to be called with the given timerId after the given number of milliseconds.  Multiple timers can be set, providing they utilize different timer ids.  timer id should be an unsigned char*/
void Glut_Window::register_timer(unsigned int delay, unsigned char timer_ID) {
  unsigned int passVal;
  // we store the timer ID as the top 8 bits, the window id as the bottom
  passVal = (timer_ID << 8)+window_ID;
  glutTimerFunc(delay,&Glut_Window::_timer,passVal);
}


/*
  Callback hooks 
  These don't get overridden, they are static members that call the 
  appropriate virtual member functions
*/

void Glut_Window::_display(void) {
  // id is the window that the display callback is being requested for
  if (_initialized==false)
    return;
  int id = glutGetWindow();
  _windows[id]->display();
}

void Glut_Window::_reshape(int w,int h) {
  int id= glutGetWindow(); 
  _windows[id]->reshape(w,h);
}

void Glut_Window::_mouse(int button, int state, int x, int y) {
  int id= glutGetWindow(); 
  _windows[id]->mouse(button,state,x,y);
}

void Glut_Window::_key(unsigned char key,int x, int y) {
  int id= glutGetWindow(); 
  _windows[id]->key(key,x,y);
}

void Glut_Window::_specialKey(int key,int x, int y) {
  int id= glutGetWindow(); 
  _windows[id]->special_key(key,x,y);
}

void Glut_Window::_mouseMotion(int x, int y) {
  int id= glutGetWindow(); 
  _windows[id]->mouse_motion(x,y);
}

void Glut_Window::_timer(int value) {
  int temp_ID;
  unsigned int * val=(unsigned int*)( &value);
  int id = (*val) & 0xff;
  unsigned char passVal = (*val) >> 8;
  temp_ID = glutGetWindow();
  // set the window id so that what ever we do is passed to that window correctly
  glutSetWindow(id);
  _windows[id]->timer_func(passVal);
  // reset the focus so as to avoid confusion
  glutSetWindow(temp_ID);
}
/**
     This function draws all of the registered objects*/
void Glut_Window::_draw_registered(void) {
  list<GL_Object *>::iterator iter;
  for (iter = _reg_objects.begin();iter!=_reg_objects.end();iter++) {
    (*iter)->draw();
  }
}


/**
     This adds an object to the list of objects that will be drawn in each
     display pass.*/
void Glut_Window::register_object(GL_Object * obj) {
  _reg_objects.push_back(obj);
}


/**
   This removes every instance of the given object from the display list */
void Glut_Window::unregister_object(GL_Object *obj) {
  _reg_objects.remove(obj);

}




/*************************************************
 *                                               *
 ************************************************/


Glut_Lit_Window::Glut_Lit_Window(char *title, int width, int height): Glut_Window(title,width,height) {
  _def_specular[0]=1.0;
  _def_specular[1]=1.0;
  _def_specular[2]=1.0;
  _def_specular[3]=1.0;

  _def_shininess[0] = 50.0;

  _def_light_pos[0] = 0;
  _def_light_pos[1] = 0;
  _def_light_pos[2] = 1;
  _def_light_pos[3] = 0;
  
  _white_light[0] = 1.0;
  _white_light[1] = 1.0;
  _white_light[2] = 1.0;
  _white_light[3] = 1.0;
  
  GLfloat lmodel_ambient[] = {0.0,0.0,0.0,0.0};
  glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,_def_specular);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,_def_shininess);
  glLightfv(GL_LIGHT0,GL_POSITION,_def_light_pos);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,_white_light);
  glLightfv(GL_LIGHT0,GL_SPECULAR,_white_light);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT_AND_BACK,GL_DIFFUSE);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
};

void Glut_Lit_Window::position_camera(void) {
  glMaterialfv(GL_FRONT,GL_SPECULAR,_def_specular);
  glMaterialfv(GL_FRONT,GL_SHININESS,_def_shininess);
  glLightfv(GL_LIGHT0,GL_POSITION,_def_light_pos);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,_white_light);
  glLightfv(GL_LIGHT0,GL_SPECULAR,_white_light);
  Glut_Window::position_camera();
};

