#define _LARGEFILE_SOURCE 1
#define _LARGEFILE64_SOURCE 1
#define _FILE_OFFSET_BITS 64
#include <list>
#include <deque>
#include <math.h>
#include <utils/port.h>
#include <utils/Time.h>
#include <CannedVideo/CannedVideo.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <vector>
#include <ffmpeg/avcodec.h>
#include "sx12.h"
#include <utils/Input.h>

__UTILS_BEGIN_NAMESPACE

#define CANNED_VIDEO_MAJOR_VERSION (0)
#define CANNED_VIDEO_MINOR_VERSION (0)

/*
 * Bitflags for tracking which headers have been written
 */
#define _HW_VERSION         0x00000001
#define _HW_DESCRIPTION     0x00000002
#define _HW_HEADER          0x00000004

/* max time drift, in seconds, that triggers a clock reset in collecting
   sx12 data
   other mechanisms prevent this from happening except the first time, btw */
#define MAXIMUM_FRAME_DRIFT 0.5

/*
 * Translate from our enum to the avcodec defines
 */
static inline int _format_translate(CannedVideoFrameFormat format)
{
        switch (format) {
        case CVFF_YUV:
                return PIX_FMT_YUV420P;
        case CVFF_RGB24:
                return PIX_FMT_RGB24;
        case CVFF_BGR24:
                return PIX_FMT_BGR24;
        default:
                ;
        }
        throw CannedDataError("Unknown frame format type");
        return 0;
}

static void _init_avcodec() {
        static int avcodec_inited = 0;
        if (!avcodec_inited) {
                avcodec_init();
                avcodec_register_all();
        }
}


/*
 * Hack...trying to access normal open() and close() functionality is a nightmare of namespaces 
 * across compilers
 */

static inline int open_wrapper(const char *pathname, int flags)
{
        return open(pathname, flags);
}

static inline int close_wrapper(int fd)
{
        return close(fd);
}


typedef struct
{
        Time time_stamp;
        uint32_t size;
        uint64_t ofs;
        uint8_t  type;
} CannedVideoRecord;


/*
 * Far too much copying of data going on here...:P...This is just
 * a simple circular buffer implementation for doing sx12 specific stuff
 */
class CircBuf
{
public:
        CircBuf() {
                _data = (unsigned char *)malloc(10 * 1024);
                assert(_data);
                _buffer_size = 10 * 1024;
                _used_size = 0;
        }
        ~CircBuf() {
                free(_data);
        }

        unsigned char *getBuf() {
                return _data;
        }

        /*
         * Remove bytes from the beginning of the buffer
         */
        void consume(unsigned int bytes) {
                assert(bytes <= _used_size);
                memmove(_data, _data + bytes, _used_size - bytes);
                _used_size -= bytes;
        }

        /*
         * Add this to the end of the buffer
         */
        void add(unsigned char *buf, unsigned int bytes) {
                if (_used_size + bytes > _buffer_size) {
                        _data = (unsigned char *)realloc(_data, _used_size + bytes);
                        assert(_data);
                        _buffer_size = _used_size + bytes;
                }
                memcpy(_data + _used_size, buf, bytes);
                _used_size += bytes;
        }
        unsigned int size() { return _used_size; }
private:
        unsigned char *_data;
        unsigned int _buffer_size;
        unsigned int _used_size;
                 
};

/*
 * A class for handling reuse of standard size chunks of memory for
 * frame buffering.
 */
class FramesBufferObjectCache
{
public:
        FramesBufferObjectCache(int w, int h, int depth) 
                : _w(w), _h(h), _depth(depth) {
        }
        ~FramesBufferObjectCache() {
                while (!_bufs.empty()) {
                         fflush(stdout);
                        free(_bufs.front());
                        _bufs.pop_front();
                }
        }
        unsigned char *getBuf() {
                unsigned char *retval;
                if (!_bufs.empty()) {
                        retval = _bufs.front();
                        _bufs.pop_front();
                } else {
                        retval = (unsigned char *)malloc(_w * _h * _depth);
                        assert(retval);
                }
                return retval;
        }
        
        void freeBuf(unsigned char *buf) {
                if (buf) {
                        _bufs.push_front(buf);
                }
        }
private:
        int _w, _h, _depth;
        std::list<unsigned char *> _bufs;
};

class FramesBufferEntry
{
public:
        FramesBufferEntry(int w, int h, FramesBufferObjectCache *cache)  :
                _w(w), _h(h), _frame(0), _cache(cache) {
        }
        ~FramesBufferEntry() {
                _cache->freeBuf(_frame);
        }
        
        
        /*
         * If you're trying to use these, you're using the class in a way
         * it wasn't meant to be used.
         */
private:
        FramesBufferEntry(const FramesBufferEntry &);
        const FramesBufferEntry &operator =(const FramesBufferEntry &);

public:
        unsigned char *getWriteableFrame(CannedVideoFrameFormat new_format) {
                if (!_frame) {
                        _frame = _cache->getBuf();
                }
                _format = new_format;
                return _frame;
        }
        
        /*
         * Return a pointer to the frame data in RGB format
         */
        const unsigned char *getFrame(CannedVideoFrameFormat format) {
                if (_format != format) {
                        if ((_format == CVFF_YUV) && (format == CVFF_GREYSCALE)) {
                                /*
                                 * Special case: YUV->grayscale.  This isn't actuall a conversion, we're 
                                 * just discarding the UV channels.  Since the Y channel is first, there's
                                 * nothing to do.  However, since the U and V channels are still there,
                                 * we leave the format flag saying it's still YUV. 
                                 */
                                return _frame;
                        }

                        if (_format == CVFF_GREYSCALE) {
                                /*
                                 * Need to go to Yuv anyways, avcodec doesn't know about greyscale.  
                                 * Easy conversion -- just zero the U and v channels
                                 */
                                memset(_frame + (_w * _h), 0, (_w * _h)/2);
                                _format = CVFF_YUV;
                                
                                if (format == CVFF_YUV) {
                                        /*
                                         * All we needed to do.
                                         */
                                        return _frame;
                                }
                        }

                        /*
                         * general case, let avcodec take care of it.
                         */
                        unsigned char *tmp = _cache->getBuf();
                        int src_fmt = _format_translate(_format);
                        int dst_fmt = _format_translate(format);
                        AVPicture src, dest;
                        avpicture_fill(&src, _frame, src_fmt, _w, _h);
                        avpicture_fill(&dest, tmp, dst_fmt, _w, _h);
                        img_convert(&dest, dst_fmt, &src, src_fmt, _w, _h);
                        _cache->freeBuf(_frame);
                        _frame = tmp;
                        _format = format;
                }
                return _frame;
        }
        
        Time           timestamp;

private:

        int            _w, _h;
        unsigned char          *_frame;
        CannedVideoFrameFormat  _format;

        FramesBufferObjectCache *_cache;
};


/*
 * Cache of decoded frames.  FIFO.
 */

class CannedVideoFramesBuffer
{
public:
        CannedVideoFramesBuffer(unsigned int w, unsigned int h, unsigned int max_frames);
        ~CannedVideoFramesBuffer();

        /*
         * Look for a frame with this specific timestamp.  Return 0 on success.
         */
        FramesBufferEntry *find(Time timestamp);

        /*
         * Allocate a frame in the buffer with this stamp.  Set data to point at the
         * frame data for this buffer.  If stamp is a new keyframe, then we implicitly
         * flush the other entries from the buffer.
         */
        FramesBufferEntry *allocFrame();

        /*
         * Deallocate and invalidate all currently buffered entries.
         */
        void   flush();

private:
        std::deque<FramesBufferEntry *>   _frames;
        std::deque<FramesBufferEntry *>   _unused_frames;
        unsigned int                      _w;
        unsigned int                      _h;
        unsigned int                      _max_frames;
        FramesBufferObjectCache           _fbcache;
};

CannedVideoFramesBuffer::CannedVideoFramesBuffer(unsigned int w, unsigned int h, unsigned int max_frames)
        : _w(w),
          _h(h),
          _max_frames(max_frames),
          _fbcache(w, h, 3)
{
        assert(max_frames > 0);
}

CannedVideoFramesBuffer::~CannedVideoFramesBuffer()
{
        std::deque<FramesBufferEntry *>::iterator i;
        for (i = _frames.begin(); i != _frames.end(); i++) {
                delete (*i);
        }
        for (i = _unused_frames.begin(); i != _unused_frames.end(); i++) {
                delete (*i);
        }
}

FramesBufferEntry *CannedVideoFramesBuffer::find(Time timestamp)
{
        /*
         * this should be a small structure, just linear search it.
         */

        std::deque<FramesBufferEntry *>::iterator i;
        for (i = _frames.begin(); i != _frames.end(); i++) {
                if (timestamp == (*i)->timestamp) {
                        return (*i);
                }
        }
        /*
         * didn't find it.
         */
        return 0;
}

#if 0
int CannedVideoFramesBuffer::findBefore(unsigned char **frame, Time timestamp)
{
        (*frame) = 0;  /* Not necessary, but help to catch bugs */
        /*
         * this should be a small structure, just linear search it.
         */

        std::deque<FramesBufferEntry *>::iterator i;
        for (i = _frames.begin(); i != _frames.end(); i++) {
                if ((timestamp < (*i)->next_timestamp)
                    && (timestamp >= (*i)->timestamp)) {
                        (*frame) = (*i)->getFrame();
                        return 0;
                }
        }
        /*
         * didn't find it.
         */
        return 1;
}

int CannedVideoFramesBuffer::findNearest(unsigned char **frame, Time timestamp)
{
        Time tmp;
        (*frame) = 0;  /* Not necessary, but help to catch bugs */
        std::deque<FramesBufferEntry *>::iterator i;
        for (i = _frames.begin(); i != _frames.end(); i++) {
                tmp = timestamp.absDiff((*i)->timestamp);
                if ((tmp < (*i)->next_timestamp - timestamp)
                    && (tmp < timestamp - (*i)->prev_timestamp)) {
                        (*frame) = (*i)->getFrame();
                        return 0;
                }
        }
        return 1;
}
#endif

FramesBufferEntry *CannedVideoFramesBuffer::allocFrame()
{
        FramesBufferEntry *newentry;
        if (!_unused_frames.empty()) {
                newentry = _unused_frames.front();
                _unused_frames.pop_front();
        } else {
                if (_frames.size() == _max_frames) {
                        newentry = _frames.back();
                        _frames.pop_back();
                } else {
                        newentry = new FramesBufferEntry(_w, _h, &_fbcache);
                }
        }
        _frames.push_front(newentry);
        return newentry;
}

void CannedVideoFramesBuffer::flush()
{
        while (!_frames.empty()) {
                _unused_frames.push_front(_frames.front());
                _frames.pop_front();
        }
}

/*
 * Index of frames for reader to find stuff
 */
class CannedVideoFrameIndex : public std::vector<CannedVideoRecord> {
 public:
        CannedVideoFrameIndex() : eof(_eof), cur(_cur){}
        ~CannedVideoFrameIndex() {}
        int seekBefore(Time time);
        int seekNearest(Time time);
        void first();
        void next();
        void prev();
        void last();
        const int               &eof;
        const CannedVideoRecord &getCurRecord();
        const unsigned int      &cur;
private:
        void _updateCur(unsigned int newcur) {
                assert(newcur < size());
                _cur = newcur;
                _cur_record = (*this)[_cur];
                _cur_frame_valid = true;
        }

        CannedVideoRecord        _cur_record;
        unsigned int _cur;
        int          _cur_frame_valid;
        int _eof;
};

const CannedVideoRecord &CannedVideoFrameIndex::getCurRecord()
{
        if (!_cur_frame_valid) {
                throw CannedDataError("Attempt to get an invalid frame record!");
        }
        return _cur_record;
}

int CannedVideoFrameIndex::seekBefore(Time time)
{
        unsigned int top, bottom, split;
        unsigned int i;


	if (time < (*this)[0].time_stamp) {
		return -1;
	}


        /*
         * Common case - we're seeking close to where we are.  Do a quick
         * linear check to see if we can find it early.
         */
        if (time <= (*this)[_cur].time_stamp) {
                /*
                 * Search up to 10 records back, or to the start of file,
                 * whichever is closer.
                 */
                unsigned int min = _cur - ((_cur < 10)?_cur:10);

                if (time >= (*this)[min].time_stamp) {
                        /*
                         * Aha!  It's nearby.  Get it linearly.
                         */
                        
                        for (i = min+1; i <= _cur; i++) {
                                if (time < (*this)[i].time_stamp) {
                                        break;
                                }
                        }
                        _updateCur(i-1);
                        return 0;
                }

        } else {
                /*
                 * Search up to 10 records, or the end of file, whichever is
                 * closer
                 */
                unsigned int max = (_cur < (size() - 10))?(_cur + 10):(size()-1);
                
                if (time <= (*this)[max].time_stamp) {
                        /*
                         * Aha!  It's nearby.  Get it linearly.
                         */
                        for (i = _cur+1; i <= max; i++) {
                                if (time < (*this)[i].time_stamp) {
                                        break;
                                }
                        }
                        _updateCur(i-1);
                        return 0;
                }
        }
        
	top = size();
	bottom = 0;
	split = bottom + (top - bottom)/2;
	while (split != bottom) {
		if (time < (*this)[split].time_stamp) {
			top = split;
		} else {
			bottom = split;
		}
		split = bottom + (top - bottom)/2;
	}
	_updateCur(split);
	return 0;
}

int CannedVideoFrameIndex::seekNearest(Time time)
{
	if (seekBefore(time)) {
                if (!size()) {
                        return -1;
                }
                _updateCur(0);
	} else {
		if (!((_cur == (size() - 1))
		      || (((*this)[_cur].time_stamp.absDiff(time)) < ((*this)[_cur + 1].time_stamp.absDiff(time))))) {
			_cur++;
		}
	}
        return 0;
}

void CannedVideoFrameIndex::first()
{
        _eof = 0;
        _updateCur(0);
}

void CannedVideoFrameIndex::last()
{
        _updateCur(size() - 1);
        _eof = 0;
}

void CannedVideoFrameIndex::next()
{
        if (_cur == (size() - 1)) {
                _eof = 1;
                _cur_frame_valid = false;
        } else {
                _updateCur(_cur+1);
        }
}

void CannedVideoFrameIndex::prev()
{
        if (size()) {
                if (_cur) {
                        _updateCur(_cur-1);
                } 
                _cur_frame_valid = true;
                _eof = 0;
        }
}
void CannedVideoWriter::_init()
{
        _init_avcodec();
        _idx_ptr = 0;
        _idx_slots = 0;
        _frames_per_index_block = 1024;  /* Arbitrary.  Possibly should be user customizable */
        _headers_written = 0;
        _state = CV_STATE_CLOSED;
        _file = 0;
        _w = 0;
        _h = 0;
        _depth = 0;
        _frame_block_size = 0;
        _saved_ptr = -1;
        _codec = NULL;
        _context = NULL;
        _outbuf = NULL;
        _outbuf_size = 0;
        _encode = false;
        _convert_buf = NULL;
        _sx12_fd = -1;
}

CannedVideoWriter::CannedVideoWriter()
{
        _init();
}

CannedVideoWriter::~CannedVideoWriter()
{
}

void CannedVideoWriter::open(unsigned int width, unsigned int height, CannedVideoType codec, 
                             const char *filename, bool encode, 
                             unsigned int frames_per_keyframe, mode_t mode, 
                             int bit_rate, int frame_rate,
                             CannedVideoFrameFormat format) 

{
        _w = width;
        _h = height;
        struct {
                uint32_t a;
                uint32_t b;
        } tmp;
        _depth = 3;
	char buf[4] = { 'J', 'D', 'C', 'V' };  /* For JDC Video.  Why not? */
	struct timeval tv;
	uint32_t cv_major_version = host_to_le_uint32(CANNED_VIDEO_MAJOR_VERSION);
	uint32_t cv_minor_version = host_to_le_uint32(CANNED_VIDEO_MINOR_VERSION);
	uint64_t tmp64;
        uint32_t tmp32;
	if (_state != CV_STATE_CLOSED) {
		_cleanup();
		throw CannedDataError("Open called when already open");
	}
	_checkClobber(filename, mode);
	_file = fopen64(filename, "w");
	if (!_file) {
		_cleanup();
		throw CannedDataError("Couldn't open %s for output: %s", 
				      filename, strerror(errno));
        }
        if (gettimeofday(&tv, 0)) {
		_cleanup();
		throw CannedDataError("gettimeofday failed: %s", strerror(errno));
	}
        _doFwrite(_file, buf, 4);
        _doFwrite(_file, &cv_major_version, 4);
        _doFwrite(_file, &cv_minor_version, 4);
        tmp64 = tv.tv_sec;
        tmp64 = host_to_le_uint64(tmp64);
        _writeRecord(_file, CVID_CREATION_TIME, &tmp64, 8);
        tmp32 = host_to_le_uint32(codec);
        _writeRecord(_file, CVID_TYPE, &tmp32, 4);
        tmp.a = host_to_le_uint32(_w);
        tmp.b = host_to_le_uint32(_h);
        assert(sizeof(tmp) == 8);
        _writeRecord(_file, CVID_SIZE, &tmp, 8);
        tmp32 = host_to_le_uint32(frames_per_keyframe);
        _writeRecord(_file, CVID_KEYFRAME_SPACING, &tmp32, 4);
        _state = CV_STATE_OPEN;
        
        _encode = encode;
        if (encode) {
                switch (codec) {
                case CVT_MJPEG:
                        _codec = avcodec_find_encoder(CODEC_ID_MJPEG);
                        break;
                case CVT_MPEG1:
                        _codec = avcodec_find_encoder(CODEC_ID_MPEG1VIDEO);
                        break;
                case CVT_MPEG4:
                        _codec = avcodec_find_encoder(CODEC_ID_MPEG4);
                        break;
                default:
                        throw CannedDataError("Unknown encode codec requested");
                }
                if (_codec == NULL) {
                        throw CannedDataError("Requested encoding codec not found in libavcodec");
                }
                _context = avcodec_alloc_context();
                _picture = avcodec_alloc_frame();
                _context->bit_rate = bit_rate;
                _context->width = _w;
                _context->height = _h;
                _context->frame_rate = frame_rate * FRAME_RATE_BASE;
                _context->gop_size = frames_per_keyframe;

                if (avcodec_open(_context, _codec) < 0) {
                        throw CannedDataError("Couldn't open codec for encoding");
                }
                
                /*
                 * Not sure about the sizing of this buffer, so just make it big enough
                 * to hold the uncompressed YUV data.  
                 */
                _outbuf_size = _w * _h * 2;
                _outbuf = new unsigned char[_outbuf_size];
                
                _incoming_format = _format_translate(format);

                if (_incoming_format != PIX_FMT_YUV420P) {
                        /*
                         * Need a buffer to do colorspace conversion into YUV space
                         */
                        _convert_buf = new unsigned char[(_w * _h * 2) + 2];
                } else {
                        _convert_buf = NULL;
                }
        }
}

/* 
 * Write all metadata to disk, and finish out the file, closing it.
 * No further write* calls will be valid after close() is invoked
 */
void CannedVideoWriter::close()
{
	_writeRecord(_file, CVID_END, 0, 0);
	_cleanup();
}

/* 
 * Set the version information for this file.  
 */

void CannedVideoWriter::writeVersion(uint32_t major, uint32_t minor)
{
        uint64_t tmp;
	if (_headers_written & _HW_VERSION) {
		_cleanup();
		throw CannedDataError("Tried to write version field twice");
	}
	_headers_written |= _HW_VERSION;
	tmp = ((uint64_t)host_to_le_uint32(major))<<32;
	tmp |= host_to_le_uint32(minor);
	_writeRecord(_file, CVID_VERSION, &tmp, 8);
}

/*
 * Add a description of this data.  The string should be null
 * terminated.
 */
void CannedVideoWriter::writeDescription(const char *desc)
{
	if (_headers_written & _HW_DESCRIPTION) {
		_cleanup();
		throw CannedDataError("Tried to write description field twice");
	}
	_headers_written |= _HW_DESCRIPTION;
	_writeRecord(_file, CVID_DESCRIPTION, desc, strlen(desc));
}

/*
 * Write some custom header information for this file.
 */
void CannedVideoWriter::writeHeader(void *buf, size_t size)
{
	if (size > 0xffff) {
		throw CannedDataError("Header too large!");
	}
	if (_headers_written & _HW_HEADER) {
		_cleanup();
		throw CannedDataError("Tried to write header field twice");
	}
	_headers_written |= _HW_HEADER;
	_writeRecord(_file, CVID_HEADER, buf, size);
}


/*
 * Create a block of frame indexes at the current end of the file, 
 * and update our internal state accordingly
 */
void CannedVideoWriter::_createIndexBlock(FILE *file)
{
        uint16_t tmp16;
        uint64_t tmp64;
        tmp16 = host_to_le_uint16((uint16_t)CVID_IDX);
        _doFwrite(file, &tmp16, 2);
        tmp64 = host_to_le_uint64(_frames_per_index_block * 25 + 2);
        _doFwrite(file, &tmp64, 8);

        _idx_slots_ptr = ftello64(file);
        tmp16 = 0;
        _doFwrite(file, &tmp16, 2);

        _idx_ptr = ftello64(file);
        _idx_slots = _frames_per_index_block;
        _idx_slots_used = 0;

        fseeko64(file, _frames_per_index_block * 25, SEEK_CUR);

        /*
         * If we create a new index block, this implies that we need a new data area at the
         * current end of file.
         */
        _writeRecord(file, CVID_FRAMEBLOCK, 0, 0);
        _frame_block_size = 0;        

}
void CannedVideoWriter::_writeIndex(FILE *file, Time time, CannedVideoFrameType type, uint32_t size)
{


        long secs;
        long usecs;
        time.getValue(secs, usecs);
        if ((_state != CV_STATE_WRITING)
            || _idx_slots == 0) {
                _createIndexBlock(file);
                _state = CV_STATE_WRITING;
        }
        uint8_t tmp8;
        uint16_t tmp16;
        uint64_t tmp64;

        tmp64 = host_to_le_uint64(ftello64(file));
        
        _pushFilePos(file);
        fseeko64(file, _idx_ptr, SEEK_SET);
        tmp8 = type;
        secs = host_to_le_uint64(secs);
        _doFwrite(file, &secs, 8);
        usecs = host_to_le_uint32(usecs);
        _doFwrite(file, &usecs, 4);
        _doFwrite(file, &tmp64, 8);
        size = host_to_le_uint32(size);
        _doFwrite(file, &size, 4);
        tmp8 = type;
        _doFwrite(file, &type, 1);
        _idx_ptr += 25;
        _idx_slots--;
        _idx_slots_used++;
        fseeko64(file, _idx_slots_ptr, SEEK_SET);
        tmp16 = host_to_le_uint16(_idx_slots_used);
        _doFwrite(file, &tmp16, 2);
        _popFilePos(file);
}

void CannedVideoWriter::_pushFilePos(FILE *file)
{
        assert(_saved_ptr == -1);
        _saved_ptr = ftello64(file);
}

void CannedVideoWriter::_popFilePos(FILE *file)
{
        assert(_saved_ptr != -1);
        fseeko64(file, _saved_ptr, SEEK_SET);
        _saved_ptr = -1;
}       

void CannedVideoWriter::_updateFrameBlockSize(FILE *file, size_t size)
{
        uint64_t tmp64;
        _frame_block_size += size;
        tmp64 = host_to_le_uint64(_frame_block_size);
        _pushFilePos(file);
        fseeko64(file, _frame_block_size_ptr, SEEK_SET);
        _doFwrite(file, &tmp64, 8);
        _popFilePos(file);
}


void CannedVideoWriter::_writeFrame(FILE *file, const Time &time, unsigned char *buf, uint32_t size, CannedVideoFrameType type)
{
        /*
         * If we've still got room in the index block, write the index there
         */
        if (_state == CV_STATE_CLOSED) {
                _cleanup();
                throw CannedDataError("Attempt to write a frame while closed");
        }


        if (!_encode) {
                /*
                 * Data comes pre-encoded, just shove it to file
                 */
                _writeIndex(file, time, type, size);
                _updateFrameBlockSize(file, size);
                _doFwrite(file, buf, size);
        } else {
                /*
                 * need to encode this frame
                 */
                int blocksize;
                if (_incoming_format != PIX_FMT_YUV420P) {
                        /*
                         * Convert colorspace
                         */
                        AVPicture src;
                        avpicture_fill(&src, buf, _incoming_format, _w, _h);
                        avpicture_fill((AVPicture *)_picture, _convert_buf, PIX_FMT_YUV420P, _w, _h);
                        img_convert((AVPicture *)_picture, _incoming_format, &src, PIX_FMT_YUV420P, _w, _h);
                } else {
                        avpicture_fill((AVPicture *)_picture, buf, PIX_FMT_YUV420P, _w, _h);
                }
                blocksize = avcodec_encode_video(_context, _outbuf, _outbuf_size, _picture);
                _writeIndex(file, time, (_context->coded_frame->key_frame)?(CVFT_I):(CVFT_P), blocksize);
                _updateFrameBlockSize(file, blocksize);
                _doFwrite(file, _outbuf, blocksize);
        }
}

/*
 * Write out a data record to the file with the given time stamp 
 */
void CannedVideoWriter::writeFrame(const Time &time, unsigned char *buf, uint32_t size, CannedVideoFrameType type)
{
        _writeFrame(_file, time, buf, size, type);
}

void CannedVideoWriter::_cleanup()
{
        if (_file) {
                fclose(_file);
        }
        if (_encode) {
                delete [] _outbuf;
                free(_picture);
                avcodec_close(_context);
                free(_context);
        }
        if (_convert_buf) {
                delete [] _convert_buf;
        }
        _init();
}

void CannedVideoWriter::_checkClobber(const char *filename, mode_t mode)
{
 	struct stat buf;
	if ((mode & O_EXCL)
	    && !stat(filename, &buf)) {
		_cleanup();
		throw CannedDataError("Can't clobber file %s", filename);
	}        
}

void CannedVideoWriter::_doFwrite(FILE *file, const void *buf, size_t size)
{
        if (size) {
                if (!fwrite(buf, size, 1, file)) {
                        _cleanup();
                        throw CannedDataError("fwrite failed: %s", strerror(errno));
                }
        }

}

void CannedVideoWriter::_writeRecord(FILE *file, CannedVideoFieldID id, const void *buf, uint64_t size)
{
	uint16_t tmp;
	uint64_t tmp_size;
	tmp = host_to_le_uint16((uint16_t)id);
	tmp_size = host_to_le_uint64(size);
	_doFwrite(file, &tmp, 2);
        if (id == CVID_FRAMEBLOCK) {
                _frame_block_size_ptr = ftello64(file);
        }
        _doFwrite(file, &tmp_size, 8);
        if (size) {
                _doFwrite(file, buf, size);
        }
}


/**********************************
 *
 *  CannedVideoReader
 *
 **********************************/
void CannedVideoReader::_init()
{
        _init_avcodec();
        _w = _h = 0;
        _file = 0;
        _state = CV_STATE_CLOSED;
        _index->clear();
        _desc = "";
        _creation_time = 0;
        _major_version = 0;
        _minor_version = 0;
        _header_size = 0;
        _header_offset = 0;
        _type = CVT_UNUSED;

        _avcodec = NULL;
        _avcontext = NULL;
        _buf = NULL;
        _frames_buffer = NULL;
        _last_decoded = 0xffffffff;

}

void CannedVideoReader::_alloc()
{
        _index = new CannedVideoFrameIndex;
        _frame = avcodec_alloc_frame();
}

CannedVideoReader::CannedVideoReader()
{
        _alloc();
        _init();
}

CannedVideoReader::CannedVideoReader(const char *filename) 
{
        _alloc();
        _init();
        open(filename);
}

CannedVideoReader::~CannedVideoReader()
{
        delete _index;
        free(_frame);
}

void CannedVideoReader::close()
{
        fclose(_file);
        
        delete [] _buf;
        avcodec_close(_avcontext);
        free(_avcontext);
        delete _frames_buffer;
        _init();
}

void CannedVideoReader::open(const char *filename)
{
	char buf[5];
	char *desc;
	uint16_t id;
        uint64_t length;
	uint32_t tmp;
	uint32_t cv_major_version, cv_minor_version;
	int partial_read = 0;

        if (_state != CV_STATE_CLOSED) {
                close();
        }

        utils::Input in;
        if (!in.openFile(filename)) {
                throw CannedDataError("Couldn't open %s for reading: %s", 
                     filename, strerror(errno));
        }
        _file = in.getCurFile();
        in.releaseFile();

	if (!((fread(buf, 1, 4, _file) == 4) 
	      && fread(&cv_major_version, 4, 1, _file)
	      && fread(&cv_minor_version, 4, 1, _file))) {
		throw CannedDataError("Unexpected end of file %s", filename);
	}
	cv_major_version = le_to_host_uint32(cv_major_version);
	cv_minor_version = le_to_host_uint32(cv_minor_version);
	buf[4] = 0;
	if (strcmp(buf, "JDCV")) {
		throw CannedDataError("No valid signature found in file %s; this doesn't appear to be a canned video file", 
				      filename);
	}
	
	/*
	 * We'll have to change this later if we want to do compatibility
	 * work
	 */
	if (!((cv_major_version == CANNED_VIDEO_MAJOR_VERSION)
	      && (cv_minor_version == CANNED_VIDEO_MINOR_VERSION))) {
		throw CannedDataError("Version mismatch.  File %s was created "
				      "with CannedData library version %x.%x."
				      "  Library in use is version %x.%x.",
				      filename,
				      cv_major_version,
				      cv_minor_version,
				      CANNED_VIDEO_MAJOR_VERSION,
				      CANNED_VIDEO_MINOR_VERSION);
	}
	    

	while (!(feof(_file) || partial_read)) {
                CannedVideoRecord rec;
		if (!(fread(&id, 2, 1, _file)
		      && fread(&length, 8, 1, _file))) {
			partial_read = 1;
			break;
		}
		id = le_to_host_uint16(id);
		length = le_to_host_uint64(length);
		
		switch(id) {
		case CVID_INVALID:
			throw CannedDataError("Encountered invalid field in file %s", filename);
                case CVID_KEYFRAME_SPACING:
                        if (length != 4) {
				throw CannedDataError("Keyframe spacing field appears to be screwed in file %s (%i)", filename, length);
                        }
                        if (!fread(&tmp, 4, 1, _file)) {
                                partial_read = 1;
                                break;
                        }
                        _frames_per_keyframe = le_to_host_uint32(tmp);
                        break;
                case CVID_SIZE:
                        if (_w) {
				throw CannedDataError("Multiple size fields encountered in file %s", filename);
                        }
                        if (length != 8) {
				throw CannedDataError("Size field appears to be screwed in file %s (%i)", filename, length);
                        }
                        if (!(fread(&_w, 4, 1, _file)
                              && fread(&_h, 4, 1, _file))) {
				throw CannedDataError("Incomplete size field!");
                        }
                        if (!(_w && _h)) {
                                throw CannedDataError("Size of frames is 0 in file %s", filename);
                        }
                        _w = le_to_host_uint32(_w);
                        _h = le_to_host_uint32(_h);
                        break;
		case CVID_TYPE:
                        if (_type != CVT_UNUSED) {
                                throw CannedDataError("Multiple Type fields encountered in file %s", filename);
                        }
			if (length != 4) {
				throw CannedDataError("Type field appears to be screwed in file %s (%i)", filename, length);
			}
			if (!fread(&tmp, 4, 1, _file)) {
				partial_read = 1;
				break;
			}
			_type = (CannedVideoType)le_to_host_uint32(tmp);
			break;
		case CVID_VERSION:
			if (length != 8) {
				throw CannedDataError("Version field appears to be screwed in file %s", filename);
			}
                        {
                                uint64_t tmp;
                                if (!fread(&tmp, 8, 1, _file)) {
                                        throw CannedDataError("Version field appears to "
                                                              "be screwed in file %s",
                                                              filename);
                                }
                                _major_version = le_to_host_uint32(tmp>>32);
                                _minor_version = le_to_host_uint32(tmp & 0xffffffff);
                        }
			break;
		case CVID_CREATION_TIME:
			if (length != 8) {
				throw CannedDataError("Creation field appears to "
						      "be screwed in file %s", 
						      filename);
			}
			if (!fread(&_creation_time, 8, 1, _file)) {
				partial_read = 1;
			}
			_creation_time = le_to_host_uint64(_creation_time);
			break;
		case CVID_DESCRIPTION:
			desc = (char *)calloc(1, length + 1);
			if (!desc) {
				throw CannedDataError("Out of memory");
			}
			if (fread(desc, 1, length, _file) != length) {
				partial_read = 1;
			}
			_desc = desc;
			free(desc);
			break;
		case CVID_HEADER:
			_header_size = length;
			_header_offset = ftello64(_file);
			fseeko64(_file, length, SEEK_CUR);
			break;
		case CVID_IDX:
                        /*
                         * Blocks of index records.  
                         */
                        uint16_t frames_in_block;
                        unsigned int i;
                        if (!fread(&frames_in_block, 2, 1, _file)) {
                                partial_read = 1;
                        } else {
                                uint64_t tmp64;
                                uint32_t tmp32;
                                length -= 2;
                                frames_in_block = le_to_host_uint16(frames_in_block);
                                for (i = 0; (i < frames_in_block) && !partial_read; i++) {
                                        if (!(fread(&tmp64, 8, 1, _file) 
                                              && fread(&tmp32, 4, 1, _file)
                                              && fread(&rec.ofs, 8, 1, _file)
                                              && fread(&rec.size, 4, 1, _file)
                                              && fread(&rec.type, 1, 1, _file))
                                            || (rec.type == CVFT_UNUSED)) {
                                                partial_read = 1;
                                        } else {
                                                rec.time_stamp.setValue(le_to_host_uint64(tmp64), le_to_host_uint32(tmp32));
                                                rec.ofs = le_to_host_uint64(rec.ofs);
                                                rec.size = le_to_host_uint32(rec.size);
                                                _index->push_back(rec);
                                        }
                                        length -= 25;
                                }
                                /*
                                 * Skip over any remaining unused frame slots
                                 */
                                fseeko64(_file, length, SEEK_CUR);
                        }
			break;
                case CVID_FRAMEBLOCK:
                        /* 
                         * Block of frames.  Just skip it.
                         */
                        fseeko64(_file, length, SEEK_CUR);
                        break;
		case CVID_END:
			/*
			 * It's not really a partial read, but this is a simple way to get
			 * out of this loop...
			 */
			partial_read = 1;
			break;
		default:
		  throw CannedDataError("Unknown field ID: %i", (int)id);
		  break;
		}
	}
	if (_index->size() == 0) {
		throw CannedDataError("No records found in file %s", filename);
	}
        if ((*_index)[0].type != CVFT_I) {
                throw CannedDataError("First frame in file is not an index frame!");
        }
        
        if (_type == CVT_UNUSED) {
                throw CannedDataError("No type field encountered in file %s", filename);
        }

        switch(_type) {
        case CVT_MPEG1:
        case CVT_MPEG2:
                _avcodec = avcodec_find_decoder(CODEC_ID_MPEG1VIDEO);
                break;
        case CVT_MPEG4:
                _avcodec = avcodec_find_decoder(CODEC_ID_MPEG4);
                break;
        case CVT_MJPEG:
                _avcodec = avcodec_find_decoder(CODEC_ID_MJPEG);
                break;
        default:
                throw CannedDataError("Unknown encoding type encountered in file %s", filename);
        }
        if (!_avcodec) {
                throw CannedDataError("Couldn't find the right codec in libavcodec for file %s", filename);
        }
        _avcontext = avcodec_alloc_context();
        _avcontext->width = _w;
        _avcontext->height = _h;
        if (avcodec_open(_avcontext, _avcodec) != 0) {
                throw CannedDataError("Couldn't open avcodec in file %s", filename);
        }

        /*
         * Allocate a decoding buffer which is as large as the largest frame;
         */
        std::vector<CannedVideoRecord>::iterator i;
        size_t bufsize = 0;
        for (i = _index->begin(); i != _index->end(); i++) {
                if ((*i).size > bufsize) {
                        bufsize = ((*i).size);
                }
        }
        _buf = new unsigned char[bufsize];
        
        /*
         * Allocate the frames buffer
         */
        _frames_buffer = new CannedVideoFramesBuffer(_w, _h, _frames_per_keyframe + 5);

}


unsigned int CannedVideoReader::width()
{
        return _w;
}

unsigned int CannedVideoReader::height()
{
        return _h;
}

uint64_t CannedVideoReader::getCreationTime()
{
        return _creation_time;
}

void CannedVideoReader::getVersion(uint32_t *major, uint32_t *minor)
{
        (*major) = _major_version;
        (*minor) = _minor_version;
}

FILE *CannedVideoReader::getHeader()
{
        fseeko64(_file, _header_offset, SEEK_SET);
        return _file;
}

size_t CannedVideoReader::getHeaderSize()
{
        return _header_size;
}

void CannedVideoReader::seek(const Time &time)
{
        assert(0);
}

int CannedVideoReader::seekBefore(const Time &time)
{
  int retval = _index->seekBefore(time);
  //  printf("Request to seek before %10.10f, cur is now %i (%10.10f, %10.10f, %10.10f)\n", (double)time, _index->cur,
  //	 _index->cur?((double)(*_index)[_index->cur-1].time_stamp):(double)0,
  //	 (double)(*_index)[_index->cur].time_stamp,
  //	 (double)(*_index)[_index->cur+1].time_stamp);
  return retval;
}

int CannedVideoReader::eof()
{
        return _index->eof;
}

void CannedVideoReader::first()
{
        _index->first();
}
void CannedVideoReader::next()
{
        _index->next();
}
void CannedVideoReader::prev()
{
        _index->prev();
}

void CannedVideoReader::last()
{
        _index->last();
}

Time CannedVideoReader::getCurTimeStamp()
{
        return _index->getCurRecord().time_stamp;
}

Time CannedVideoReader::getFirstTimeStamp()
{
        return _index->front().time_stamp;
}

Time CannedVideoReader::getLastTimeStamp()
{
        return _index->back().time_stamp;
}

int CannedVideoReader::getPrevTimeStamp(Time *time)
{
        if (_index->cur == 0) {
                (*time) = _index->front().time_stamp;
                return -1;
        }
        (*time) = (*_index)[_index->cur-1].time_stamp;
        return 0;
}

int CannedVideoReader::getNextTimeStamp(Time *time)
{
        if (_index->cur == (_index->size() - 1)) {
                (*time) =  _index->back().time_stamp;
                return -1;
        }
        (*time) = (*_index)[_index->cur+1].time_stamp;
        return 0;
}

unsigned int CannedVideoReader::getNumRecords()
{
        if (_index == NULL) {
                throw CannedDataError("getNumRecords() called with no valid file opened");
        }
        return _index->size();
}


void CannedVideoWriter::open(CannedVideoReader &reader, const char *filename, const Time &startTime, const Time &endTime, bool raw, mode_t mode)
{
	uint16_t id;
        uint64_t length;
        unsigned char *buf = NULL;
        size_t bufsize = 0;

	if (_state != CV_STATE_CLOSED) {
                close();
	}
        
        _state = CV_STATE_OPEN;

        _encode = false;

        _checkClobber(filename, mode);
        FILE *out = fopen64(filename, "w");
 	if (!out) {
		throw CannedDataError("Couldn't open %s for output: %s", 
				      filename, strerror(errno));
        }
        if (!raw) {
                /*
                 * Skip the JDCV header and versioning information
                 */
                fseeko64(reader._file, 12, SEEK_SET);
                if (!raw) {
                        char buf[4] = {'J', 'D', 'C', 'V' };
                        uint32_t cv_major_version = host_to_le_uint32(CANNED_VIDEO_MAJOR_VERSION);
                        uint32_t cv_minor_version = host_to_le_uint32(CANNED_VIDEO_MINOR_VERSION);
                        _doFwrite(out, buf, 4);
                        _doFwrite(out, &cv_major_version, 4);
                        _doFwrite(out, &cv_minor_version, 4);
                }
                
                /*
                 * XXX - we don't do any validation checking here, but we probably should.  The
                 * reader should have already validated the file, but it could change from
                 * underneath us.  If that happens, we'll "reward" the user (who is probably doing
                 * something silly) with silly behaviour.
                 */
                while (true) {
                        fread(&id, 2, 1, reader._file);
                        fread(&length, 8, 1, reader._file);
                        id = le_to_host_uint16(id);
                        length = le_to_host_uint64(length);
                        if (id == CVID_IDX) {
                                break;
                        }
                        if (bufsize < length) {
                                buf = (unsigned char *)realloc(buf, length);
                                bufsize = length;
                        }
                        fread(buf, 1, length, reader._file);
                        _writeRecord(out, (CannedVideoFieldID)id, buf, length);
                }
        }
        
        /*
         * Now we're into the frame blocks.  Seek backwards from the desired time to the previous
         * key frame:
         */
        CannedVideoRecord tmp;
        reader._index->seekBefore(startTime);
        while(true) {
                tmp = reader._index->getCurRecord();
                if (tmp.type == CVFT_I) {
//                        printf("Found starting keyframe at %10.10f\n", (double)tmp.time_stamp);
                        break;
                }
                reader._index->prev();
                
        } 

        /*
         * We're at a keyframe.  Just dump successive frames until we hit a timestamp which is beyond endTime
         */
        while (tmp.time_stamp <= endTime) {
                if (bufsize < tmp.size) {
                        buf = (unsigned char *)realloc(buf, tmp.size);
                        assert(buf);
                        bufsize = tmp.size;
                }
                fseeko64(reader._file, tmp.ofs, SEEK_SET);
                fread(buf, 1, tmp.size, reader._file);
                if (!raw) {
//                        printf("Wrote %c frame (%10li bytes) from time %10.10f\n", (tmp.type == CVFT_I)?'I':'P', (long)tmp.size, (double)tmp.time_stamp);
                        _writeFrame(out, tmp.time_stamp, buf, tmp.size, (CannedVideoFrameType)tmp.type);
                } else {
                        _doFwrite(out, buf, tmp.size);
                }
                reader._index->next();
                if (reader.eof()) {
                        break;
                }
                tmp = reader._index->getCurRecord();
        }

        free(buf);

        if (raw) {
                /*
                 * If it's an mpeg stream, stick an end of stream on it.  Most players
                 * don't really care if this is here, but be nice anyways.
                 */
                unsigned char eos[4] = {0, 0, 1, 0xb7};
                _doFwrite(out, eos, 4);
        } else {
                _writeRecord(out, CVID_END, 0, 0);
        }
        fclose(out);
        _state = CV_STATE_CLOSED;
}



const std::string &CannedVideoReader::getDescription()
{
        return _desc;
}

const unsigned char *CannedVideoReader::getFrame(CannedVideoFrameFormat format)
{
        
        FramesBufferEntry *fbe;
        FramesBufferEntry *right_fbe = 0;
        Time stamp = (*_index)[_index->cur].time_stamp;
        /*
         * Check the frame buffer to see if we've already got it
         */

        if ((fbe = _frames_buffer->find(stamp)) != NULL) {
	        const unsigned char *tmp;
                tmp = fbe->getFrame(format);
		//		printf("Cache hit!  Returning %p for time %10.10f (%i?)\n", tmp, (double)stamp, _index->cur);
		return tmp;
        }

        /*
         *  Check to see if we need to restart decode at a 
         *  more recent keyframe
         */
        unsigned int decode_start;
        unsigned int dont_cache = 0;
        bool         decode_restart;
        unsigned int needed_keyframe;
        unsigned int i, j, k;
        for (needed_keyframe = _index->cur; (*_index)[needed_keyframe].type != CVFT_I; needed_keyframe--) {}
        
        if ((_index->cur > _last_decoded) &&
            (needed_keyframe <= _last_decoded)) {
          // printf("smooth sailing %d %d %d!\n", _index->cur, _last_decoded, needed_keyframe);
                decode_start = _last_decoded + 1;
                decode_restart = 0;
        } else {
	        decode_start = needed_keyframe;
                decode_restart = 1;
        }
        /*
         * Decode the relevant frame(s) and buffer the results.
         */

        if (decode_restart) {
	  //                avcodec_flush_buffers(_avcontext);
        }
        
        
        /*
         * Number of frames we've decoded
         */
        int frames_got = 0;

        // printf("Cur index %d\n", _index->cur);
        for (i = decode_start; decode_start + frames_got <= _index->cur; i++) {
          // printf("\tDecode %d\n", i);
                int got_picture;
                int len;
                unsigned char *b = _buf;
                int size = (*_index)[i].size;
                

                fseeko64(_file, (*_index)[i].ofs, SEEK_SET);
                fread(_buf, (*_index)[i].size, 1, _file);
                /*
                 * Shove it at the decoder
                 */
                do {
                        len = avcodec_decode_video(_avcontext, _frame, &got_picture, b, size);
                        size -= len;
                        b += len;
                } while (size || !got_picture);

//                while (got_picture) {
                
                if (dont_cache) {        
                        dont_cache--;
                } else {
                        fbe = _frames_buffer->allocFrame();
//                        printf("Got a frame!\n");
                        fbe->timestamp = (*_index)[decode_start + frames_got].time_stamp;
                        
                        /*
                         * If this is the one we're looking for, save a pointer to it.
                         */
                        if (decode_start + frames_got == _index->cur) {
                                right_fbe = fbe;
                        }
                
                        _last_decoded = decode_start + frames_got;
                
                
                        /*
                         * Copy the data out to our buffer
                         */
//                        printf("Got a frame!\n");
                        unsigned char *dst = fbe->getWriteableFrame(CVFF_YUV);
                        unsigned char *src = _frame->data[0];
                        for (j = 0; j < _h; j++) {
                                memcpy(dst, src, _w);
                                dst += _w;
                                src += _frame->linesize[0];
                        }
                        for (k = 1; k <= 2; k++) {
                                src = _frame->data[k];
                                for (j = 0; j < _h/2; j++) {
                                        memcpy(dst, src, _w/2);
                                        dst += _w/2;
                                        src += _frame->linesize[k];
                                }
                        }
                }

                frames_got++;
                /*
                 * If we're doing bidirectional encoding, we may have more frames waiting now, 
                 * even beyond the one we're looking for.  To be computationaly efficient, pull
                 * those frames out now and buffer them.
                 */
                //                      avcodec_decode_video(_avcontext, _frame, &got_picture, NULL, 0);
//                        assert(!got_picture);
//                }
        }
        
        return right_fbe->getFrame(format);
}




/****************************************************************
 * 
 * sx12 specific functionality
 *
 ****************************************************************/

typedef enum {
        STATE_START,    /* Haven't found any 0's      */
        STATE_FOUND1,   /* Found 1 0's                */
        STATE_FOUND2,   /* Found 2 0's                */
        STATE_GET_CODE  /* Found 0x001, look for code */
} ScanState;

typedef enum {
        HEADER_SEQUENCE,
        HEADER_GOP,
        HEADER_PICTURE,
        HEADER_NONE,
} HeaderType;


/*
 * Scan the bytestream for 0x001<code> sequences.  Return 
 * the ones that interest us.
 */
static HeaderType _find_header(const unsigned char *buf, int size, uint64_t *ofs)
{
        int i;
        ScanState state = STATE_START;
        for (i = 0; i < size; i++) {
                (*ofs)++;
                if (state == STATE_GET_CODE) {
                        switch(buf[0]) {
                        case 0xb8:
                                /*
                                 * Group of pictures
                                 */
                                return HEADER_GOP;
                        case 0xb3:
                                /*
                                 * Start of sequence
                                 */
                                return HEADER_SEQUENCE;
                        case 0:
                                /*
                                 * Start of picture
                                 */
                                return HEADER_PICTURE;
                        default:
                                break;
                        }
                        state = STATE_START;
                } else if (buf[0]) {
                        if ((buf[0] == 1)
                            && (state == STATE_FOUND2)) {
                                state = STATE_GET_CODE;
                        } else {
                                state = STATE_START;
                        }
                } else {
                        /*
                         * Found a 0;
                         */
                        switch(state) {
                        case STATE_START:
                                state = STATE_FOUND1;
                                break;
                        case STATE_FOUND1:
                        case STATE_FOUND2:
                                state = STATE_FOUND2;
                                break;
                        default:
                                assert(0);
                        }
                        
                }
                buf++;
        }
        return HEADER_NONE;
}

/*
 * Find the end of this frame.  Put the end in ofs, and whether it's a keyframe
 * into keyframe.  Return 0 on success, nonzero on failure.
 */
static int _find_frame_boundary(const unsigned char *buf, int size, uint64_t *ofs, int *keyframe)
{
        *ofs = 0;
        *keyframe = 0;
        int found_picture = 0;
        HeaderType header;
        while(1) {
                header = _find_header(buf + *ofs, size - *ofs, ofs);
                if (header == HEADER_NONE) {
                        /*
                         * Didn't find enough headers, kick it back.
                         */
                        return -1;
                }
                switch(header) {
                case HEADER_SEQUENCE:
                case HEADER_GOP:
                        if (found_picture) {
                                *ofs -= 4;
                                return 0;
                        }
                        *keyframe = 1;
                        break;
                case HEADER_PICTURE:
                        if (found_picture) {
                                *ofs -= 4;
                                return 0;
                        }
                        found_picture = 1;
                        break;
                default:
                        assert(0);
                        break;
                }
        }
        assert(0);
        return 0;
}

#define READ_SIZE (1024 * 16)


/*
 * Pthread start routine
 */
static void *_start_encode(void *arg)
{
        ((CannedVideoWriter *)arg)->_sx12_encode_loop();
        return NULL;
}



/*
 * Specific interface for the PC104 hardware encoder we're using.  If
 * you use this, then a thread is spawned that writes the output
 * to the file until you call stop_sx12()
 */
void CannedVideoWriter::open_sx12(const char *dev_name, const char *outfile_name, 
                                  int bit_rate, int frame_rate)
{
        _expected_frame_interval = 1.0/(double) frame_rate;
	_sx12_fd = open_wrapper(dev_name, O_RDONLY);
        if (_sx12_fd < 0) {
                throw CannedDataError("Couldn't open xs12 device %s: %s", dev_name, strerror(errno));
        }
	ioctl(_sx12_fd, X12_IOC_SET_BITRATE, bit_rate);

        /*
         * Many more possibilities exist, I just wanted to keep things simple for the moment.
         */
        switch(frame_rate) {
        case 30: 
                ioctl(_sx12_fd, X12_IOC_SET_REDUCED_FRAME_RATE, 0xff00);
                break;
        case 15:
                ioctl(_sx12_fd, X12_IOC_SET_REDUCED_FRAME_RATE, 0x0101);
                break;
        case 10:
                ioctl(_sx12_fd, X12_IOC_SET_REDUCED_FRAME_RATE, 0x0102);
                break;
        case 5:
                ioctl(_sx12_fd, X12_IOC_SET_REDUCED_FRAME_RATE, 0x0103);
                break;
        default:
                throw CannedDataError("Unsupported frame rate: %i", frame_rate);
        }

        open(704, 480, CVT_MPEG2, outfile_name, 0, 15);
        writeVersion(1, 0);
}

void CannedVideoWriter::start_sx12()
{
         if (_sx12_fd == -1)
           throw CannedDataError("Starting SX12 before opening");
        /*
         * Make sure all writes hit before the new thread starts on them 
         */
        fflush(_file);
        _stop_sx12 = 0;
        pthread_attr_t attr;
        pthread_attr_init(&attr);
        pthread_create(&_encode_thread, &attr, _start_encode, this);
}
        

void CannedVideoWriter::_sx12_encode_loop()
{
        CircBuf cb;
        unsigned char tmp[READ_SIZE];
        size_t bytes_read;
        uint64_t ofs = 0;
        unsigned int start = 0;
        int keyframe;

        Time start_time = Time::getRealTimeOfDay();
        int num_frames = 0;
        double total_error = 0;
        int error_n = 0;
        Time frame_time;

        while (!_stop_sx12) {
                while (_find_frame_boundary(cb.getBuf(), cb.size(), &ofs, &keyframe)) {
                    
                        bytes_read = read(_sx12_fd, tmp, READ_SIZE);
                        if (!bytes_read) {
                                goto done;
                                break;
                        }
                        if (int(bytes_read) < 0) {
                          perror("Reading bytes");
                          assert(0);
                        }
                        cb.add(tmp, bytes_read);
                }
                num_frames++;
                double expected_elapsed = _expected_frame_interval*num_frames;
                Time now = Time::getRealTimeOfDay();
                double real_elapsed = (now-start_time).getValue();
                double error = real_elapsed - expected_elapsed;
                total_error += error;
                error_n++;
                double avg_err;
                //printf("%d %d %f %f\n", num_frames, error_n,
                // total_error/error_n, error);
                if (fabs(real_elapsed - expected_elapsed)
                    > MAXIMUM_FRAME_DRIFT) {
                    start_time = frame_time = now;
                    num_frames = 0;
                    total_error = 0;
                    error_n = 0;
                } else if (error_n > 10 &&
                           (fabs(avg_err = total_error/error_n) >
                            _expected_frame_interval)) {
                  error_n = 0;
                  total_error = 0;
                  if (avg_err > 0) {
                    // printf("\tNudging time\n");
                    num_frames++;
                    frame_time = start_time +
                      Time(_expected_frame_interval + expected_elapsed);
                  }  else {
                    // printf("\tSkipping 1\n");
                    num_frames--;
                    cb.consume(ofs);
                    start += ofs;
                    continue;
                  }
                } else {
                  frame_time = start_time + Time(expected_elapsed);
                }

                //printf("%d %d %f\n", num_frames, error_n,
                //       frame_time.getValue());
                _writeFrame(_file, frame_time,
                            cb.getBuf(), ofs, keyframe?CVFT_I:CVFT_P);
                cb.consume(ofs);
                start += ofs;
        }
 done:
        close_wrapper(_sx12_fd);
        if (cb.size()) {
          _writeFrame(_file, Time::getRealTimeOfDay(), cb.getBuf(), cb.size(), utils::CVFT_P);
        }
        close();
}

/*
 * Stop a running encode thread.  Won't return until the encode thread is stopped.
 */
void CannedVideoWriter::stop_sx12()
{
        _stop_sx12 = 1;
        pthread_join(_encode_thread, NULL);
}



__UTILS_END_NAMESPACE
