{***********************************************************************
*                                                                      *
*       dvdControl.pas                                                 *
*                                                                      *
*         Thanks to Takuya Murata for the orginal C++ code.            *
*                                                                      *
*         Delphi 7 translation by Bruce Christensen.                   *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}
//
// Copyright (C) Takuya Murata 2003
// Released under the GPL version 2 or later.
//

unit dvdControl;

interface
  Uses
    Classes , SysUtils , Variants , Windows  ,

    dvdCSS_Auth ;

  Const
    FILE_ANY_ACCESS   = 0         ;
    FILE_READ_ACCESS  = $00000001 ;
    FILE_WRITE_ACCESS = $00000002 ;

    METHOD_BUFFERED = 0 ;

    FILE_DEVICE_CONTROLLER = $00000004 ;
    FILE_DEVICE_DVD        = $00000033 ;

    IOCTL_DVD_BASE     = FILE_DEVICE_DVD        ;
    IOCTL_STORAGE_BASE = $2D                    ;
    IOCTL_SCSI_BASE    = FILE_DEVICE_CONTROLLER ;

    // Used with IOCTL_DVD_END_SESSION to end all DVD sessions at once
    DVD_END_ALL_SESSIONS : LongWord = $ffffffff ;

Type
  TDVD_REGION = packed
    Record
      copy_system   : Byte ;
      region_data   : Byte ;  // current media region (not playable when set)
      system_region : Byte ;  // current drive region (playable when set)
      reset_count   : Byte ;  // number of resets available
    End ;

  TDVD_READ_STRUCTURE = packed
    Record
      // Contains an offset to the logical block address of the descriptor
      // to be retrieved.
      block_byte_offset : Int64 ;  // fixed

      // 0 : Physical descriptor
      // 1 : Copyright descriptor
      // 2 : Disk key descriptor
      // 3 : BCA descriptor
      // 4 : Manufacturer descriptor
      // 5 : Max descriptor
      format : Integer ;

      // Session ID, that is obtained by IOCTL_DVD_START_SESSION
      session : Integer ;

      // From 0 to 4 */
      layer_no : Byte ;
    End ;

  TDVD_LAYER_DESCRIPTOR = packed
    Record
      nLength      : Word ;  // unsigned short
      book_version : Byte ;  // : 4

      // 0:DVD-ROM, 1:DVD-RAM, 2:DVD-R, 3:DVD-RW, 9:DVD-RW
      book_type : Byte ;  // : 4

      minimum_rate : Byte ;  // : 4

      // The physical size of the media. 0:120 mm, 1:80 mm.
      disk_size : Byte ;  // : 4

      // 1:Read-only layer, 2:Recordable layer, 4:Rewritable layer
      layer_type : Byte ;  // : 4

      // 0:parallel track path, 1:opposite track path
      track_path : Byte ;  // : 1

      // 0:one layers, 1:two layers, and so on
      num_of_layers : Byte ;  // : 2
      reserved1 : Byte ;  // : 1

      track_density : Byte ;  // : 4

      linear_density : Byte ;  // : 4

      // Must be either 0x30000:DVD-ROM or DVD-R/-RW or 0x31000:DVD-RAM or DVD+RW
      starting_data_sector : LongWord ;  // unsigned long

      end_data_sector : LongWord ;  // unsigned long

      end_layer_zero_sector : LongWord ;  // unsigned long

      reserved5 : Byte ;  // : 7

      // 0 indicates no BCA data
      BCA_flag : Byte ;  // : 1

      reserved6 : Byte ;
    End ;

  TDVD_COPYRIGHT_DESCRIPTOR = packed
    Record
      protection : Byte ;
      region     : Byte ;
      reserved   : Word ;  // unsigned short
    End ;

  TDVD_MANUFACTURER_DESCRIPTOR = packed
    Record
      manufacturing : Array[0..2047] of Char ;
    End ;

Const
  nDvdChallengeKey = $01;

  nDvdBusKey1   = $02 ;
  nDvdBusKey2   = $03 ;
  nDvdTitleKey  = $04 ;
  nDvdAsf       = $05 ;
  nDvdSetRpcKey = $06 ;

  nDvdGetRpcKey      = $08 ;
  nDvdDiskKey        = $80 ;
  nDvdInvalidateAGID = $3F ;

Type
  TParameters = packed
    Record
      Case Byte of
        1 : (FileHandle : ULONG ;     // declared as LARGE_INTEGER
             Reserved   : ULONG  ) ;  // used for NT alignment

        2 : (TitleOffset : Int64) ;  // declared as LARGE_INTEGER
    End ;

  TDVD_SESSION_ID = ULONG ;
  TDVD_KEY_TYPE   = Integer ;

  TKey_Data = Array[0..2047] of Byte ;

  TDVD_COPY_PROTECT_KEY = packed
    Record
      nLength    : LongWord         ;  // unsigned long
      session_id : TDVD_SESSION_ID  ;  // unsigned long

      KeyType : TDVD_KEY_TYPE  ; // byte?
      flags   : LongWord       ; // unsigned long

      Parameters : TParameters ;

      key_data : TKey_Data ;
    End ;
//
// Predefined (Mt. Fuji) key sizes
// Add sizeof(DVD_COPY_PROTECT_KEY) to get allocation size for
// the full key structure
//
Const
  nKSize = SizeOf(TKey_Data) ;

  nDVD_CHALLENGE_KEY_LENGTH = (  12 + SizeOf(TDVD_COPY_PROTECT_KEY) - nKSize) ;
  nDVD_BUS_KEY_LENGTH       = (   8 + SizeOf(TDVD_COPY_PROTECT_KEY) - nKSize) ;
  nDVD_TITLE_KEY_LENGTH     = (   8 + SizeOf(TDVD_COPY_PROTECT_KEY) - nKSize) ;
  nDVD_DISK_KEY_LENGTH      = (2048 + SizeOf(TDVD_COPY_PROTECT_KEY) - nKSize) ;

Type
  SCSI_ADDRESS = packed
      Record
        Length     : LongWord ;
        PortNumber : Byte     ;
        PathId     : Byte     ;
        TargetId   : Byte     ;
        Lun        : Byte     ;
      End ;

Type
  TOpt = packed
    Record
      info    : Boolean ;
      debug   : Boolean ;
      verbose : Boolean ;
    End ;

  Var
    Opt : TOpt ;

    oDriveInfo : TStringList ;
    
  Procedure Show_Drive_Info(    oFile   : THandle     ;
                                Session : Integer     ;
                                Region  : TDVD_REGION ;
                            Var oList   : TStringList  ) ;

  Function Authenticate(    Letter   : Char     ;
                        Var Disk_Key : TDiskKey  ) : Boolean ;

Implementation

Uses
  Dialogs ,

  bcStrUtils ;

{***********************************************************************
*                                                                      *
*       StrError_Win32                                                 *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function StrError_Win32(    errnum : Integer ;
                        Var buf    : String  ;
                            nSize  : Integer ) : DWORD ;
  Begin  { StrError_Win32 }
    SetLength(buf , nSize) ;
    Result := FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM or
                            FORMAT_MESSAGE_IGNORE_INSERTS ,
	                          nil                           ,
	                          errnum                        ,
	                          0                             ,
	                          PChar(buf)                    ,
	                          nSize                         ,
	                          nil                            ) ;
    SetLength(buf , StrLen(PChar(buf))) ;  // RealizeLength(buf) ;
  End ;  { StrError_Win32 }

  
{***********************************************************************
*                                                                      *
*       Perror_Win32                                                   *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Procedure Perror_Win32(who      : String  ;
                       filename : String  ;
                       lineno   : integer  ) ;
  Var
    Msg : String ;

  Begin  { Perror_Win32 }
    SetLength(msg , 1024) ;
    FillChar(msg[1] , 1024 , ' ') ;
    StrError_Win32(GetLastError , Msg , Length(Msg)) ;
    MessageDlg(who + ' ' + msg + ' at ' + FileName + ':' + IntToStr(LineNo) ,
               mtError ,
               [mbOK]  ,
               0        ) ;
  End ;  { Perror_Win32 }


{***********************************************************************
*                                                                      *
*       CTL_CODE                                                       *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function CTL_CODE(nDeviceType : LongWord ;
                  nFunction   : LongWord ;
                  nMethod     : LongWord ;
                  nAccess     : LongWord  ) : LongWord ;
  Begin  { CTL_CODE }
    Result := ((nDeviceType shl 16) or
               (nAccess shl 14)     or
               (nFunction shl 2)    or
               nMethod                ) ;
  End ;  { CTL_CODE }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_START_SESSION                                        *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_START_SESSION : LongWord ;
  Begin  { IOCTL_DVD_START_SESSION }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0400            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_START_SESSION }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_READ_KEY                                             *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_READ_KEY : LongWord ;
  Begin  { IOCTL_DVD_READ_KEY }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0401            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_READ_KEY }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_SEND_KEY                                             *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_SEND_KEY : LongWord ;
  Begin  { IOCTL_DVD_SEND_KEY }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0402            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_SEND_KEY }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_END_SESSION                                          *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_END_SESSION : LongWord ;
  Begin  { IOCTL_DVD_END_SESSION }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0403            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_END_SESSION }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_SET_READ_AHEAD                                       *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_SET_READ_AHEAD : LongWord ;
  Begin  { IOCTL_DVD_SET_READ_AHEAD }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0404            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_SET_READ_AHEAD }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_GET_REGION                                           *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_GET_REGION : LongWord ;
  Begin  { IOCTL_DVD_GET_REGION }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0405            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_GET_REGION }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_SEND_KEY2                                            *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_SEND_KEY2 : LongWord ;
  Begin  { IOCTL_DVD_SEND_KEY2 }
    Result := CTL_CODE(IOCTL_DVD_BASE  ,
                       $0406           ,
                       METHOD_BUFFERED ,
                       FILE_READ_ACCESS or FILE_WRITE_ACCESS) ;
  End ;  { IOCTL_DVD_SEND_KEY2 }


{***********************************************************************
*                                                                      *
*       IOCTL_DVD_READ_STRUCTURE                                       *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_DVD_READ_STRUCTURE : LongWord ;
  Begin  { IOCTL_DVD_READ_STRUCTURE }
    Result := CTL_CODE(IOCTL_DVD_BASE   ,
                       $0450            ,
                       METHOD_BUFFERED  ,
                       FILE_READ_ACCESS  ) ;
  End ;  { IOCTL_DVD_READ_STRUCTURE }


{***********************************************************************
*                                                                      *
*       IOCTL_STORAGE_SET_READ_AHEAD                                   *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function IOCTL_STORAGE_SET_READ_AHEAD : LongWord ;
  Begin  { IOCTL_STORAGE_SET_READ_AHEAD }
    Result := CTL_CODE(IOCTL_STORAGE_BASE ,
                       $0100              ,
                       METHOD_BUFFERED    ,
                       FILE_READ_ACCESS    ) ;
  End ;  { IOCTL_STORAGE_SET_READ_AHEAD }


{***********************************************************************
*                                                                      *
*       DVD_End_Session                                                *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function DVD_End_Session(ofile   : THandle ;
                         session : Integer  ) : Boolean ;
  Var
    Bytes_Read : DWORD ;

  Begin  { DVD_End_Session }
    If DeviceIoControl(oFile                 ,
                       IOCTL_DVD_END_SESSION ,
                       @session              ,
                       SizeOf(session)       ,
                       nil                   ,
                       0                     ,
                       Bytes_Read            ,
                       nil                    ) then
      Result := True
    Else
      Begin
        MessageDlg('IOCTL_DVD_END_SESSION has failed.' , mtError, [mbOK], 0);
        Result := False ;
      End
  End ;  { DVD_End_Session }


{***********************************************************************
*                                                                      *
*       Show_Drive_Info                                                *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Procedure Show_Drive_Info(    oFile   : THandle     ;
                              Session : Integer     ;
                              Region  : TDVD_REGION ;
                          Var oList   : TStringList  ) ;
  Var
    Read_Struct  : TDVD_READ_STRUCTURE          ;
    CopyRight    : TDVD_COPYRIGHT_DESCRIPTOR    ;
    Layer        : TDVD_LAYER_DESCRIPTOR        ;
    Manufacturer : TDVD_MANUFACTURER_DESCRIPTOR ;

    Bytes_Read : Cardinal ;

  Begin  { Show_Drive_Info }
    FillChar(Read_Struct , SizeOf(Read_Struct) , 0) ;
    read_struct.session  := session ;
    read_struct.layer_no := 0       ;
    read_struct.format   := 0       ;

    If not DeviceIoControl(oFile                    ,
                           IOCTL_DVD_READ_STRUCTURE ,
                           @read_struct             ,
                           SizeOf(Read_Struct)      ,
                           @Layer                   ,
                           SizeOf(Layer)            ,
                           bytes_read               ,
                           nil                       ) then
      Begin
        MessageDlg('Invalid DeviceIoControl return in Show_Drive_Info for ' +
                   CHR($0D) + CHR($0A) + 'IOCTL_DVD_READ_STRUCTURE (1)'     ,
                   mtError ,
                   [mbOK]  ,
                   0        ) ;
        Dvd_End_Session(ofile , Session) ;
        Exit ;
      End ;

    Read_Struct.Format := 1 ;
    If not DeviceIoControl(oFile ,
                           IOCTL_DVD_READ_STRUCTURE ,
                           @read_struct             ,
                           SizeOf(Read_Struct)      ,
	                         @copyright               ,
                           SizeOf(Copyright)        ,
                           Bytes_Read               ,
                           nil                       ) then
      Begin
        MessageDlg('Invalid DeviceIoControl return in Show_Drive_Info for ' +
                   CHR($0D) + CHR($0A) + 'IOCTL_DVD_READ_STRUCTURE (2)'     ,
                   mtError ,
                   [mbOK]  ,
                   0        ) ;
        Dvd_End_Session(ofile , Session) ;
        Exit ;
      End ;

    Read_Struct.Format := 4 ;
    If not DeviceIoControl(oFile                    ,
                           IOCTL_DVD_READ_STRUCTURE ,
                           @read_struct             ,
                           SizeOf(Read_Struct)      ,
                           @manufacturer            ,
                           SizeOf(manufacturer)     ,
                           Bytes_Read               ,
                           nil                       ) then
      Begin
        MessageDlg('Invalid DeviceIoControl return in Show_Drive_Info for ' +
                   CHR($0D) + CHR($0A) + 'IOCTL_DVD_READ_STRUCTURE (2)'     ,
                   mtError ,
                   [mbOK]  ,
                   0        ) ;
        Dvd_End_Session(ofile , Session) ;
        Exit ;
      End ;

    If Pointer(oList) <> nil then
      Begin
        oList.Add('Region code') ;
        oList.Add('    Copy system . . . . : ' + IntToStr(region.copy_system))   ;
        oList.Add('    Region data . . . . : ' + IntToStr(region.region_data))   ;
        oList.Add('    System region . . . : ' + IntToStr(region.system_region)) ;
        oList.Add('    Reset count . . . . : ' + IntToStr(region.reset_count))   ;
        oList.Add(' ') ;

        oList.Add('Layer info ' + IntToStr(0) + '[' + IntToStr(layer.num_of_layers + 1) + ']') ;
        oList.Add('    Book version  . . . : ' + IntToStr(layer.book_version)) ;
        oList.Add('    Book type . . . . . : ' + IntToStr(layer.book_type))    ;
        oList.Add('    Minimum rate  . . . : ' + IntToStr(layer.minimum_rate)) ;
        oList.Add('    Disk size . . . . . : ' + IntToStr(layer.disk_size))    ;
        oList.Add('    Layer type  . . . . : ' + IntToStr(layer.layer_type))   ;
        oList.Add('    Track path  . . . . : ' + IntToStr(layer.track_path))   ;
        oList.Add('    # of layers . . . . : ' + IntToStr(layer.num_of_layers + 1)) ;
        oList.Add('    Track density . . . : ' + IntToStr(layer.track_density))     ;
        oList.Add('    Linear density  . . : ' + IntToStr(layer.linear_density))    ;
        oList.Add('    Start sector  . . . : ' + IntToHex(layer.starting_data_sector , 2))  ;
        oList.Add('    End sector  . . . . : ' + IntToHex(layer.end_data_sector , 2))       ;
        oList.Add('    End sector L0 . . . : ' + IntToHex(layer.end_layer_zero_sector , 2)) ;
        oList.Add('    BCA Flag  . . . . . : ' + IntToStr(layer.BCA_flag))       ;
        oList.Add('    Copy protection . . : ' + IntToStr(copyright.protection)) ;
        oList.Add('    Play region . . . . : ' + IntToStr(copyright.region))     ;
        oList.Add('    Manufacturing . . . : ' + MakeStrV(Manufacturer.Manufacturing , 2048)) ;
      End ;
  End ;  { Show_Drive_Info }


{***********************************************************************
*                                                                      *
*       Is_DVD_Drive                                                 *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function Is_DVD_Drive(Letter : Char) : Boolean ;
  Var
    cPath : String ;

  Begin  { Is_DVD_Drive }
    cPath    := 'c:'   ;
    cPath[1] := Letter ;

    // FIXME: Differentiate the drive is DVD or CD-ROM
    Result := (GetDriveType(PChar(cPath)) = DRIVE_CDROM) ;
  End ;  { Is_DVD_Drive }


{***********************************************************************
*                                                                      *
*       Show_Key_Data                                                  *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function Show_Key_Data(cName    : String  ;
                       pKeyData : Pointer ;
                       nSize    : Integer  ) : String ;
  Begin  { Show_Key_Data }
    Result := cName + ' : ' + MakeHexStr(pKeyData , nSize) ;
  End ;  { Show_Key_Data }


{***********************************************************************
*                                                                      *
*       Reverse_Key_Data                                               *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Procedure Reverse_Key_Data (pKeyData : Pointer ;
                            nSize    : Integer  ) ;
  Type
    TByteArray = Array[0..65535] of Byte ;

  Var
    nI    : Integer ;
    nTemp : Byte    ;

    oKeyData : TByteArray ;

  Begin  { Reverse_Key_Data }
    Move(pKeyData^ , oKeyData , nSize) ;

    For nI := 0 to ((nSize div 2) - 1) do
      Begin
        nTemp := oKeyData[nI] ;
        oKeyData[nI] := oKeyData[(nSize - 1) - nI] ;
        oKeyData[(nSize - 1) - nI] := nTemp ;
      End ;

    Move(oKeyData , pKeyData^ , nSize) ;
  End ;  { Reverse_Key_Data }


{***********************************************************************
*                                                                      *
*       Dvd_Start_Session                                              *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function Dvd_Start_Session(oFile : THandle) : Integer ;
  Var
    Bytes_Read : DWORD    ;
    Session    : Cardinal ;

  Begin  { Dvd_Start_Session }
    If DeviceIoControl(oFile ,
                       IOCTL_DVD_START_SESSION ,
                       nil                     ,
                       0                       ,
                       @Session                ,
                       SizeOf(Session)         ,
                       Bytes_Read              ,
                       nil                      ) then
      Begin
        Result := Session ;
        Exit ;
      End ;

    If GetLastError = ERROR_INVALID_FUNCTION then
      Begin
        Session := DVD_END_ALL_SESSIONS ;

        If DeviceIoControl(oFile                 ,
                           IOCTL_DVD_END_SESSION ,
                           @Session              ,
                           SizeOf(Session)       ,
                           nil                   ,
                           0                     ,
                           Bytes_Read            ,
                           nil                    ) then
          Begin
            Result := Dvd_Start_Session(oFile) ;
            Exit ;
          End ;

        For Session := 0 to 9 do
          Begin
            If DeviceIoControl(oFile ,
                               IOCTL_DVD_END_SESSION ,
                               @Session              ,
                               SizeOf(Session)       ,
                               nil                   ,
                               0                     ,
                               Bytes_Read            ,
                               nil                    ) then
              Begin
                Result := Dvd_Start_Session(oFile) ;
                Exit
              End ;
          End ;
      End ;

    MessageDlg('Error: No result from ', mtError, [mbOK], 0);
    Result := 0 ;
  End ;  { Dvd_Start_Session }


{***********************************************************************
*                                                                      *
*       Dvd_Get_Disk_Key                                               *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function Dvd_Get_Disk_Key(    oFile    : THandle  ;
                              Session  : Integer  ;
                          Var Disk_Key : TDiskKey  ) : Boolean ;
  Var
    Read_Struct : TDVD_READ_STRUCTURE ;

    Bytes_Read : DWORD ;

  Begin  { Dvd_Get_Disk_Key }
    FillChar(Read_Struct , SizeOf(Read_Struct) , 0) ;

    Read_Struct.Session  := Session ;
    Read_Struct.Layer_No := 0       ;

    Read_Struct.Format := 2 ;
    If DeviceIoControl(oFile ,
                       IOCTL_DVD_READ_STRUCTURE ,
                       @read_struct             ,
                       SizeOf(Read_Struct)      ,
                       @disk_key                ,
                       SizeOf(disk_key)         ,
                       Bytes_Read               ,
                       nil                       ) then
      Result := True
    Else
      Begin
        MessageDlg('DeviceIoControl error in Dvd_Get_Disk_Key' ,
                   mtError ,
                   [mbOK]  ,
                   0        ) ;
        Dvd_End_Session(oFile , Session) ;
        Result := False ;
      End ;
  End ;  { Dvd_Get_Disk_Key }


{***********************************************************************
*                                                                      *
*       Authenticate                                                   *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Function Authenticate(    Letter   : Char     ;
                      Var Disk_Key : TDiskKey  ) : Boolean ;
  Const
    c_key : TChallengeKey = (9, 8, 7, 6, 5, 4, 3, 2, 1, 0, 0, 0) ;

  Var
    oFile : THandle ;

    cPath : String ;

    Bytes_Read : DWORD ;

    Session : Integer ;
    Region : TDVD_REGION ;
    Key : TDVD_COPY_PROTECT_KEY ;

    Key2 : TCSS_Key ;

    KeyChallenge : TChallenge ;

  Begin  { Authenticate }
    cPath := '\\.\x:' ;
    FillChar(Key , SizeOf(Key) , 0) ;

    If not Is_Dvd_Drive(Letter) then
      Begin
        MessageDlg(Letter + ': drive is not a DVD drive', mtError, [mbOK], 0);
        Result := False ;
        Exit ;
      End ;

    cPath[5] := Letter ;
    oFile := CreateFile(PChar(cPath)                        ,
                        GENERIC_READ                        ,
                        FILE_SHARE_READ or FILE_SHARE_WRITE ,
                        nil                                 ,
                        OPEN_EXISTING                       ,
                        0                                   ,
                        0                                   ) ;
    If (oFile = INVALID_HANDLE_VALUE) then
      Begin
        oDriveInfo.Add('CreateFile failed in procedure Authenticate') ;
        Result := False ;
        Exit ;
      End ;

    If not DeviceIoControl(oFile                ,
                           IOCTL_DVD_GET_REGION ,
                           nil                  ,
                           0                    ,
                           @region              ,
                           SizeOf(Region)       ,
                           Bytes_Read           ,
                           nil                   ) then
      Begin
        oDriveInfo.Add('Unable to retrieve Region in procedure Authenticate') ;
        Result := False ;
        Exit ;
      End ;

    Session := Dvd_Start_Session(oFile) ;

//    If opt.info then
      Show_Drive_Info(oFile , Session , Region , oDriveInfo) ;

    // Send our challenge key
    key.nLength    := nDVD_CHALLENGE_KEY_LENGTH ; // 12 + 24
    key.session_id := Session          ;
    key.KeyType    := nDvdChallengeKey ;
    key.Flags      := 0                ;

//		key.challenge_key := c_key ;
		Move(c_key , key.key_data , SizeOf(c_key)) ;

    If not DeviceIoControl(oFile              ,
                           IOCTL_DVD_SEND_KEY ,
                           @key               ,
                           key.nLength        ,
                           nil                ,
                           0                  ,
                           Bytes_Read         ,
                           nil                 ) then
      Begin
        oDriveInfo.Add('**** Error sending challenge key in procedure Authenticate') ;
        Dvd_End_Session(oFile , Session );
        Result := False ;
        Exit ;
      End ;

    // Receive the bus-key1 (or authentication key)
    key.nLength := nDVD_BUS_KEY_LENGTH ;  // 8 + 24
    key.KeyType := nDvdBusKey1 ;
    If not DeviceIoControl(oFile              ,
                           IOCTL_DVD_READ_KEY ,
                           @key               ,
                           key.nLength        ,
                           @key               ,
                           key.nLength        ,
                           Bytes_Read         ,
                           nil                 ) then
      Begin
        oDriveInfo.Add('**** Error retrieving Bus Key 1 in procedure Authenticate') ;
        Dvd_End_Session(oFile , Session) ;
        Result := False ;
        Exit ;
      End ;

//    If opt.verbose then
        oDriveInfo.Add(Show_Key_Data('Bus-key1' , @key.key_data , 8)) ;

    // Receive the challenge key
    key.nLength := nDVD_CHALLENGE_KEY_LENGTH ;  // 12 + 24
    key.KeyType := nDvdChallengeKey ;

    If not DeviceIoControl(oFile              ,
                           IOCTL_DVD_READ_KEY ,
                           @Key               ,
                           key.nLength        ,
                           @Key               ,
                           key.nLength        ,
                           Bytes_Read         ,
                           nil                 ) then
      Begin
        Dvd_End_Session(oFile , Session) ;

        oDriveInfo.Add('**** Error receiving challenge key in procedure Authenticate') ;
        Result := False ;
        Exit ;
      End ;

    Reverse_Key_Data(@key.key_data , 10) ;

//    If opt.verbose then
      oDriveInfo.Add(Show_Key_Data('Challenge-key' , @key.key_data , 10)) ;

    Move(Key.Key_Data[0] , KeyChallenge[0] , SizeOf(KeyChallenge)) ;
    CryptKey2(0 , KeyChallenge , key2) ;

    Reverse_Key_Data(@key2 , 5) ;

//    If opt.verbose then
    oDriveInfo.Add(Show_Key_Data('Bus-key2' , @key2 , 8)) ;

    Move(Key2 , Key.key_data , 8) ; //memcpy (key.key_data, key2, 8);

    // Send the bus key2
    key.nLength := 8 + 24 ;  // nDVD_BUS_KEY_LENGTH ;  // 8 + Length(...) 
    key.KeyType := nDvdBusKey2 ;

    If not DeviceIoControl(oFile              ,
                           IOCTL_DVD_SEND_KEY ,
                           @key               ,
                           key.nLength        ,
                           nil                ,
                           0                  ,
                           Bytes_Read         ,
                           nil                 ) then
      Begin
        oDriveInfo.Add('**** Error sending Bus Key 2 in procedure Authenticate') ;
        Dvd_End_Session(oFile , Session) ;

        Result := False ;
        Exit ;
      End ;

    Result := True ;  // Drive is authourised!

    // It is necessary to retrieve the disk key to complete authentication.
    Dvd_Get_Disk_Key(oFile , Session , Disk_Key) ;
    oDriveInfo.Add(Show_Key_Data('Drive Key (12 bytes)' , @Disk_Key , 12)) ;

    Dvd_End_Session(oFile, Session) ;
  End ;  { Authenticate }


{***********************************************************************
*                                                                      *
*       Do_Test                                                        *
*                                                                      *
*       Modifications                                                  *
*       ===============                                                *
*                                                                      *
***********************************************************************}

Procedure Do_Test ;
  Var
    nI : Integer ;

    Enc_Key : TCSS_Key ;

  Const
    c_key : TChallenge = (0, 1, 2, 3, 4, 5, 6, 7, 8, 9) ;

  Begin  { Do_Test }
    For nI := 0 to 31 do
      Begin
        CryptKey1(nI , c_key , Enc_Key) ;
        Reverse_Key_Data(@Enc_Key , 5) ;
        Show_Key_Data('Encrypted key' , @Enc_key , 5) ;
      End ;
  End ;  { Do_Test }


{***********************************************************************
*                                                                      *
*       IOCTL_SCSI_GET_ADDRESS                                         *
*                                                                      *
*       Modifications                                                  *
*       =============                                                  *
*                                                                      *
***********************************************************************}

Function IOCTL_SCSI_GET_ADDRESS : LongWord ;
  Begin  { IOCTL_SCSI_GET_ADDRESS }
    Result := CTL_CODE(IOCTL_SCSI_BASE ,
                       $0406           ,
                       METHOD_BUFFERED ,
                       FILE_ANY_ACCESS  ) ;
  End ;  { IOCTL_SCSI_GET_ADDRESS }


{***********************************************************************

*                                                                      *
*       GetDriveInfoNT                                                 *
*                                                                      *
*       Modifications                                                  *
*       =============                                                  *
*                                                                      *
***********************************************************************}

Function GetDriveInfoNT(    DriveLetter : Char ;
                        Var Host        : Byte ;
                        Var Target      : Byte ;
                        Var Lun         : Byte  ) : Integer ;
  Const
    DrivePath : String[7] = '\\.\x:'#$00 ;

  Var
    H : THandle ;

    BytesReturned : LongWord ;
    DataBuffer : SCSI_ADDRESS ;

  Const
    Overlap : Boolean = False ;

  Begin  { DeCSSauth.GetDriveInfoNT }
//    printf("Using WinNT drive-searcher\n");

	  DrivePath[5] := DriveLetter ;
  	h := CreateFile(@DrivePath[1] ,
                    GENERIC_READ,
                    FILE_SHARE_READ or FILE_SHARE_WRITE,
			              nil ,
                    OPEN_EXISTING ,
                    0 ,
                    THandle(0)) ;
  	If h = INVALID_HANDLE_VALUE then
      Begin
        Result := 0 ;
        Exit ;
      End ;

    FillChar(DataBuffer , Sizeof(DataBuffer) , 0) ;
    If not DeviceIoControl(h                      ,
                           IOCTL_SCSI_GET_ADDRESS ,
                           nil                    ,
                           0                      ,
                           @DataBuffer            ,
                           SizeOf(DataBuffer)     ,
                           BytesReturned          ,
                           nil                     ) then
      Begin
        Result := 0 ;
        Exit
      End ;

    Host   := dataBuffer.PortNumber ;
    Target := dataBuffer.TargetId   ;
    Lun    := dataBuffer.Lun        ;

    CloseHandle(h);
    Result := 1 ;
  End ;  { GetDriveInfoNT }

Initialization
  With Opt do
    Begin
      info    := True ;
      debug   := True ;
      verbose := True ;
    End ;

  oDriveInfo := TStringList.Create ;

Finalization
  oDriveInfo.Free ;
End.
