(* AstToEnglish by Jonathan Baccash *)
structure AstToEnglish :> AST2ENGLISH =
struct
  (* CKit wrappers. *)
  structure H = HashTable
  open Ast

  exception eHash

                     (* list of keywords -- id's that conflict are quoted *)
  val lKeyWords =   ["and", "or", "starting", "where", "additive", "address",
                     "After",
                     "after", "a", "an", "and", "otherwise", "another",
                     "are", "argument", "arguments", "array", "as", "assign",
                     "assigning", "Assume", "assume", "at", "automatic",
                     "be", "before", "beginning", "bits", "bitwise", "block",
                     "both", "Break", "break", "by", "Call", "call", "called",
                     "calling", "case", "case-block", "cast", "character",
                     "code", "complement", "completion", "Compute", "compute",
                     "computing", "constant", "Continue", "continue",
                     "contents", "continuing", "Continuing",
                     "Decrement", "decrement", "decrementing", "default",
                     "defined", "divided", "Do", "do", "does",
                     "double-precision", "each", "either", "element",
                     "elements", "enclosing", "end", "ends", "enumeration",
                     "equal", "Evaluate", "evaluate", "exclusive",
                     "execution", "exists", "external", "extralong",
                     "First", "first", "floating", "point", "number",
                     "follows", "for", "fractional", "from", "function",
                     "Go", "go", "greater", "if", "in", "Increment",
                     "increment", "index", "initial", "integer", "inverse",
                     "is", "It", "it", "iteration", "Label", "label",
                     "left", "length", "less", "Let", "let", "list", "long",
                     "loop", "member", "members", "minus", "modulus",
                     "name", "nearest", "no", "nonzero", "not", "nothing",
                     "on", "next", "Next",
                     "of", "one", "only", "operation", "or", "otherwise",
                     "Pass", "pass", "passing",
                     "Perform", "perform", "plus", "point", "pointed",
                     "pointer", "prior", "provided", "Repeatedly",
                     "repeatedly", "register", "result", "Return", "return",
                     "returning", "right", "saturate", "Set", "set",
                     "shifted", "short", "signed", "size", "start", "starting",
                     "static", "string", "structure", "such", "than", "the",
                     "then", "Then", "that",
                     "This", "this", "times", "To", "to", "type", "union",
                     "unknown", "unnamed", "unsigned", "upon", "value",
                     "values", "variable", "void", "volatile", "where",
                     "we",
                     "which", "whose", "with", "zero"]
  val hKeyWords : (string, unit) H.hash_table = (* keywords hash table *)
       H.mkTable (HashString.hashString, op =) (128 (* sizeHint *), eHash)
  val _ = map (H.insert hKeyWords) (map (fn s => (s,())) lKeyWords)

  datatype ARTICLE = AN
  datatype LETTER = VOWEL | CONS

  fun letterClass #"a" = VOWEL
    | letterClass #"e" = VOWEL
    | letterClass #"i" = VOWEL
    | letterClass #"o" = VOWEL
    | letterClass #"u" = VOWEL
    | letterClass #"A" = VOWEL
    | letterClass #"E" = VOWEL
    | letterClass #"I" = VOWEL
    | letterClass #"O" = VOWEL
    | letterClass #"U" = VOWEL
    | letterClass #"8" = VOWEL
    | letterClass _ = CONS

  fun astToEnglish strm (tidtab: Tables.tidtab) =
    let
      val nNext : int ref = ref 0

      (* all prints are to the output stream *)
     fun print s = TextIO.output(strm, s)

      (* conditionally print a string beginning in upper or lower case *)
      fun lPrint (s, lowerCase) =
        if lowerCase then print s
        else (print (implode[Char.toUpper(hd(explode s))]);
              print (implode(tl(explode s))))

      fun peArt (SOME AN, VOWEL) = print "an "
        | peArt (SOME AN, CONS) = print "a "
        | peArt (NONE, _) = ()

      (* puts an id in quotes if it conflicts with our lexer tokens *)
      fun mkNice s =
        let fun checkKeyWords s = (H.find hKeyWords s) <> NONE
            fun isInt nil = true
              | isInt (x::xs) = Char.isDigit x andalso isInt xs
            fun isTemp nil = true
              | isTemp (c::cs) = (c = #"e") andalso isInt cs
            fun isElement' nil = true
              | isElement' (c::cs) = ((c = #"_") orelse Char.isDigit c)
                                     andalso isElement' cs
            fun isElement nil = true
              | isElement (c::cs) = (c = #"a") andalso isElement' cs
            fun isBad s = checkKeyWords s orelse isTemp (explode s)
                          orelse isElement (explode s)
        in if isBad s then "\"" ^ s ^ "\"" else s end

      fun peSymbol name = print (mkNice (Symbol.name name))

      fun peIdName (id as {name, ...} : Ast.id) =
            peSymbol name

      fun generatePrePrefix (initial, counter) =
        let fun noTrailDigits nil = nil
              | noTrailDigits (x::xs) =
                 let val L = noTrailDigits xs
                 in case L of nil => if Char.isDigit x then nil else x::nil
                            | _ => x::xs end
            fun stripEndInt s = implode(noTrailDigits(explode s))
            fun try (s, n) = initial=n orelse
                 (case Tidtab.find (tidtab,n)
                  of SOME {name=SOME id, ...} => s <> stripEndInt id andalso
                                                 try(s, n-1)
                   | _ => try(s, n-1))
            fun f(prePrefix, s) = if try(prePrefix ^ s, counter) then prePrefix
                               else f(prePrefix ^ s, s)
        in f("", "t") end
      val prePrefix = generatePrePrefix (Tid.initial, Tid.new())

      fun peTid (art, tid) =
            case Tidtab.find (tidtab, tid)
            of SOME {name=SOME id, ...} =>
                     (peArt(art, letterClass( hd(explode(id)) ));
                      print (mkNice id))
             | _ => (peArt (art, CONS); print (prePrefix ^ (Tid.toString tid)))

      fun peStorageClass (a, STATIC) = (peArt (a, CONS); print "static ")
        | peStorageClass (a, EXTERN) = (peArt (a, VOWEL); print "external ")
        | peStorageClass (a, REGISTER) = (peArt (a, CONS); print "register ")
        | peStorageClass (a, AUTO) = (peArt (a, VOWEL); print "automatic ")
        | peStorageClass (_, DEFAULT) = ()

      fun peQualifier (art, qual) =
        let fun peQual CONST = print "constant "
              | peQual VOLATILE = print "volatile "
        in peArt(art, CONS); peQual qual end

      fun peSaturatedness SATURATE = print "saturate "
        | peSaturatedness NONSATURATE = ()

      fun peFractionality FRACTIONAL = print "fractional "
        | peFractionality WHOLENUM = ()

      fun peSignedness SIGNED = ()
        | peSignedness UNSIGNED = print "unsigned "

      fun peIntKind (a, ik) =
        let val s = case ik
                      of CHAR => (peArt (a, CONS); "character")
                       | SHORT => (peArt (a, CONS); "short integer")
                       | INT => (peArt (a, VOWEL); "integer")
                       | LONG => (peArt (a, CONS); "long integer")
                       | LONGLONG => (peArt (a, VOWEL); "extralong integer")
                       | FLOAT => (peArt (a, CONS); "floating point number")
                       | DOUBLE => (peArt (a, CONS);
                                    "double-precision floating point number")
                       | LONGDOUBLE => (peArt (a, CONS);
                            "long double-precision floating point number")
        in print s end

      fun peList f nil = ()
        | peList f (x::nil) = f x
        | peList f (x1::x2::nil) = (f x1; print " and "; f x2)
        | peList f (x::xs) = let fun loop (x::nil) = (print ", and "; f x)
                                   | loop (x::xs) = (print ", "; f x; loop xs)
                             in (f x; loop xs) end

      fun peLargeInt i = if IntInf.>=(i,IntInf.fromInt 0) then print (IntInf.toString i)
                         else print ("-" ^ IntInf.toString (IntInf.~i))
      fun myIntToS i = if i >=0 then (Int.toString i)
                       else ("-" ^ Int.toString (~i))
      fun peInt i = print (myIntToS i)

      fun peStringConst s = (print "the string \""; print (String.toCString s);
                             print "\"")

      fun stripCoreStmt (STMT (s, _, _)) = s
      fun stripCoreExpr (EXPR (e, _, _)) = e

      (* isNestable (e1, e2) = is e2 nestable in e1?               *)
      (* Can nest when it won't cause an ambiguity in the grammar. *)
      val isNestable =
           fn (IntConst li,_)       => true (* a sentinel *)
            | (_,IntConst li)       => true
            | (_,RealConst r)       => true
            | (_,StringConst s)     => true
            | (_,Id id)             => true
            | (_,SizeOf _)          => true (* child of SizeOf is a type *)
            | (_,EnumId _)          => true
            | (Call _, Assign _)    => true
            | (Call _, Sub _)       => true
            | (Call _, Member _)    => true
            | (Call _, Arrow _)     => true
            | (Call _, Deref _)     => true
            | (Call _, AddrOf _)    => true
            | (Call _, Unop (Negate,_))    => true
            | (Call _, Unop (BitNot,_))    => true
            | (Assign _, Call _)    => true
            | (Assign _, Assign _)  => true
            | (Assign _, Comma _)   => true
            | (Assign _, Sub _)     => true
            | (Assign _, Member _)  => true
            | (Assign _, Arrow _)   => true
            | (Assign _, Deref _)   => true
            | (Assign _, AddrOf _)  => true
            | (Assign _, Unop (Negate,_))  => true
            | (Assign _, Unop (BitNot,_))  => true
            | (Assign _, Binop (BitOr,_,_)) => true
            | (Assign _, Binop (BitAnd,_,_)) => true
            | (Assign _, Binop (BitXor,_,_)) => true
            | (Member _, Call _)    => true
            | (Member _, Sub _)     => true
            | (Member _, Member _)  => true
            | (Member _, Arrow _)   => true
            | (Member _, Deref _)   => true
            | (Arrow _, Call _)     => true
            | (Arrow _, Sub _)      => true
            | (Arrow _, Member _)   => true
            | (Arrow _, Arrow _)    => true
            | (Arrow _, Deref _)    => true
            | (Deref _, Call _)     => true
            | (Deref _, Sub _)      => true
            | (Deref _, Member _)   => true
            | (Deref _, Arrow _)    => true
            | (Deref _, Deref _)    => true
            | (Deref _, AddrOf _)   => true
            | (AddrOf _, Sub _)     => true
            | (AddrOf _, Member _)  => true
            | (AddrOf _, Arrow _)   => true
            | (Binop (BitOr,_,_), Deref _) => true
            | (Binop (BitAnd,_,_), Deref _) => true
            | (Binop (BitXor,_,_), Deref _) => true
            | (Binop (BitOr,_,_), Sub _) => true
            | (Binop (BitAnd,_,_), Sub _) => true
            | (Binop (BitXor,_,_), Sub _) => true
            | (Binop (BitOr,_,_), Member _) => true
            | (Binop (BitAnd,_,_), Member _) => true
            | (Binop (BitXor,_,_), Member _) => true
            | (Binop (BitOr,_,_), Arrow _) => true
            | (Binop (BitAnd,_,_), Arrow _) => true
            | (Binop (BitXor,_,_), Arrow _) => true
            | (Binop (BitOr,_,_), AddrOf _) => true
            | (Binop (BitAnd,_,_), AddrOf _) => true
            | (Binop (BitXor,_,_), AddrOf _) => true
            | (Binop (BitOr,_,_), Unop(BitNot,_)) => true
            | (Binop (BitAnd,_,_), Unop(BitNot,_)) => true
            | (Binop (BitXor,_,_), Unop(BitNot,_)) => true
            | (Unop (Not,_), _)     => true  (* ?? i think so *)
            | (Unop (Uplus,_), _)   => true
            | _                     => false

      fun nNestedExprTemps e1 e2 = if isNestable (e1,e2)
                                     then nExprTemps e2 else 1
      and nNestedListExprTemps e L =
             foldr (fn (x,i) => i+nNestedExprTemps e x) 0 L
      and nExprTemps e =
        let val f  = nNestedListExprTemps e
            val g = nNestedExprTemps e
        in
          (fn IntConst li => 0
            | RealConst r => 0
            | StringConst s => 0
            | Id id => 0
            | Call(ex, exs) => f (map stripCoreExpr (ex::exs))
            | QuestionColon(e0, e1, e2) => f (map stripCoreExpr [e0, e1, e2])
            | Assign(e0, e1) => f (map stripCoreExpr [e0, e1])
            | Comma(e0, e1) => f (map stripCoreExpr [e0, e1])
            | Sub(e0, e1) => f (map stripCoreExpr [e0, e1])
            | Member(exp, member) => g (stripCoreExpr exp)
            | Arrow(exp, member) => g (stripCoreExpr exp)
            | Deref exp => g (stripCoreExpr exp)
            | AddrOf exp => g (stripCoreExpr exp)
            | Binop(binop, e0, e1) => f (map stripCoreExpr [e0, e1])
            | Unop(unop, exp) => g (stripCoreExpr exp)
            | Cast(ctype, exp) => g (stripCoreExpr exp)
            | EnumId (id, li) => 0
            | SizeOf ctype => 0
            | ExprExt ee => 0 (* ee is of unit type *)
            | ErrorExpr => 0) e end
      fun mapNestedExprTemps (e, nil, i) = nil
        | mapNestedExprTemps (e, x::xs, i) =
           (x,i)::(mapNestedExprTemps (e, xs, i+nNestedExprTemps e x))

      fun peNestedBoolExpr e1 (e2,i) = if isNestable (e1,e2)
                                       then peSimpleBoolExpr (e2,i)
                                       else (print "zero does not equal e";
                                             peInt i)

      and peNestedExpr e1 (e2,i) = if isNestable (e1,e2)
                                   then peSimpleExpr (e2, i)
                                   else (print "e"; peInt i)

      and peNestedExprList e L = peList (peNestedExpr e) L

      (* print an expression in a boolean context *)
      and peSimpleBoolExpr (e, i) =
        let fun pe() = (print "zero does not equal "; peSimpleExpr (e, i))
            val nNestedListExprTemps = nNestedListExprTemps e
            val peNestedExprList = peNestedExprList e
            val peNestedExpr = peNestedExpr e
            val nNestedExprTemps = nNestedExprTemps e
        in case e
           of Binop(binop, e0, e1) =>
              let val [e0, e1] = map stripCoreExpr [e0, e1]
                  val i1 = i+nNestedExprTemps e0
              in (case binop
                  of Gt => (peNestedExpr (e0, i); print " is greater than ";
                            peNestedExpr (e1, i1))
                   | Lt => (peNestedExpr (e0, i); print " is less than ";
                            peNestedExpr (e1, i1))
                   | Gte => (peNestedExpr (e0, i);
                             print " is greater than or equal to ";
                             peNestedExpr (e1, i1))
                   | Lte => (peNestedExpr (e0, i);
                             print " is less than or equal to ";
                             peNestedExpr (e1, i1))
                   | Eq => (peNestedExpr (e0, i); print " is equal to ";
                            peNestedExpr (e1, i1))
                   | Neq => (peNestedExpr (e0, i); print " is not equal to ";
                             peNestedExpr (e1, i1))
                   | And => (print "both "; peNestedExpr (e0, i);
                             print " and ";
                             peNestedExpr (e1, i1); print " are nonzero")
                   | Or => (print "either "; peNestedExpr (e0, i);
                            print " or ";
                            peNestedExpr (e1, i1); print " is nonzero")
                   | _ => pe()) end
            | Unop(Not, e) =>
              let val e = stripCoreExpr e
              in (print "zero equals "; peNestedExpr(e,i)) end
            | _ => pe() end

      (* print an expression in a context that requires a result *)
      and peSimpleExpr (e, i) =
       let val peNestedExpr = peNestedExpr e
           val peNestedBoolExpr = peNestedBoolExpr e
           val peNestedExprList = peNestedExprList e
           val nNestedExprTemps = nNestedExprTemps e
           val nNestedListExprTemps = nNestedListExprTemps e
       in
        case e
        of IntConst li => peLargeInt li
         | RealConst r => print (Real.toString r)
         | StringConst s => peStringConst s
         | Id id => peIdName id
         | Call(ex, exs) =>
            let val exs = map stripCoreExpr exs
                val ex = stripCoreExpr ex
            in (case exs of nil => print "the result of calling "
                | _ => (print "the result of passing ";
                        peNestedExprList (mapNestedExprTemps (e, exs,
                                                i+nNestedExprTemps ex));
                        print " to ");
                peNestedExpr (ex, i))
            end
         | QuestionColon(e0, e1, e2) =>
            let val [e0, e1, e2] = map stripCoreExpr [e0, e1, e2]
                val [_, (_,i1), (_,i2)] = mapNestedExprTemps (e, [e0,e1,e2], i)
            in (peNestedExpr (e1, i1);
                print " if "; peNestedBoolExpr (e0, i);
                print " and otherwise "; peNestedExpr (e2, i2))
            end
         | Assign(e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (print "the contents of "; peNestedExpr (e0, i);
                print " upon assigning "; peNestedExpr (e1, i1);
                print " to it") end
         | Comma(e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (peNestedExpr (e1, i1);
                print " after computing "; peNestedExpr (e0, i)) end
         | Sub(e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (print "the element of "; peNestedExpr (e0, i);
                print " at index "; peNestedExpr (e1, i1)) end
         | Member(exp, {name, ...}) =>
             let val exp = stripCoreExpr exp
             in (print "member "; peSymbol name;
                 print " of "; peNestedExpr (exp, i)) end
         | Arrow(exp, {name, ...}) =>
             let val exp = stripCoreExpr exp
             in (print "member "; peSymbol name;
                 print " of the structure pointed to by ";
                 peNestedExpr (exp, i)) end
         | Deref exp =>
             let val exp = stripCoreExpr exp
             in (print "the variable pointed to by ";
                 peNestedExpr (exp, i)) end
         | AddrOf exp =>
             let val exp = stripCoreExpr exp
             in (print "the address of "; peNestedExpr (exp, i)) end
         | Binop(binop, e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (case binop
              of Plus => (peNestedExpr (e0, i); print " plus ";
                          peNestedExpr (e1, i1))
               | Minus => (peNestedExpr (e0, i); print " minus ";
                           peNestedExpr (e1, i1))
               | Times => (peNestedExpr (e0, i); print " times ";
                           peNestedExpr (e1, i1))
               | Divide => (peNestedExpr (e0, i); print " divided by ";
                            peNestedExpr (e1, i1))
               | Mod => (peNestedExpr (e0, i); print " modulus ";
                         peNestedExpr (e1, i1))
               | Gt => (print "nonzero if and only if "; peNestedExpr (e0, i);
                        print " is greater than "; peNestedExpr (e1, i1))
               | Lt => (print "nonzero if and only if "; peNestedExpr (e0, i);
                        print " is less than "; peNestedExpr (e1, i1))
               | Gte => (print "nonzero if and only if "; peNestedExpr (e0, i);
                         print " is greater than or equal to ";
                         peNestedExpr (e1, i1))
               | Lte => (print "nonzero if and only if "; peNestedExpr (e0, i);
                         print " is less than or equal to ";
                         peNestedExpr (e1, i1))
               | Eq => (print "nonzero if and only if "; peNestedExpr (e0, i);
                        print " is equal to ";
                        peNestedExpr (e1, i1))
               | Neq => (print "nonzero if and only if "; peNestedExpr (e0, i);
                         print " is not equal to ";
                         peNestedExpr (e1, i1))
               | And => (print "nonzero if and only if "; peNestedExpr (e0, i);
                         print " and "; peNestedExpr (e1, i1);
                         print " are nonzero")
               | Or => (print "nonzero if and only if "; peNestedExpr (e0, i);
                        print " or "; peNestedExpr (e1, i1);
                        print " is nonzero")
               | BitOr => (print "the bitwise or of "; peNestedExpr (e0, i);
                           print " and "; peNestedExpr (e1, i1))
               | BitAnd => (print "the bitwise and of "; peNestedExpr (e0, i);
                            print " and "; peNestedExpr (e1, i1))
               | BitXor => (print "the bitwise exclusive or of ";
                            peNestedExpr (e0, i);
                            print " and "; peNestedExpr (e1, i1))
               | Lshift => (peNestedExpr (e0, i); print " shifted left ";
                            peNestedExpr (e1, i1); print " bits")
               | Rshift => (peNestedExpr (e0, i); print " shifted right ";
                            peNestedExpr (e1, i1); print " bits")
               | PlusAssign => (peNestedExpr (e0, i);
                                print " after incrementing ";
                                peNestedExpr (e0, i); print " by ";
                                peNestedExpr (e1, i1))
               | MinusAssign => (peNestedExpr (e0, i);
                                 print " after decrementing ";
                                 peNestedExpr (e0, i); print " by ";
                                 peNestedExpr (e1, i1))
               | TimesAssign => (peNestedExpr (e0, i);
                                 print " after assigning ";
                                 peNestedExpr (e0, i); print " times ";
                                 peNestedExpr (e1, i1);
                                 print " to ";
                                 peNestedExpr (e0, i))
               | DivAssign => (peNestedExpr (e0, i);
                               print " after assigning ";
                               peNestedExpr (e0, i); print " divided by ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | ModAssign => (peNestedExpr (e0, i);
                               print " after assigning ";
                               peNestedExpr (e0, i); print " modulus ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | XorAssign => (peNestedExpr (e0, i);
                               print " after assigning the bitwise";
                               print " exclusive or of ";
                               peNestedExpr (e0, i); print " and ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | OrAssign => (peNestedExpr (e0, i);
                              print " after assigning the bitwise or of ";
                              peNestedExpr (e0, i); print " and ";
                              peNestedExpr (e1, i1);
                              print " to ";
                              peNestedExpr (e0, i))
               | AndAssign => (peNestedExpr (e0, i);
                               print " after assigning the bitwise and of ";
                               peNestedExpr (e0, i); print " and ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | LshiftAssign => (peNestedExpr (e0, i);
                                  print " after assigning ";
                                  peNestedExpr (e0, i);
                                  print " shifted left ";
                                  peNestedExpr (e1, i1);
                                  print " bits to ";
                                  peNestedExpr (e0, i))
               | RshiftAssign => (peNestedExpr (e0, i);
                                  print " after assigning ";
                                  peNestedExpr (e0, i);
                                  print " shifted right ";
                                  peNestedExpr (e1, i1);
                                  print " bits to ";
                                  peNestedExpr (e0, i))
               | BinopExt () => ()) end
         | Unop(unop, exp) => let val exp = stripCoreExpr exp in
            (case unop of
                 Uplus => (peNestedExpr (exp, i))
               | Not => (print "zero if and only if ";
                         peNestedBoolExpr (exp, i))
               | Negate => (print "the additive inverse of ";
                            peNestedExpr (exp, i))
               | BitNot => (print "the bitwise complement of ";
                            peNestedExpr (exp, i))
               | PreInc => (peNestedExpr (exp, i);
                            print " after incrementing ";
                            peNestedExpr (exp, i); print " by one")
               | PostInc => (peNestedExpr (exp, i);
                             print " before incrementing ";
                             peNestedExpr (exp, i); print " by one")
               | PreDec => (peNestedExpr (exp, i);
                            print " after decrementing ";
                            peNestedExpr (exp, i); print " by one")
               | PostDec => (peNestedExpr (exp, i);
                             print " before decrementing ";
                             peNestedExpr (exp, i); print " by one")
               | UnopExt () => ()) end
         | Cast(ctype, exp) =>
            (peNestedExpr (stripCoreExpr exp, i); print " cast to ";
             peType (SOME AN, DEFAULT, ctype))
         | EnumId ({name, ...}, li) => (print "enumeration ";
                                        peSymbol name)
         | SizeOf ctype =>
            (print "the size of "; peType (SOME AN, DEFAULT, ctype))
         | ExprExt () => ()
         | ErrorExpr => () end

      (* a follow up expression -- e## is a temporary variable *)
      and peSimpleTaggedExpr ((e, i), n) =
            (print "e"; peInt n; print " is "; peSimpleExpr (e,i))

      and peSimpleExprList L =
        let fun f (nil, i) = nil
              | f (x::xs, i) = (x, i)::f(xs, i+1)
        in peList peSimpleTaggedExpr (f (L, 1)) end

      and getImmediateSubExprs e =
        case e
        of IntConst li => nil
         | RealConst r => nil
         | StringConst s => nil
         | Id id => nil
         | Call (ex, exs) => map stripCoreExpr (ex::exs)
         | QuestionColon(e0, e1, e2) => map stripCoreExpr [e0, e1, e2]
         | Assign(e0, e1) => map stripCoreExpr [e0, e1]
         | Comma(e0, e1) => map stripCoreExpr [e0, e1]
         | Sub(e0, e1) => map stripCoreExpr [e0, e1]
         | Member(e, _) => [stripCoreExpr e]
         | Arrow(e, _) => [stripCoreExpr e]
         | Deref e => [stripCoreExpr e]
         | AddrOf e => [stripCoreExpr e]
         | Binop(_, e0, e1) => map stripCoreExpr [e0, e1]
         | Unop(_, e) => [stripCoreExpr e]
         | Cast(_, e) => [stripCoreExpr e]
         | EnumId _ => nil
         | SizeOf _ => nil
         | ExprExt _ => nil
         | ErrorExpr => nil
      and getImmediateNestableSubExprs e =
        let fun f nil = nil
              | f (x::xs) = if isNestable (e,x)
                            then (getImmediateNestableSubExprs x)@(f xs)
                            else x::(f xs)
        in f (getImmediateSubExprs e) end

      (* Returns list of "tiles" -- first tile is the original
         expression, subsequent tiles are expressions that could
         not be nested in the original expression, and must be
         listed later (i.e., [expr0], where e1 is [expr1], ...).
         Also included in each tile is the number of its first
         temporary.  For instance, temporary number 5 we call e5. *)
      and tileCoreExpr (e, i) =
        let fun g (nil, i) = (nil, nil, i)
              | g (x::xs, i) = 
                  let val (ei::L1, n1) = tileCoreExpr (x, i)
                      val (L2, L3, n2) = g (xs, n1)
                  in (ei::L2, L1 @ L3, n2) end
            fun f (L, i) =
              let val (L1, L2, n) = g (L, i+List.length L)
              in (L1 @ L2, n) end
            val (L, n) = f (getImmediateNestableSubExprs e, i)
        in ((e, i)::L, n) end

      and peCoreExpr coreExpr =
        let val (L, n) = tileCoreExpr (coreExpr, 1)
        in case L
           of ((e, i)::nil) => peSimpleExpr (e, i)
            | ((e, i)::es) =>
               (peSimpleExpr (e, i);
                case es of nil => ()
                         | _ => (print ", where "; peSimpleExprList es)) end

      and peExpr (EXPR (coreExpr, _, _)) = peCoreExpr coreExpr

      and peTaggedElement (e, s, i) =
            (print ("a" ^ s); peInt i;
             print " is "; peInitExpression (s ^ (myIntToS i) ^ "_", e))

      and peElementList x = peList peTaggedElement x

      and isSimpleInitExpr (Simple e) =
            let val (L, n) = tileCoreExpr (stripCoreExpr e, 1)
            in case L of (_::nil) => true
                       |        _ => false end
        | isSimpleInitExpr _ = false

      and stripSimpleInitExpr (Simple e) =
            let val (x::nil, _) = tileCoreExpr (stripCoreExpr e, 1)
            in x end

      and areSimpleInitExprs nil = true
        | areSimpleInitExprs (x::xs) =
            isSimpleInitExpr x andalso areSimpleInitExprs xs

      and peInitExpression (sPrefix, Simple expr) = peExpr expr
        | peInitExpression (sPrefix, Aggregate nil) =
            print "the array with no elements"
        | peInitExpression (sPrefix, Aggregate (e::nil)) =
            (print "the array whose only element is ";
             peInitExpression (sPrefix, e))
        | peInitExpression (sPrefix, Aggregate initExprs) =
           let
            fun numberElements (nil, n) = nil
              | numberElements (x::xs, n) =
                  ((x, sPrefix, n)::numberElements(xs, n+1))
           in (print "the array with elements ";
               if areSimpleInitExprs initExprs
               then peList peSimpleExpr (map stripSimpleInitExpr initExprs)
               else
                (print ("a" ^ sPrefix);
                 print ("0 to a" ^ sPrefix);
                 print (Int.toString ((List.length initExprs) - 1));
                 print ", where ";
                 peElementList (numberElements (initExprs, 0)))) end

      and peType (article, stClass, cType) =
          let
            fun looplist' (a, nil) = ()
              | looplist' (a, ct::nil) = (print "and "; loop (a, ct))
              | looplist' (a, ct::cts) = (loop (a, ct); print ", ";
                                          looplist' (a, cts))
            and looplist (a, nil) = ()
              | looplist (a, ct::nil) = loop (a, ct)
              | looplist (a, ct1::ct2::nil) = (loop (a, ct1);
                                               print " and ";
                                               loop (a, ct2))
              | looplist (a, L) = looplist' (a, L)
            and args nil = print "no arguments"
              | args (ct::nil) = (print "argument ";
                                  loop (SOME AN, ct))
              | args L = (print "arguments ";
                          looplist (SOME AN, L))

            and loop (a, Void) = (peArt(a, CONS); print "void")
              | loop (a, Ellipses) = (peArt (a, VOWEL); print "argument list")
              | loop (a, Qual (qf, ct)) = (peQualifier (a, qf);
                                           loop (NONE, ct))
              | loop (a, Numeric (NONSATURATE, WHOLENUM, _, CHAR,
                                                            SIGNASSUMED)) =
                  (peArt (a, CONS); print "character")
              | loop (a, Numeric (NONSATURATE, WHOLENUM, SIGNED, CHAR,
                               SIGNDECLARED)) =
                  (peArt (a, CONS); print "signed character")
              | loop (a, Numeric (NONSATURATE, WHOLENUM, UNSIGNED, ik, _)) =
                  (peArt (a, VOWEL); print "unsigned "; peIntKind (NONE, ik))
              | loop (a, Numeric (NONSATURATE, FRACTIONAL, sign, ik, _)) =
                  (peArt (a, CONS); print "fractional ";
                   peSignedness sign; peIntKind (NONE, ik))
              | loop (a, Numeric (SATURATE, frac, sign, ik, _)) =
                  (peArt (a, CONS); print "saturate ";
                   peSignedness sign; peIntKind (NONE, ik))
              | loop (a, Numeric (NONSATURATE, WHOLENUM, SIGNED, ik, _)) =
                  peIntKind (a, ik)
              | loop (a, Pointer ct) = (peArt (a, CONS); print "pointer to ";
                                        loop (SOME AN, ct))
              | loop (a, EnumRef tid) = (peTid (a, tid); print " enumeration")
              | loop (a, StructRef tid) = (peTid (a, tid); print " structure")
              | loop (a, UnionRef tid) = (peTid (a, tid); print " union")
              | loop (a, TypeRef tid) = (peTid (a, tid))
              | loop (a, Array (NONE, ct)) = (peArt(a, VOWEL);
                        print "array each element of which is ";
                        loop (SOME AN, ct))
              | loop (a, Array (SOME(i, expr), ct)) = (peArt(a, VOWEL);
                        print "array of length ";
                        peExpr expr;
                        print " each element of which is ";
                        loop (SOME AN, ct))
              | loop (a, Function (Void, nil)) = (peArt(a, CONS);
                        print "function with no arguments and no result")
              | loop (a, Function (ct, nil)) = (peArt(a, CONS);
                        print "function with no arguments returning ";
                        loop (SOME AN, ct))
              | loop (a, Function (ct, cts)) = (peArt(a, CONS);
                        print "function with "; args cts;
                        case ct of Void => print ", and with no result"
                        | _ => (print ", and returning "; loop (SOME AN, ct)))
              | loop (a, Error) = (peArt(a, VOWEL); print "unknown type")
          in
            case stClass of DEFAULT => loop(article, cType)
                          | _ => (peStorageClass(article, stClass);
                                  loop (NONE, cType))
          end

      fun peIdType (article, id as {stClass, ctype, ...} : Ast.id) =
            peType (article, stClass, ctype)

      (* print names of arguments, or argument list for ellipses *)
      fun peArgName (x as {stClass, ctype, ...} : Ast.id) =
        (case ctype
         of Ellipses => (print "an unnamed "; peType (NONE, stClass, ctype))
          | _ => peIdName x)

      (* print name and type of a ctype, not including ellipses *)
      fun peArgNameAndType (x as {stClass, ctype, ...} : Ast.id) =
        (case ctype
         of Ellipses => ()
          | _ => (peIdName x; print " is "; peType (SOME AN, stClass, ctype)))

      (* returns only the non-ellipses arguments to a function *)
      fun namedArgs nil = nil
        | namedArgs ((x as {ctype, ...}: Ast.id)::xs) =
            case ctype of Ellipses => namedArgs xs | _ => (x::(namedArgs xs))

      fun peArgList nil = print "no arguments"
        | peArgList L =
            let val L' = namedArgs L
            in (print "argument";
                case L of (x::nil) => ()
                        | _ => print "s";
                print " "; peList peArgName L;
                case L' of nil => ()
                         | _ => (print ", where "; peList peArgNameAndType L'))
            end

      fun nameOfMember (ct:ctype, {name, ...}:member) = name

      fun peMemberNameAndType (ct,m) = (peSymbol (nameOfMember (ct, m));
                                        print " is ";
                                        peType (SOME AN, DEFAULT, ct))

      (* used for structure bit fields *)
      fun peSizeOpt NONE = ()
        | peSizeOpt (SOME li) = (peLargeInt li; print "-bit ")

      fun peStructMemberName (ct, NONE, liOpt) =
           (print "an unnamed "; peSizeOpt liOpt; peType (NONE, DEFAULT, ct))
        | peStructMemberName (ct, SOME mem, liOpt) =
           peSymbol (nameOfMember (ct, mem))

      fun peStructMember (ct, SOME mem, liOpt) =
           (peSymbol (nameOfMember (ct, mem)); print " is ";
            case liOpt of NONE => peType (SOME AN, DEFAULT, ct)
                        | SOME li => (peArt (SOME AN,
                                       letterClass (hd
                                           (explode (IntInf.toString li))));
                                peSizeOpt liOpt; peType (NONE, DEFAULT, ct)))

      fun peUnionMemberList nil = print "no members"
        | peUnionMemberList L =
            (print "member";
             case L of (x::nil) => ()
                     | _ => print "s";
             print " ";
             peList peStructMemberName (map (fn (x,y) => (x,SOME y,NONE)) L);
             print ", where ";
             peList peStructMember (map (fn (x,y) => (x,SOME y,NONE)) L))

      fun namedStructMembers nil = nil
        | namedStructMembers ((_, NONE, _)::xs) = namedStructMembers xs
        | namedStructMembers (x::xs) = x::(namedStructMembers xs)

      fun peStructureMemberList nil = print "no members"
        | peStructureMemberList L =
            let val L' = namedStructMembers L
            in (print "member";
                case L of (x::nil) => ()
                        | _ => print "s";
                print " "; peList peStructMemberName L;
                case L' of nil => ()
                         | _ => (print ", where "; peList peStructMember L'))
            end

      fun peMemberInt ({name, ...}:member, li) =
        (peSymbol name; print " equal to "; peLargeInt li)

      (* print definition of a typedef, enum, union, or struct *)
      fun peNamedCType (nct, b) =
        case nct
        of Bindings.Struct (tid, members) =>
            (lPrint ("let ", b);
             peTid (NONE, tid); print " be a structure with ";
             peStructureMemberList members)
         | Bindings.Union (tid, members) =>
            (lPrint ("let ", b);
             peTid (NONE, tid); print " be a union with ";
             peUnionMemberList members)
         | Bindings.Enum (tid, members) =>
            (lPrint ("let ", b);
             peTid (NONE, tid); print " be an enumeration with ";
             case members of nil => print "no values"
                           | (x::nil) => print "value "
                           | (x::xs) => print "values ";
             peList peMemberInt members)
         | Bindings.Typedef (tid, ctype) =>
            (lPrint ("let ", b);
             peTid (NONE, tid); print " be another name for ";
             peType (SOME AN, DEFAULT, ctype))

      (* declarations and definitions of vars, structs, etc. *)
      fun peDeclaration (TypeDecl {shadow=NONE, tid}, b) =
           (case Tidtab.find (tidtab, tid)
            of SOME {ntype=SOME nct, ...} => peNamedCType (nct, b)
             | _ => ())
        | peDeclaration (TypeDecl {shadow=SOME{strct=true}, tid}, b) =
           (lPrint ("assume ", b);
            peTid (NONE, tid);
            print " is a structure")
        | peDeclaration (TypeDecl {shadow=SOME{strct=false}, tid}, b) =
           (lPrint ("assume ", b);
            peTid (NONE, tid);
            print " is a union")
        | peDeclaration (VarDecl (id, initOpt), b) =
           (lPrint ("let ", b);
            peIdName id;
            print " be ";
            peIdType (SOME AN, id);
            case initOpt of
             SOME initExpr => (print ", whose initial value is ";
                               peInitExpression ("", initExpr))
           | NONE => ())

      fun peLabel ({name, ...}:label) = peSymbol name

      val nBlocks = ref 0
      val numberedBlocks : (coreStatement ref * int) list ref = ref nil

      fun peekNextNumber () = !nBlocks + 1
      fun newNumber () = (nBlocks := peekNextNumber(); !nBlocks)
      fun isNumberedStmt stm =
        let fun f (s, nil) = false
              | f (s1, (s2,i)::xs) = if s1=s2 then true else f (s1, xs)
        in f(stm, !numberedBlocks) end
      fun getNumberOfStmt stm =
        let fun f (s, nil) = ~1
              | f (s1, (s2,i)::xs) = if s1=s2 then i else f (s1, xs)
            val n = f(stm, !numberedBlocks)
        in if n=(~1) then peekNextNumber() else n end
      fun newNumberedStmt stm =
        let val n = newNumber()
        in (numberedBlocks := (stm, n)::(!numberedBlocks); n) end

      (* print an expression in a context requiring no result *)
      fun peSimpleExprStmt (e, i, b) =
       let val nNestedListExprTemps = nNestedListExprTemps e
           val peNestedExprList = peNestedExprList e
           val peNestedExpr = peNestedExpr e
           val nNestedExprTemps = nNestedExprTemps e
       in
        case e
        of Call(ex, exs) =>
            let val exs = map stripCoreExpr exs
                val ex = stripCoreExpr ex
            in (case exs of nil => lPrint ("call ", b)
                | _ => (lPrint ("pass ", b);
                        peNestedExprList (mapNestedExprTemps (e, exs,
                                                i+nNestedExprTemps ex));
                        print " to ");
                peNestedExpr (ex, i))
            end
         | Assign(e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (lPrint ("set ", b);
                peNestedExpr (e0, i); print " equal to ";
                peNestedExpr (e1, i1)) end
         | Binop(binop, e0, e1) =>
            let val [e0, e1] = map stripCoreExpr [e0, e1]
                val i1 = i+nNestedExprTemps e0
            in (case binop
              of PlusAssign => (lPrint ("increment ", b);
                                peNestedExpr (e0, i);
                                print " by ";
                                peNestedExpr (e1, i1))
               | MinusAssign => (lPrint ("decrement ", b);
                                 peNestedExpr (e0, i);
                                 print " by ";
                                 peNestedExpr (e1, i1))
               | TimesAssign => (lPrint ("assign ", b);
                                 peNestedExpr (e0, i);
                                 print " times ";
                                 peNestedExpr (e1, i1);
                                 print " to ";
                                 peNestedExpr (e0, i))
               | DivAssign => (lPrint ("assign ", b);
                               peNestedExpr (e0, i);
                               print " divided by ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | ModAssign => (lPrint ("assign ", b);
                               peNestedExpr (e0, i);
                               print " modulus ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | XorAssign => (lPrint ("assign the exclusive or of ", b);
                               peNestedExpr (e0, i);
                               print " and ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | OrAssign => (lPrint ("assign the bitwise or of ", b);
                              peNestedExpr (e0, i);
                              print " and ";
                              peNestedExpr (e1, i1);
                              print " to ";
                              peNestedExpr (e0, i))
               | AndAssign => (lPrint ("assign the bitwise and of ", b);
                               peNestedExpr (e0, i);
                               print " and ";
                               peNestedExpr (e1, i1);
                               print " to ";
                               peNestedExpr (e0, i))
               | LshiftAssign => (lPrint ("assign ", b);
                                  peNestedExpr (e0, i);
                                  print " shifted left ";
                                  peNestedExpr (e1, i1);
                                  print " bits to ";
                                  peNestedExpr (e0, i))
               | RshiftAssign => (lPrint ("assign ", b);
                                  peNestedExpr (e0, i);
                                  print " shifted right ";
                                  peNestedExpr (e1, i1);
                                  print " bits to ";
                                  peNestedExpr (e0, i))
               | _ => (lPrint ("evaluate ", b); peSimpleExpr (e, i)))
            end
         | Unop(unop, exp) => let val exp = stripCoreExpr exp in
            (case unop
              of PreInc => (lPrint ("increment ", b);
                            peNestedExpr (exp, i);
                            print " by one")
               | PostInc => (lPrint ("increment ", b);
                             peNestedExpr (exp, i);
                             print " by one")
               | PreDec => (lPrint ("decrement ", b);
                            peNestedExpr (exp, i);
                            print " by one")
               | PostDec => (lPrint ("decrement ", b);
                             peNestedExpr (exp, i);
                             print " by one")
               | _ => (lPrint ("evaluate ", b); peSimpleExpr (e, i))) end
         | _ => (lPrint ("evaluate ", b); peSimpleExpr (e, i)) end

      fun peExprStmt (e, inSentence) =
        let val e = stripCoreExpr e
            val (L, n) = tileCoreExpr (e, 1)
        in case L
           of ((e, i)::nil) => peSimpleExprStmt (e, i, inSentence)
            | ((e, i)::es) =>
               (peSimpleExprStmt (e, i, inSentence);
                case es of nil => ()
                         | _ => (print ", where "; peSimpleExprList es)) end

      fun peBoolExpr e = (* must be lower case *)
        let val e = stripCoreExpr e
            val (L, n) = tileCoreExpr (e, 1)
        in case L
           of ((e, i)::nil) => peSimpleBoolExpr (e, i)
            | ((e, i)::es) =>
               (peSimpleBoolExpr (e, i);
                case es of nil => ()
                         | _ => (print ", where "; peSimpleExprList es)) end

      fun pePerformOp (rCoreStmt, b) =
        case (!rCoreStmt)
        of Compound _ => (lPrint ("perform block ", b);
                          peInt (newNumberedStmt rCoreStmt))
         | _ => (lPrint ("perform operation ", b);
                 peInt (newNumberedStmt rCoreStmt))

      (* can a statement be a subclause of a sentence *)
      fun isStmtNestable (Expr _)   = true
        | isStmtNestable (Goto _)   = true
        | isStmtNestable (Return _) = true
        | isStmtNestable Break      = true
        | isStmtNestable Continue   = true
        | isStmtNestable _          = false

      fun isSingleSentenceStmt coreStmt =
        isStmtNestable coreStmt orelse case coreStmt
        of Compound _ => false
         | While (e,s) => isStmtNestable (stripCoreStmt s)
         | Do (e,s) => isStmtNestable (stripCoreStmt s)
         | For (_,_,_,s) => isStmtNestable (stripCoreStmt s)
         | Labeled (L,s) => isSingleSentenceStmt (stripCoreStmt s)
                            (* deal w transitions as if single *)
         | CaseLabel (li,s) => isSingleSentenceStmt (stripCoreStmt s)
         | DefaultLabel s => isSingleSentenceStmt (stripCoreStmt s)
         | IfThen (e,s) => isStmtNestable (stripCoreStmt s)
         | IfThenElse (e,s1,s2) => isStmtNestable (stripCoreStmt s1) andalso
                                   isStmtNestable (stripCoreStmt s2)
         | Switch (e,s) => false
         | _ => true

      fun isMultipleSentenceStmt coreStmt = not (isSingleSentenceStmt coreStmt)

      fun peOpBlock (Compound (dL,sL), n) = (print "block ";peInt n;print ", ";
                                             if length dL + length sL > 1
                                             then print "first " else ())
        | peOpBlock (_, n)          = (print "operation ";peInt n;print ", ")

      fun sameStorage (st1,st2:Ast.storageClass) = st1=st2
      fun sameQual (q1, q2:Ast.qualifier) = q1=q2
      fun sameNumeric (n1, n2:(saturatedness * fractionality * signedness
                               * intKind * signednessTag)) = n1=n2
      fun sameCTList (nil,nil) = true
        | sameCTList (ct1::ct1L, ct2::ct2L) =
            sameCT(ct1, ct2) andalso sameCTList (ct1L, ct2L)
        | sameCTList _ = false
      and sameTid (t1,t2) =
        let val s1 = Tidtab.find (tidtab, t1)
            val s2 = Tidtab.find (tidtab, t2)
        in case (s1, s2)
           of (SOME{name=SOME id1, ...},SOME{name=SOME id2, ...}) => id1=id2
            | _ => false end
      and sameExpr (e1, e2) =
           case (stripCoreExpr e1, stripCoreExpr e2)
           of (IntConst li1, IntConst li2) => li1=li2
            | _ => false

      and sameCT (ct1,ct2) =
       (case (ct1,ct2)
        of (Void,Void) => true
         | (Ellipses, Ellipses) => true
         | (Qual (q1,ct1), Qual(q2,ct2)) => sameQual (q1,q2) andalso
                                            sameCT (ct1,ct2)
         | (Numeric n1, Numeric n2) => sameNumeric (n1, n2)
         | (Array(NONE,ct1), Array(NONE,ct2)) => sameCT(ct1,ct2)
         | (Array(SOME(_,e1),ct1), Array(SOME(_,e2),ct2)) =>
             sameExpr (e1, e2) andalso sameCT (ct1, ct2)
         | (Pointer ct1, Pointer ct2) => sameCT(ct1, ct2)
         | (Function(ct1,ct1L),Function(ct2,ct2L)) =>
             sameCT (ct1, ct2) andalso sameCTList (ct1L, ct2L)
         | (StructRef t1, StructRef t2) => sameTid (t1, t2)
         | (UnionRef t1, UnionRef t2) => sameTid (t1, t2)
         | (EnumRef t1, EnumRef t2) => sameTid (t1, t2)
         | (TypeRef t1, TypeRef t2) => sameTid (t1, t2)
         | _ => false)

      fun sameVarDec (VarDecl ({stClass=st1,ctype=ct1,...}:Ast.id, NONE),
                      VarDecl ({stClass=st2,ctype=ct2,...}:Ast.id, NONE)) =
            sameStorage (st1,st2) andalso sameCT (ct1, ct2)
        | sameVarDec (_, _) = false

      (* Returns (L1, L2), where L1 is list of successive variable
         declarations with no initializer and same type. L2 is
         what's left over. *)
      fun packDeclarations nil = (nil,nil)
        | packDeclarations (x::nil) = ([x], nil)
        | packDeclarations (x1::x2::xs) =
           if (sameVarDec (x1, x2))
           then let val (L1,L2) = packDeclarations(x2::xs)
                in (x1::L1, L2) end
           else ([x1], x2::xs)

      fun decl2Id (VarDecl (id,_)) = id

      (* declare a list of variables all with the same type *)
      fun peSameDeclList (nil, b) = ()
        | peSameDeclList (L as (x::xs), b) =
            (lPrint ("let ", b); peList peIdName (map decl2Id L);
             print " each be "; peIdType (SOME AN, decl2Id x))

      fun peDeclarationList (nil, b) = ()
        | peDeclarationList (x::nil, b) = peDeclaration (x, b)
        | peDeclarationList (L, b) =
            (case packDeclarations L
             of (x::nil, nil) => peDeclaration (x,b)
              | (x::xs,  nil) => peSameDeclList (x::xs, b)
              | (x::nil, xL) => (peDeclaration (x, b);
                                 print ".  ";
                                 peDeclarationList (xL, false))
              | (x::xs,  xL) => (peSameDeclList (x::xs, b); print ".  ";
                                 peDeclarationList (xL, false)))

      fun peStmtInSentence (rC, b) =
        if isStmtNestable (!rC) then peCoreStmt (rC, b)
        else pePerformOp (rC, b)

      and peFollowUpStmt rC = if isStmtNestable (!rC) then ()
                             else (print ".  "; peCoreStmt (rC, false))

      and peDeclStmtList (nil, nil, b) = lPrint ("do nothing", b)
        | peDeclStmtList (nil, sL, b) = peFunStatementList (sL, b)
        | peDeclStmtList (dL, nil, b) = peDeclarationList (dL, b)
        | peDeclStmtList (dL, sL, b) = (peDeclarationList (dL, b); print ".  ";
                                        peStatementList (sL, false))

      and peTransition b =
        (nNext := (!nNext + 1) mod 3;
         if (!nNext) = 1 then lPrint ("then ", b)
         else if (!nNext) = 2 then lPrint ("next, ", b)
         else lPrint ("after that, ", b))

      and peFunStatementList (nil, b) = ()
        | peFunStatementList (x::xs, b) =
           (peCoreStmt (ref (stripCoreStmt x), b);
            case xs of nil => ()
                     | _ => (print ".  ";
                             if isMultipleSentenceStmt (stripCoreStmt x)
                             then print "Continuing on, we next "
                             else peTransition false;
                             peFunStatementList (xs, true)))

      and peStatementList' (nil, b, _) = ()
        | peStatementList' (x::xs, b, SOME prev) =
           let val rC = ref (stripCoreStmt x)
           in (if isMultipleSentenceStmt (!prev)
               then lPrint ("continuing on, we next ", b)
               else peTransition b;
               peCoreStmt (rC, true);
               case xs of nil => () | _ => print ".  ";
               peStatementList' (xs, false, SOME rC)) end
        | peStatementList' (x::xs, b, NONE) =
           let val rC = ref (stripCoreStmt x)
           in (if b then (peStmtInSentence (rC, b); peFollowUpStmt rC)
               else peCoreStmt (rC, b);
               case xs of nil => () | _ => print ".  ";
               peStatementList' (xs, false, SOME rC)) end

      and peStatementList (L,b) = peStatementList' (L,b,NONE)

      and peCoreStmt (rCoreStmt, inSentence) =
        let val coreStmt = !rCoreStmt
            val bN = isNumberedStmt rCoreStmt
            val nN = getNumberOfStmt rCoreStmt
            val lower = bN orelse inSentence
        in (if bN then (lPrint ("to perform ", inSentence);
                        peOpBlock (coreStmt, nN))
            else ();
        case coreStmt
          of Expr expOpt =>
              (case expOpt of NONE => (lPrint ("compute nothing", lower))
                            | SOME exp => peExprStmt (exp, lower))
           | Compound (dL, sL) =>
              (if bN then peDeclStmtList (dL, sL, lower)
               else (lPrint ("perform block ", lower);
                     peInt (newNumberedStmt rCoreStmt);
                     print ", which is defined as follows.  ";
                     if (length dL + length sL) > 1
                     then (print "First "; peDeclStmtList (dL, sL, true))
                     else peDeclStmtList (dL, sL, false));
               if (length dL + length sL) > 1
               then (print ".  This ends block "; peInt nN)
               else ())
           | While (e, s) =>
              let val rC = ref (stripCoreStmt s)
              in (lPrint ("continue to ", lower); peStmtInSentence (rC, true);
                  print " as long as "; peBoolExpr e;
                  print " at the beginning of the iteration";
                  peFollowUpStmt rC) end
           | Do (e, s) =>
              let val rC = ref (stripCoreStmt s)
              in (lPrint ("continue to ", lower); peStmtInSentence (rC, true);
                  print " as long as "; peBoolExpr e;
                  print " at the end of the iteration";
                  peFollowUpStmt rC) end
           | For (eo1, eo2, eo3, s) =>
              let val rC = ref (stripCoreStmt s)
              in (case eo2 of NONE => (lPrint ("repeatedly ", lower);
                                        peStmtInSentence (rC, true))
                            | SOME e => (lPrint ("continue to ", lower);
                                         peStmtInSentence(rC, true);
                                         print " as long as ";
                                         peBoolExpr e;
                                         print " prior to the iteration");
                  case eo1 of NONE => ()
                            | SOME e =>
                   (print "; before starting this loop, ";
                    peExprStmt (e, true));
                  case eo3 of NONE => ()
                            | SOME e =>
                   (print "; upon completion of each iteration of the loop, ";
                      peExprStmt (e, true));
                  peFollowUpStmt rC
                 ) end
           | Labeled (L, s) =>
              (lPrint ("label this point in the code ", lower);
               peLabel L; print ".  "; peStatement s)
           | CaseLabel (li, s) =>
              (lPrint ("let this be the start of case ", lower);
               peLargeInt li; print ".  "; peStatement s)
           | DefaultLabel s =>
              (lPrint ("let this be the start of the default case.  ",
                        lower);
               peStatement s)
           | Goto L => (lPrint ("go to label ", lower); peLabel L)
           | Break =>
              lPrint ("break from the nearest enclosing loop or case-block",
                      lower)
           | Continue =>
               lPrint ("continue execution at the start of the loop",
                        lower)
           | Return expOpt => (lPrint ("return ", lower);
              case expOpt
              of NONE => print "from the function"
               | SOME exp => peExpr exp)
           | IfThen (e, s) =>
              let val rC = ref (stripCoreStmt s)
              in (peStmtInSentence (rC, lower); print " provided ";
                  peBoolExpr e; peFollowUpStmt rC) end
           | IfThenElse (e, s1, s2) =>
              let val rC1 = ref (stripCoreStmt s1)
                  val rC2 = ref (stripCoreStmt s2)
              in (peStmtInSentence (rC1, lower); print " provided ";
                  peBoolExpr e; print "; otherwise, ";
                  peStmtInSentence (rC2, true);
                  peFollowUpStmt rC1; peFollowUpStmt rC2) end
           | Switch (e,s) =>
              let val rC = ref (stripCoreStmt s)
              in (peStmtInSentence (rC, lower);
                  print ", starting at case ";
                  peExpr e;
                  print ", or the default case if no such case exists";
                  peFollowUpStmt rC) end
           | StatExt () => ()
           | ErrorStmt => ()) end

      and peStatement stmt = peCoreStmt (ref (stripCoreStmt stmt), false)

      fun peFunctionDef (id, idL, stmt) =
            (print "Let "; peIdName id; print " be ";
             case id of {stClass, ...} =>
                (case stClass of DEFAULT => print "a"
                               | _ => peStorageClass(SOME AN, stClass));
             print " function ";
             case id of {ctype=Function(Void, _), ...} =>
                             print "with no result"
                      | {ctype=Function(t,_), ...} =>
                             (print "returning "; peType (SOME AN, DEFAULT, t))
                      | _ => peIdType (SOME AN, id); (* error case *)
             print ".  It is called with ";
             peArgList idL; print ".  To perform the function, ";
             case (stripCoreStmt stmt)
                of Compound (dL, sL) => peDeclStmtList (dL, sL, true)
                 | _ => (print "do the following.  "; peStatement stmt))

      fun peExternalDeclExt () = ()

      fun peCoreExternalDecl (ExternalDecl decl) =
            (peDeclaration (decl, false); print ".  ")
        | peCoreExternalDecl (ExternalDeclExt externalDeclExt) =
            peExternalDeclExt externalDeclExt
        | peCoreExternalDecl (FunctionDef (id, idL, stm)) =
            peFunctionDef (id, idL, stm)

      fun peExternalDecl (DECL (coreExternalDecl, aid, SMLoc)) =
            peCoreExternalDecl coreExternalDecl

      fun stripExternalDecl (DECL (c, _, _)) = c

      fun splitDefs nil = (nil, nil)
        | splitDefs (x::xs) =
            (case stripExternalDecl x
             of ExternalDecl d => let val (dL, yL) = splitDefs xs
                                  in (d::dL, yL) end
              | _ => (nil, x::xs))

      fun peExternalDeclList nil = nil
        | peExternalDeclList L =
           let val (decls, fundefs) = splitDefs L
           in (peDeclarationList (decls, false); fundefs) end

      fun peAst nil = ()
        | peAst (L as ast::asts) =
           let val asts =
            (case stripExternalDecl ast
             of ExternalDecl _ => peExternalDeclList L
              | x => (peCoreExternalDecl x; asts))
           in (print ".\n\n"; peAst asts) end
    in
      peAst
    end

  fun astBundleToEnglish (strm,{ast=a,tidtab=t, ...} : ParseToAst.astBundle) =
    astToEnglish strm t a
end
