;;;  H E A P . S C H
;;;  Brian Beckman, JPL, 11 Nov 1987
;;;  A little laboratory for fiddling around with heap-form trees.
;;;  (see Bentley, Programming Pearls, Addison Wesley, 1986, Ch 12)

; A heap is a list: (current-size  max-size  vector-containing-tree)
;
; rules are: heap index varies from 1 to current-size, inclusive
;   (vector indices vary from 0 to n-1) -- 
;   check current-size when heap is accessed.
;
; current-size must be less than max size
;   check this when heap is updated.


(define (make-heap max-size)
  (list  0  max-size  (make-vector max-size 0)))


; Here are some essential operators on heaps
; --------------------------------------------------------------------


(define root 1)


(define (index-too-far? heap i)
  (let
    ((current-size (car heap)))
    (or
     (< i root)
     (> i current-size))))


(define (value heap i)
  (let
    ((vec (caddr heap)))
    (if (index-too-far? heap i)
        ()
        (vector-ref vec (1- i)))))


;; index thingies

(define (left-child i)
  (* 2 i))

(define (right-child i)
  (1+ (left-child i)))

(define (parent i)
  (quotient i 2))


;; ordering stuff

(define (order-routine a b)  ;; change this to use a different ordering
  (<= a b))

(define (in-order-node? heap node)  ;; is a give node in order ?
  (order-routine
   (value heap (parent node))
   (value heap node)))


;; heap constructors and modifiers

(define (set-node! heap node val)  ;; set heap's node to val
  (let
    ((vec (caddr heap)))
    (if index-too-far? heap node)
      (begin (display "Heap Error (set-node!): invalid index!") 
             (newline) ())  ;; return nil
      (begin (vector-set! vec (1- node) val) val))))  ;; return val


; Here is the meat of the algorithm -- sift-up and insert-in-heap
; --------------------------------------------------------------------

(define sift-up!
  
  (lambda (heap i) 
    
    (let
      ((mom (parent i))
       (temp 0))  ;; local variable for later use
      
      (if (=? i root)
          #!true
          
          (if (in-order-node? heap i)
              #!true
              
              (begin  ;; swap i with parent, then sift parent
               (set! temp (value heap mom))
               (set-node! heap mom (value heap i))
               (set-node! heap i temp)
               (sift-up! heap mom)))))))


(define insert-in-heap!
  (lambda (heap number)
    
    (define (cant-expand? heap)  ;; local routine
      (let
        ((max-size (cadr heap))
         (current-size (car heap)))
         (> (1+ current-size) max-size)))
    
    (if (cant-expand? heap)
        (begin
         (display "Heap Error (insert-in-heap): no room!")
         (newline)
         ())
        
        (begin
         (set-car! heap (1+ (car heap)))      ;; bump current-size
         (set-node! heap (car heap) number)   ;; set last element
         (sift-up! heap (car heap))))))       ;; sift last elt up



(define insert-random!
  (lambda (heap limit)
    (insert-in-heap! heap (random limit))
    heap))  ;; return heap
